/*
 * ===========================
 * VDK Visual Develeopment Kit
 * Version 2.0.0
 * november 2000
 * ===========================
 *
 * Copyright (C) 1998, Mario Motta
 * Developed by Mario Motta <mmotta@guest.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#ifndef _canvascompo_h
#define _canvascompo_h
#include <vdk/vdk.h>
#define MAX_SHAPES 10
typedef VDKValueList<VDKPoint> PointList;
typedef VDKValueListIterator<VDKPoint> PointListIterator;
/*
 */
class CanvasShape;
class CanvasComponent : public VDKBox
{
  VDKCanvas *canvas;
  VDKLabel  *label;
  VDKCustomButton *mlButton;
  VDKCustomButton *clearButton;
  CanvasShape* shape_array[MAX_SHAPES];
  PointList plist;
 public:
  CanvasComponent(VDKForm* owner): VDKBox(owner){}
  ~CanvasComponent();
  void Setup();
  bool OnMotion(VDKObject* sender, GdkEvent* event);
  bool OnExpose(VDKObject*, GdkEvent* event);
  bool Clear(VDKObject*);
  bool OnButtonPress(VDKObject*, GdkEvent* ev);
  bool OnButtonRelease(VDKObject*, GdkEvent* ev);
  DECLARE_SIGNAL_MAP(CanvasComponent);
  DECLARE_EVENT_LIST(CanvasComponent);
};

/*
some shapes
*/
class CanvasShape
{
 protected:
  VDKCanvas* canvas;
  VDKRgb color;
  bool filled;
 public:
  VDKRect bound;
  CanvasShape(VDKCanvas* owner,
	      int x, int y, int w, int h, 
	      VDKRgb color, bool filled = false): 
    canvas(owner),color(color),filled(filled),bound(x,y,w,h) {}
  virtual ~CanvasShape() {}
  virtual void Draw() = 0;
};

class 
CanvasRectangle: public CanvasShape
{

 public:
  CanvasRectangle(VDKCanvas* owner,
		  int x, int y, int w, int h,
		  VDKRgb color,
		  bool filled = false): 
    CanvasShape(owner,x,y,w,h,color,filled) {}
  virtual ~CanvasRectangle() {}
  void Draw() 
  {
    canvas->Foreground = color;
    canvas->DrawRect(filled,bound.Origin().x,
		     bound.Origin().y,
		     bound.w,bound.h);
  }
};

/*
  Arc constructor is probably one that needs more
  explanation becaues it has quite a few parameters:
  x,y origin of arc bounding rect
  w,h width and height of arc bounding rect
  start_angle,end_angle are a bit more complicated to explain
  since unlike most arc function aren't expressed in radians
  or degrees. The angles should be converted into degrees and
  multiplied by 64 (which is done by the Draw() method itself).
  So pass start_angle and end_angle in degrees with angle 0 at
  12 o'clock and counting clockwise.
 */
class
CanvasArc: public CanvasShape
{
 private:
  int start_angle, end_angle;
 public:
    CanvasArc(VDKCanvas* owner,
	      int x, int y, int w, int h,
	      int start_angle, int end_angle,
	      VDKRgb color,
	      bool filled = false): 
    CanvasShape(owner,x,y,w,h,color,filled),
      start_angle(start_angle),end_angle(end_angle) {}
    virtual ~CanvasArc() {}
    virtual void Draw()
      {
	canvas->Foreground = color;
	canvas->DrawArc(filled,
			   bound.Origin().x,
			   bound.Origin().y,
			   bound.w,bound.h,
			   start_angle*64,end_angle*64);
      }
};

class 
CanvasCircle: public CanvasArc
{
 public:
  CanvasCircle(VDKCanvas* owner, 
	       int x,int y, 
	       int radius, 
	       VDKRgb color,
	       bool filled = false):
    CanvasArc(owner,x-radius,y-radius,radius*2,radius*2,
		0,360*64,color,filled) {}
  ~CanvasCircle() {}
    
};

class 
CanvasPixmap: public CanvasShape
{
 private:
  char pixfile[256];
  VDKRawPixmap* pixmap;
 public:
  CanvasPixmap(VDKCanvas* owner, int x, int y, char* pixfile):
    CanvasShape(owner,x,y,0,0,VDKRgb(-1,-1,-1))
    { 
      pixmap = new VDKRawPixmap(owner,pixfile); 
      bound = VDKRect(bound.left,bound.top, pixmap->Width(), pixmap->Height());
    }
  virtual ~CanvasPixmap(){ pixmap->Destroy(); }
  void Draw() 
    { 
      canvas->DrawPixmap(bound.Origin().x,bound.Origin().y,pixmap);
    }
};

typedef VDKArray<VDKPoint> PointArray;
class 
CanvasPolygon: public CanvasShape
{
 private:
  GdkPoint *points;
  int npoints;
 public:
  CanvasPolygon(VDKCanvas *owner,
		PointArray& array, 
		VDKRgb color,
		bool filled = false);
  virtual ~CanvasPolygon() { delete[] points; }
  void Draw()
    {
      canvas->Foreground = color;
      canvas->DrawPolygon(filled,points,npoints);
    }

};
#endif
