/************************************************************************
 *
 * Copyright (C) 2014-2018 IRCAD France
 * Copyright (C) 2014-2018 IHU Strasbourg
 *
 * This file is part of Sight.
 *
 * Sight is free software: you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Sight is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with Sight. If not, see <https://www.gnu.org/licenses/>.
 *
 ***********************************************************************/

#include "fwRenderOgre/interactor/TrackballInteractor.hpp"
#include "fwRenderOgre/Layer.hpp"
#include "fwRenderOgre/registry/macros.hpp"

#include <fwCom/Signal.hxx>

#include <fwServices/registry/ActiveWorkers.hpp>

#include <OGRE/OgreCamera.h>
#include <OGRE/OgreNode.h>
#include <OGRE/OgreSceneNode.h>
#include <OGRE/OgreVector3.h>

fwRenderOgreRegisterInteractorMacro( ::fwRenderOgre::interactor::TrackballInteractor );

namespace fwRenderOgre
{
namespace interactor
{

// ----------------------------------------------------------------------------

TrackballInteractor::TrackballInteractor() :
    IMovementInteractor(),
    m_width(1),
    m_height(1),
    m_animate(false)
{
}

// ----------------------------------------------------------------------------

TrackballInteractor::~TrackballInteractor()
{
    // Join the animation thread if necessary
    if(m_timer)
    {
        m_animate = false;
        m_timer->stop();
        m_timer.reset();
    }
}
// ----------------------------------------------------------------------------

void TrackballInteractor::mouseMoveEvent(MouseButton button, int, int, int dx, int dy)
{
    if(button == LEFT)
    {
        cameraRotate(dx, dy);
    }
    else if(button == MIDDLE)
    {
        cameraTranslate(dx, dy);
    }
}

// ----------------------------------------------------------------------------

void TrackballInteractor::wheelEvent(int delta, int /*x*/, int /*y*/)
{
    // The zoom factor is reduced when coming closer and increased when going away
    const float fNewZoom = (delta > 0) ? m_fZoom * 0.85f : m_fZoom / 0.85f;

    // Moreover we cannot pass through the center of the trackball
    const float z = (m_fZoom - fNewZoom) * 200.f / (m_mouseScale );

    // Update the center of interest for future rotations
    m_lookAtZ -= z;

    m_fZoom = fNewZoom;

    // Last, translate the camera
    ::Ogre::Camera* camera     = m_sceneManager->getCamera(::fwRenderOgre::Layer::DEFAULT_CAMERA_NAME);
    ::Ogre::SceneNode* camNode = camera->getParentSceneNode();
    camNode->translate( ::Ogre::Vector3(0, 0, -1)*z, ::Ogre::Node::TS_LOCAL );
}

// ----------------------------------------------------------------------------

void TrackballInteractor::resizeEvent(int x, int y)
{
    m_width  = x;
    m_height = y;
}

// ----------------------------------------------------------------------------

void TrackballInteractor::keyPressEvent(int key)
{
    if(key == 'R' || key == 'r')
    {
        auto sig = this->signal<ResetCameraSignalType>( s_RESET_CAMERA_SIG );
        sig->asyncEmit();
    }

    if(key == 'A' || key == 'a')
    {
        m_animate = !m_animate;

        if(!m_animate)
        {
            m_timer->stop();
            m_timer.reset();
        }

        if(m_animate)
        {
            // We use a timer on the main thread instead of a separate thread.
            // Otherwise we get random visual errors even if we mutex all the functions.
            auto worker = ::fwServices::registry::ActiveWorkers::getDefault()->getDefaultWorker();
            m_timer = worker->createTimer();

            m_timer->setFunction([&]()
                    {
                        this->cameraRotate(10, 0);
                        m_sigRenderRequested->asyncEmit();
                    } );
            m_timer->setDuration(std::chrono::milliseconds(33));
            m_timer->start();
        }
    }
}

//------------------------------------------------------------------------------

void TrackballInteractor::keyReleaseEvent(int)
{
}

//------------------------------------------------------------------------------

void TrackballInteractor::buttonReleaseEvent(MouseButton, int, int)
{
}

//------------------------------------------------------------------------------

void TrackballInteractor::buttonPressEvent(MouseButton, int, int)
{
}

//------------------------------------------------------------------------------

void TrackballInteractor::focusInEvent()
{
}

//------------------------------------------------------------------------------

void TrackballInteractor::focusOutEvent()
{
}

// ----------------------------------------------------------------------------

void TrackballInteractor::cameraRotate(int dx, int dy)
{
    ::Ogre::Real dx_float = static_cast< ::Ogre::Real>(dx);
    ::Ogre::Real dy_float = static_cast< ::Ogre::Real>(dy);

    ::Ogre::Camera* camera     = m_sceneManager->getCamera(::fwRenderOgre::Layer::DEFAULT_CAMERA_NAME);
    ::Ogre::SceneNode* camNode = camera->getParentSceneNode();

    // Current orientation of the camera
    ::Ogre::Quaternion orientation = camNode->getOrientation();
    ::Ogre::Vector3 viewRight      = orientation.xAxis();
    ::Ogre::Vector3 viewUp         = orientation.yAxis();

    // Rotate around the right vector according to the dy of the mouse
    {
        // 1 - Move to the center of the target
        camNode->translate(::Ogre::Vector3(0.f, 0.f, -m_lookAtZ), ::Ogre::Node::TS_LOCAL);

        // 2 - Find rotation axis. We project the mouse movement onto the right and up vectors of the camera
        // We take the absolute to get a positive axis, and then we invert the angle when needed to rotate smoothly
        // Otherwise we would get a weird inversion
        ::Ogre::Vector3 vecX(std::abs(dy_float), 0.f, 0.f);
        ::Ogre::Vector3 rotateX = vecX * viewRight;
        rotateX.normalise();

        // 3 - Now determine the rotation direction
        if(rotateX.dotProduct(::Ogre::Vector3(1.f, 0.f, 0.f)) < 0.f)
        {
            dy_float *= -1;
        }

        // 4 - Compute the angle so that we can rotate around 180 degrees by sliding the whole window
        float angle = (dy_float * ::Ogre::Math::PI / static_cast< float>(m_height));

        // 5 - Apply the rotation on the scene node
        ::Ogre::Quaternion rotate(::Ogre::Radian(angle), rotateX);
        camNode->rotate( rotate );

        // 6 - Go backward in the inverse direction
        camNode->translate(::Ogre::Vector3(0.f, 0.f, m_lookAtZ), ::Ogre::Node::TS_LOCAL);
    }

    // Rotate around the up vector according to the dx of the mouse
    {
        // 1 - Move to the center of the target
        camNode->translate(::Ogre::Vector3(0.f, 0.f, -m_lookAtZ), ::Ogre::Node::TS_LOCAL);

        // 2 - Find rotation axis. We project the mouse movement onto the right and up vectors of the camera
        // We take the absolute to get a positive axis, and then we invert the angle when needed to rotate smoothly
        // Otherwise we would get a weird inversion
        ::Ogre::Vector3 vecY(0.f, std::abs(dx_float), 0.f);
        ::Ogre::Vector3 rotateY = vecY * viewUp;
        rotateY.normalise();

        // 3 - Now determine the rotation direction
        if(rotateY.dotProduct(::Ogre::Vector3(0.f, 1.f, 0.f)) < 0.f)
        {
            dx_float *= -1;
        }

        // 4 - Compute the angle so that we can rotate around 180 degrees by sliding the whole window
        float angle = (dx_float * ::Ogre::Math::PI / static_cast< float>(m_width));

        // 5 - Apply the rotation on the scene node
        ::Ogre::Quaternion rotate(::Ogre::Radian(angle), rotateY);
        camNode->rotate( rotate );

        // 6 - Go backward in the inverse direction
        camNode->translate(::Ogre::Vector3(0.f, 0.f, m_lookAtZ), ::Ogre::Node::TS_LOCAL);
    }
}

// ----------------------------------------------------------------------------

void TrackballInteractor::cameraTranslate(int xmove, int ymove)
{
    float dx = static_cast<float>(xmove) / (m_mouseScale * 10.f);
    float dy = static_cast<float>(-ymove) / (m_mouseScale * 10.f);
    ::Ogre::Camera* camera     = m_sceneManager->getCamera(::fwRenderOgre::Layer::DEFAULT_CAMERA_NAME);
    ::Ogre::SceneNode* camNode = camera->getParentSceneNode();

    ::Ogre::Vector3 vec(dx, dy, 0.f);

    camNode->translate(vec, ::Ogre::Node::TS_LOCAL);
}

// ----------------------------------------------------------------------------

} // namespace interactor
} // namespace fwRenderOgre
