use core::hash::Hash;
use malachite_base::chars::exhaustive::exhaustive_chars;
use malachite_base::chars::random::graphic_weighted_random_char_inclusive_range;
use malachite_base::num::random::geometric::geometric_random_unsigneds;
use malachite_base::num::random::random_primitive_ints;
use malachite_base::random::{Seed, EXAMPLE_SEED};
use malachite_base::test_util::vecs::random::random_vecs_helper_helper;
use malachite_base::vecs::random::random_unique_vecs_min_length;
use std::fmt::Debug;

fn random_unique_vecs_min_length_helper<
    T: Clone + Debug + Eq + Hash + Ord,
    I: Clone + Iterator<Item = T>,
>(
    min_length: u64,
    xs_gen: &dyn Fn(Seed) -> I,
    mean_length_numerator: u64,
    mean_length_denominator: u64,
    expected_values: &[&[T]],
    expected_common_values: &[(&[T], usize)],
    expected_median: (&[T], Option<&[T]>),
) {
    random_vecs_helper_helper(
        random_unique_vecs_min_length(
            EXAMPLE_SEED,
            min_length,
            xs_gen,
            mean_length_numerator,
            mean_length_denominator,
        ),
        expected_values,
        expected_common_values,
        expected_median,
    );
}

#[test]
fn test_random_unique_vecs_min_length() {
    random_unique_vecs_min_length_helper(
        0,
        &random_primitive_ints::<u8>,
        4,
        1,
        &[
            &[],
            &[85, 11, 136, 200, 235, 134, 203, 223, 38, 217, 177, 162, 32, 166],
            &[234, 30, 218, 90],
            &[106, 9, 216, 204],
            &[151],
            &[],
            &[213, 97, 253, 78, 91],
            &[39, 191],
            &[175, 170, 232, 233],
            &[],
            &[2, 35, 22, 217, 198, 114],
            &[],
            &[],
            &[17, 32, 173, 114, 65, 121, 222, 25, 144, 148, 79],
            &[115, 52, 73, 69, 137, 91, 153, 178],
            &[],
            &[112, 34, 95],
            &[],
            &[106, 167, 197, 130, 168],
            &[122, 207, 172, 177, 86, 150, 221, 218, 101],
        ],
        &[
            (&[], 199913),
            (&[7], 705),
            (&[25], 689),
            (&[184], 681),
            (&[213], 681),
            (&[255], 676),
            (&[215], 675),
            (&[54], 673),
            (&[122], 672),
            (&[207], 672),
        ],
        (&[96], None),
    );
    random_unique_vecs_min_length_helper(
        3,
        &random_primitive_ints::<u8>,
        7,
        1,
        &[
            &[85, 11, 136],
            &[200, 235, 134, 203, 223, 38, 217, 177, 162, 32, 166, 234, 30, 218, 90, 106, 9],
            &[216, 204, 151, 213, 97, 253, 78],
            &[91, 39, 191, 175, 170, 232, 233],
            &[2, 35, 22, 217],
            &[198, 114, 17],
            &[32, 173, 114, 65, 121, 222, 25, 144],
            &[148, 79, 115, 52, 73],
            &[69, 137, 91, 153, 178, 112, 34],
            &[95, 106, 167],
            &[197, 130, 168, 122, 207, 172, 177, 86, 150],
            &[221, 218, 101],
            &[115, 74, 9],
            &[123, 109, 52, 201, 159, 247, 250, 48, 133, 235, 196, 40, 97, 104],
            &[68, 190, 216, 7, 157, 43, 112, 217, 24, 11, 103],
            &[211, 84, 135],
            &[55, 29, 206, 89, 65, 191],
            &[51, 9, 79],
            &[148, 34, 22, 62, 3, 114, 118, 20],
            &[47, 194, 50, 32, 120, 176, 166, 23, 204, 248, 177, 238],
        ],
        &[
            (&[8, 94, 244], 3),
            (&[233, 14, 180], 3),
            (&[46, 247, 166], 3),
            (&[84, 118, 223], 3),
            (&[4, 52, 2], 2),
            (&[80, 2, 0], 2),
            (&[1, 116, 5], 2),
            (&[1, 5, 192], 2),
            (&[1, 96, 39], 2),
            (&[20, 10, 9], 2),
        ],
        (
            &[127, 218, 55, 167, 163, 19, 99, 71, 32, 117, 72, 38, 27, 29],
            Some(&[127, 218, 69, 59, 30, 91]),
        ),
    );
    random_unique_vecs_min_length_helper(
        0,
        &|seed| geometric_random_unsigneds::<u32>(seed, 32, 1),
        4,
        1,
        &[
            &[],
            &[1, 14, 42, 12, 141, 19, 9, 68, 16, 79, 21, 17, 41, 124],
            &[10, 1, 99, 0],
            &[77, 36, 2, 12],
            &[1],
            &[],
            &[103, 9, 19, 1, 5],
            &[7, 6],
            &[51, 159, 15, 18],
            &[],
            &[52, 75, 40, 64, 2, 26],
            &[],
            &[],
            &[67, 34, 51, 30, 31, 49, 43, 7, 5, 4, 3],
            &[14, 47, 24, 16, 52, 29, 1, 41],
            &[],
            &[13, 11, 62],
            &[],
            &[47, 3, 109, 42, 14],
            &[37, 86, 25, 96, 41, 13, 16, 42, 5],
        ],
        &[
            (&[], 199913),
            (&[0], 4861),
            (&[1], 4593),
            (&[2], 4498),
            (&[3], 4405),
            (&[4], 4330),
            (&[5], 4078),
            (&[6], 4050),
            (&[7], 3858),
            (&[8], 3848),
        ],
        (&[15, 3], None),
    );
    random_unique_vecs_min_length_helper(
        3,
        &|seed| geometric_random_unsigneds::<u32>(seed, 32, 1),
        7,
        1,
        &[
            &[1, 14, 42],
            &[12, 141, 19, 9, 68, 16, 79, 21, 17, 41, 124, 10, 1, 99, 0, 77, 36],
            &[2, 12, 1, 103, 9, 19, 5],
            &[7, 6, 51, 159, 15, 18, 52],
            &[75, 40, 64, 2],
            &[26, 67, 34],
            &[51, 30, 31, 49, 43, 7, 5, 4],
            &[3, 14, 47, 24, 16],
            &[52, 29, 1, 41, 13, 11, 62],
            &[47, 3, 109],
            &[42, 14, 37, 86, 25, 96, 41, 13, 16],
            &[42, 5, 20],
            &[2, 82, 74],
            &[7, 76, 11, 17, 127, 36, 20, 56, 89, 45, 6, 80, 3, 66],
            &[23, 43, 1, 25, 6, 41, 19, 97, 10, 13, 32],
            &[41, 7, 134],
            &[47, 105, 26, 10, 9, 25],
            &[94, 68, 109],
            &[1, 9, 84, 3, 43, 44, 28, 13],
            &[13, 5, 0, 31, 50, 75, 32, 4, 7, 37, 42, 6],
        ],
        &[
            (&[0, 1, 7], 13),
            (&[0, 2, 8], 12),
            (&[6, 2, 8], 12),
            (&[0, 2, 5], 11),
            (&[2, 7, 3], 11),
            (&[9, 3, 1], 11),
            (&[0, 9, 2], 10),
            (&[1, 8, 7], 10),
            (&[3, 5, 1], 10),
            (&[5, 0, 2], 10),
        ],
        (&[22, 28, 18, 89, 4, 52, 51], Some(&[22, 28, 19])),
    );
    random_unique_vecs_min_length_helper(
        0,
        &random_primitive_ints::<u8>,
        1,
        4,
        &[
            &[],
            &[],
            &[85],
            &[11],
            &[136, 200],
            &[],
            &[],
            &[],
            &[],
            &[],
            &[],
            &[],
            &[235, 134],
            &[203],
            &[],
            &[223, 38],
            &[],
            &[],
            &[],
            &[],
        ],
        &[
            (&[], 800023),
            (&[162], 692),
            (&[235], 690),
            (&[90], 688),
            (&[65], 687),
            (&[249], 686),
            (&[175], 684),
            (&[108], 683),
            (&[211], 682),
            (&[237], 680),
        ],
        (&[], None),
    );
    random_unique_vecs_min_length_helper(
        3,
        &random_primitive_ints::<u8>,
        13,
        4,
        &[
            &[85, 11, 136],
            &[200, 235, 134],
            &[203, 223, 38, 235],
            &[217, 177, 162, 32],
            &[166, 234, 30, 218, 90],
            &[106, 9, 216],
            &[204, 151, 213],
            &[97, 253, 78],
            &[91, 39, 191],
            &[175, 170, 232],
            &[233, 2, 35],
            &[22, 217, 198],
            &[114, 17, 32, 173, 65],
            &[121, 222, 173, 25],
            &[144, 148, 79],
            &[115, 52, 73, 69, 137],
            &[91, 153, 178],
            &[112, 34, 95],
            &[106, 167, 197],
            &[130, 168, 122],
        ],
        &[
            (&[8, 57, 90], 4),
            (&[185, 30, 26], 4),
            (&[106, 20, 234], 4),
            (&[152, 245, 81], 4),
            (&[108, 192, 235], 4),
            (&[75, 2, 5], 3),
            (&[0, 106, 9], 3),
            (&[2, 91, 90], 3),
            (&[212, 1, 8], 3),
            (&[49, 5, 75], 3),
        ],
        (&[128, 38, 40], Some(&[128, 38, 42])),
    );
    random_unique_vecs_min_length_helper(
        0,
        &|seed| {
            graphic_weighted_random_char_inclusive_range(
                seed,
                'a',
                exhaustive_chars().nth(200).unwrap(),
                1,
                1,
            )
        },
        4,
        1,
        &[
            &[],
            &['q', 'á', 'g', 'Á', 'À', 'È', '»', 'ì', '³', 'ą', 'â', 'Ã', 'ñ', 'Ā'],
            &['Ã', 'ª', 'ä', '´'],
            &['½', 'Á', 'Ï', 'ý'],
            &['j'],
            &[],
            &['ï', 'Ñ', 'u', 'Â', '½'],
            &['õ', 'x'],
            &['Â', 'ù', '¡', 'Ċ'],
            &[],
            &['¬', 'b', 'Ñ', 's', 'Â', 'r'],
            &[],
            &[],
            &['n', '¿', 'Þ', 'ô', 'Ø', 'º', 'ü', 't', '¬', 'j', 'Á'],
            &['±', 'è', 'k', 'æ', 'b', 'Î', 'ā', 'Ü'],
            &[],
            &['¹', '«', 'Î'],
            &[],
            &['~', '´', 'Ý', 'â', '¯'],
            &['é', 'æ', 'Þ', 'ö', 'g', 'Î', 'Ç', 'Ü', '¼'],
        ],
        &[
            (&[], 199913),
            (&['Ó'], 1270),
            (&['Â'], 1249),
            (&['§'], 1244),
            (&['¿'], 1243),
            (&['õ'], 1241),
            (&['ĉ'], 1234),
            (&['¤'], 1232),
            (&['¼'], 1232),
            (&['Ì'], 1229),
        ],
        (
            &['¶', 'Ă', 'ą', '®', 'Á', 'í', '¬', '¾', '¸', 'Ã', '}', 'ù', 'ý', '½', 'a'],
            Some(&['¶', 'Ă', 'ć']),
        ),
    );
    random_unique_vecs_min_length_helper(
        3,
        &|seed| {
            graphic_weighted_random_char_inclusive_range(
                seed,
                'a',
                exhaustive_chars().nth(200).unwrap(),
                1,
                1,
            )
        },
        7,
        1,
        &[
            &['q', 'á', 'g'],
            &['Á', 'À', 'È', '»', 'ì', '³', 'ą', 'â', 'g', 'Ã', 'ñ', 'Ā', 'ª', 'ä', '´', '½', 'Ï'],
            &['ý', 'j', 'ï', 'Ñ', 'u', 'Â', '½'],
            &['õ', 'x', 'Â', 'ù', '¡', 'Ċ', '¬'],
            &['b', 'Ñ', '¬', 's'],
            &['Â', 'r', 'n'],
            &['¿', 'Þ', 'ô', 'Ø', 'º', 'ü', 't', '¬'],
            &['j', 'Á', '±', 'è', 'k'],
            &['æ', 'b', 'Î', 'ā', 'Ü', '¹', '«'],
            &['Î', '~', '´'],
            &['Ý', 'â', '¯', 'é', 'æ', 'Þ', 'ö', 'g', 'Î'],
            &['Ç', 'Ü', '¼'],
            &['¡', '§', 'Ì'],
            &['±', 'd', 'ê', 'm', '®', 'ì', '¨', 'ý', 'þ', 'Ë', '{', 'Ü', 'z', '¼'],
            &['ï', 'ă', 'Ą', 'û', 'ª', 'ċ', 'x', 'À', 'ì', 'Õ', '½'],
            &['¢', '«', 'Ć'],
            &['{', 'È', 'ÿ', '½', '¢', 'ä'],
            &['Ë', 'Õ', 'ê'],
            &['º', 'ü', '×', '¨', 'Å', 'p', '°', 'Ó'],
            &['d', 'Ê', 'æ', 'ß', 'v', 'Ć', 'k', 'Ä', '±', 'È', '¥', 'o'],
        ],
        &[
            (&['a', 'Ã', '¤'], 4),
            (&['Ā', 'Ì', 'k'], 4),
            (&['w', 'm', 'u'], 3),
            (&['a', 'k', 'Ä'], 3),
            (&['b', 'Ã', 'n'], 3),
            (&['c', '|', 'Æ'], 3),
            (&['d', 'm', 'ï'], 3),
            (&['d', 'z', '¢'], 3),
            (&['d', '¯', 't'], 3),
            (&['f', 'g', '¹'], 3),
        ],
        (
            &['Ç', 'Ĉ', 's', 'ò', 'c', '¿', 'Å', 'Ô', 'Æ'],
            Some(&['Ç', 'Ĉ', 'v', '³', 'ò']),
        ),
    );
}

#[test]
#[should_panic]
fn random_unique_vecs_min_length_fail_1() {
    random_unique_vecs_min_length(EXAMPLE_SEED, 3, &random_primitive_ints::<u32>, 3, 1);
}

#[test]
#[should_panic]
fn random_unique_vecs_min_length_fail_2() {
    random_unique_vecs_min_length(EXAMPLE_SEED, 1, &random_primitive_ints::<u32>, 1, 0);
}

#[test]
#[should_panic]
fn random_unique_vecs_min_length_fail_3() {
    random_unique_vecs_min_length(
        EXAMPLE_SEED,
        0,
        &random_primitive_ints::<u32>,
        u64::MAX,
        u64::MAX - 1,
    );
}
