//keyval.c:

/*
 *      Copyright (C) Philipp 'ph3-der-loewe' Schafft - 2010-2019
 *
 *  This file is part of libroar a part of RoarAudio,
 *  a cross-platform sound system for both, home and professional use.
 *  See README for details.
 *
 *  This file is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 3
 *  as published by the Free Software Foundation.
 *
 *  libroar is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 *
 *  NOTE for everyone want's to change something and send patches:
 *  read README and HACKING! There a addition information on
 *  the license of this document you need to read before you send
 *  any patches.
 *
 *  NOTE for uses of non-GPL (LGPL,...) software using libesd, libartsc
 *  or libpulse*:
 *  The libs libroaresd, libroararts and libroarpulse link this lib
 *  and are therefore GPL. Because of this it may be illigal to use
 *  them with any software that uses libesd, libartsc or libpulse*.
 */

#include "libroar.h"

struct roar_keyval * roar_keyval_lookup (struct roar_keyval *  kv, const char * key, ssize_t len, int casesens) {
 int (*sc)(const char *s1, const char *s2) = strcasecmp;
 ssize_t i;

 ROAR_DBG("roar_keyval_lookup(kv=%p, key=%p'%s', len=%li, casesens=%i) = ?", kv, key, key, (long int)len, casesens);

 if ( kv == NULL || key == NULL ) {
  ROAR_DBG("roar_keyval_lookup(kv=%p, key=%p'%s', len=%li, casesens=%i) = NULL //error=FAULT", kv, key, key, (long int)len, casesens);
  roar_err_set(ROAR_ERROR_FAULT);
  return NULL;
 }

 if ( casesens )
  sc = strcmp;

 for (i = 0; len != -1 ? (i < len) : kv[i].key != NULL; i++) {
  if ( kv[i].key != NULL && !sc(key, kv[i].key) )
   return &(kv[i]);
 }

 ROAR_DBG("roar_keyval_lookup(kv=%p, key=%p'%s', len=%li, casesens=%i) = NULL //error=NOENT", kv, key, key, (long int)len, casesens);
 roar_err_set(ROAR_ERROR_NOENT);
 return NULL;
}

static inline int is_in (const char c, const char * delm) {
 for (; *delm != 0; delm++)
  if ( *delm == c )
   return 1;

 return 0;
}

static inline void skip_char(char * str) {
 memmove(str, str+1, roar_mm_strlen(str));
}

ssize_t              roar_keyval_split  (struct roar_keyval ** kv, char * str, const char * fdel, const char * kdel, int quotes) {
 struct roar_keyval * kvs;
 int    pos = -1;
 size_t len =  0;
 char * sp;
 char quote =  0;
 int last_was_seg = 0;

 if ( kv == NULL || str == NULL ) {
  roar_err_set(ROAR_ERROR_FAULT);
  return -1;
 }

 if ( fdel == NULL )
  fdel = " \t,";

 if ( kdel == NULL )
  kdel = "=:";

 // count num of segements:
 for (sp = str; *sp != 0; sp++) {
  if ( quote ) {
   if ( *sp == quote )
    quote = 0;
  } else if ( quotes && (*sp == '\'' || *sp == '\"') ) {
   quote = *sp;
  } else {
   if ( last_was_seg ) {
    last_was_seg = !is_in(*sp, fdel);
   } else {
    if ( !is_in(*sp, fdel) ) {
     last_was_seg = 1;
     len++;
    }
   }
  }
 }

 kvs = roar_mm_malloc(sizeof(struct roar_keyval)*(len+1));

 if ( kvs == NULL )
  return -1;

 *kv = kvs;

 // End of Array Mark:
 kvs[len].key   = NULL;
 kvs[len].value = NULL;

 // do the acctual filling:
 last_was_seg = 0;

 for (sp = str; *sp != 0; sp++) {
  if ( quote ) {
   if ( *sp == quote ) {
    skip_char(sp);
    quote = 0;
   } else {
    continue;
   }
  } else if ( quotes && (*sp == '\'' || *sp == '\"') ) {
   quote = *sp;
   skip_char(sp);
   continue;
  }

  if ( last_was_seg ) {
   if ( is_in(*sp, fdel) ) {
    last_was_seg = 0;
    *sp = 0;
   } else {
    last_was_seg = 1;
    if ( kvs[pos].value == NULL && is_in(*sp, kdel) ) {
     *sp = 0;
     kvs[pos].value = sp+1;
    }
   }
  } else {
   if ( !is_in(*sp, fdel) ) {
    last_was_seg = 1;
    pos++;
    kvs[pos].key   = sp;
    kvs[pos].value = NULL;
   }
  }
 }

 return len;
}

static inline void _copy_str(char ** dst, const char * src) {
 for (; *src; src++, (*dst)++) **dst = *src;
 *((*dst)++) = 0;
}

void * roar_keyval_copy(struct roar_keyval ** copy, const struct roar_keyval * src, ssize_t len) {
 size_t buflen = 0;
 ssize_t i;
 void * ret;
 char * p;
 struct roar_keyval * c;

 if ( copy == NULL || src == NULL ) {
  roar_err_set(ROAR_ERROR_FAULT);
  return NULL;
 }

 // TODO: optimize this.
 if ( len == -1 ) {
  len = 0;
  for (i = 0; src[i].key != NULL || src[i].value != NULL; i++)
   len++;
 }

 for (i = 0; i < len; i++) {
  if ( src[i].key != NULL )
   buflen += roar_mm_strlen(src[i].key) + 1;
  if ( src[i].value != NULL )
   buflen += roar_mm_strlen(src[i].value) + 1;
 }

 c   = roar_mm_malloc(len * sizeof(struct roar_keyval));
 if ( c == NULL )
  return NULL;

 memset(c, 0, len * sizeof(struct roar_keyval));

 ret = roar_mm_malloc(buflen);
 if ( ret == NULL )
  return NULL;

 memset(ret, 0, buflen);

 p = ret;

 for (i = 0; i < len; i++) {
  if ( src[i].key == NULL ) {
   c[i].key = NULL;
  } else {
   c[i].key = p;
   _copy_str(&p, src[i].key);
  }

  if ( src[i].value == NULL ) {
   c[i].value = NULL;
  } else {
   c[i].value = p;
   _copy_str(&p, src[i].value);
  }
 }

 *copy = c;
 return ret;
}

//ll
