context("Node tests")
test_that("redland library loads", {
  library(redland)
  # Add a line to this test to prevent 'Empty test' (skipped) msg.
  expect_true(require(redland))
})
test_that("Node constructor", {
  library(redland)
  world <- new("World")
  expect_false(is.null(world))
  
  # Test creating a blank node
  node <- new("Node", world)
  expect_false(is.null(node))
  expect_match(class(node), "Node")
  expect_match(getNodeType(node), "blank")
  
  # Test creating a blank node with librdf generated identifier
  node <- new("Node", world, blank=NULL)
  expect_match(class(node@librdf_node), "_p_librdf_node_s")
  expect_match(getNodeType(node), "blank")

  # Test creating a blank node with librdf generated identifier
  node <- new("Node", world, blank="_:1234")
  expect_match(class(node@librdf_node), "_p_librdf_node_s")
  expect_match(getNodeType(node), "blank")
  
  # Test creating a node with a literal value
  node <- new("Node", world, "Fee Fi Fo Fum")
  expect_match(class(node@librdf_node), "_p_librdf_node_s")
  expect_match(getNodeType(node), "literal")
  #   err <- try(freeNode(node), silent=TRUE)
  #   expect_match(class(err), not("try-error"))
  
  # Test creating a node with a literal value
  node <- new("Node", world, literal="Fee Fi Fo Fum")
  expect_match(class(node@librdf_node), "_p_librdf_node_s")
  expect_match(getNodeType(node), "literal")
  
  # Test creating a node with a URI value
  node <- new("Node", world, uri="http://www.example.com/")
  expect_match(class(node@librdf_node), "_p_librdf_node_s")
  expect_match(getNodeType(node), "resource")
  
  # Test that node creation fails if world is not provided or is null
  err <- try(node <- new("Node", literal="Fee Fi Fo Fum"), silent=TRUE)
  expect_match(class(err), "try-error")
  err <- try(node <- new("Node", world=NULL, literal="Fee Fi Fo Fum"), silent=TRUE)
  expect_match(class(err), "try-error")
  
  # Test that node values can be assigned and retrieved
  # a blank node is created with a unique identifier generated by librdf
  node <- new("Node", world, blank=NULL)
  expect_gt(nchar(getNodeValue(node)), 0)
  # a blank node is created with the user specified identifier, i.e. "_:id1"
  # THe blank node id will be returned as a string
  node <- new("Node", world, blank="_:b1")
  expect_match(getNodeValue(node), "_:b1")
  # a node type of 'literal' is created
  node <- new("Node", world, literal="A Node Value")
  expect_match(getNodeValue(node), "A Node Value")
  # a Node type of 'resource' is created
  node <- new("Node", world, uri="http://www.example.com")
  expect_match(getNodeValue(node), "http://www.example.com")
  # Create a literal node  without a language tag
  node <- new("Node", world, literal="just plain text")
  expect_match(getNodeValue(node), "just plain text")
  
  node <- new("Node", world, literal="Jean Pierre Rampal", language="fr")  
  expect_match(getNodeValue(node), '\"Jean Pierre Rampal"@fr\"')
  
  # Activate this check when testthat properly handles UTF-8 in tests. This
  # test works fine from the command line on all platforms, but when testthat
  # reads a test in, it is reading in the code as ASCII and ignoring the package
  # encoding, therefor the test is not performed with the correct string.
  #
  # Create a literal node, specifying a language encoding
  #node <- new("Node", world, literal="Gérard de La Martinière", language="fr")  
  #expect_match(getNodeValue(node), "\"Gérard de La Martinière\"@fr\"")
 
})