% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bag_imp.R
\name{step_bagimpute}
\alias{step_bagimpute}
\alias{imp_vars}
\alias{tidy.step_bagimpute}
\title{Imputation via Bagged Trees}
\usage{
step_bagimpute(
  recipe,
  ...,
  role = NA,
  trained = FALSE,
  impute_with = imp_vars(all_predictors()),
  trees = 25,
  models = NULL,
  options = list(keepX = FALSE),
  seed_val = sample.int(10^4, 1),
  skip = FALSE,
  id = rand_id("bagimpute")
)

imp_vars(...)

\method{tidy}{step_bagimpute}(x, ...)
}
\arguments{
\item{recipe}{A recipe object. The step will be added to the
sequence of operations for this recipe.}

\item{...}{One or more selector functions to choose variables. For
\code{step_bagimpute}, this indicates the variables to be imputed. When used with
\code{imp_vars}, the dots indicates which variables are used to predict the
missing data in each variable. See \code{\link[=selections]{selections()}} for more details. For the
\code{tidy} method, these are not currently used.}

\item{role}{Not used by this step since no new variables are created.}

\item{trained}{A logical to indicate if the quantities for
preprocessing have been estimated.}

\item{impute_with}{A call to \code{imp_vars} to specify which variables are used
to impute the variables that can include specific variable names separated
by commas or different selectors (see \code{\link[=selections]{selections()}}). If a column is
included in both lists to be imputed and to be an imputation predictor, it
will be removed from the latter and not used to impute itself.}

\item{trees}{An integer for the number bagged trees to use in each model.}

\item{models}{The \code{\link[ipred:ipredbagg]{ipred::ipredbagg()}} objects are stored here once this
bagged trees have be trained by \code{\link[=prep.recipe]{prep.recipe()}}.}

\item{options}{A list of options to \code{\link[ipred:ipredbagg]{ipred::ipredbagg()}}. Defaults are set
for the arguments \code{nbagg} and \code{keepX} but others can be passed in. \strong{Note}
that the arguments \code{X} and \code{y} should not be passed here.}

\item{seed_val}{A integer used to create reproducible models. The same seed
is used across all imputation models.}

\item{skip}{A logical. Should the step be skipped when the
recipe is baked by \code{\link[=bake.recipe]{bake.recipe()}}? While all operations are baked
when \code{\link[=prep.recipe]{prep.recipe()}} is run, some operations may not be able to be
conducted on new data (e.g. processing the outcome variable(s)).
Care should be taken when using \code{skip = TRUE} as it may affect
the computations for subsequent operations}

\item{id}{A character string that is unique to this step to identify it.}

\item{x}{A \code{step_bagimpute} object.}
}
\value{
An updated version of \code{recipe} with the new step added to the
sequence of existing steps (if any). For the \code{tidy} method, a tibble with
columns \code{terms} (the selectors or variables selected) and \code{model} (the
bagged tree object).
}
\description{
\code{step_bagimpute} creates a \emph{specification} of a recipe step that will
create bagged tree models to impute missing data.
}
\details{
For each variables requiring imputation, a bagged tree is created
where the outcome is the variable of interest and the predictors are any
other variables listed in the \code{impute_with} formula. One advantage to the
bagged tree is that is can accept predictors that have missing values
themselves. This imputation method can be used when the variable of interest
(and predictors) are numeric or categorical. Imputed categorical variables
will remain categorical. Also, integers will be imputed to integer too.

Note that if a variable that is to be imputed is also in \code{impute_with},
this variable will be ignored.

It is possible that missing values will still occur after imputation if a
large majority (or all) of the imputing variables are also missing.
}
\examples{
library(modeldata)
data("credit_data")

## missing data per column
vapply(credit_data, function(x) mean(is.na(x)), c(num = 0))

set.seed(342)
in_training <- sample(1:nrow(credit_data), 2000)

credit_tr <- credit_data[ in_training, ]
credit_te <- credit_data[-in_training, ]
missing_examples <- c(14, 394, 565)

rec <- recipe(Price ~ ., data = credit_tr)
\dontrun{
impute_rec <- rec \%>\%
  step_bagimpute(Status, Home, Marital, Job, Income, Assets, Debt)

imp_models <- prep(impute_rec, training = credit_tr)

imputed_te <- bake(imp_models, new_data = credit_te, everything())

credit_te[missing_examples,]
imputed_te[missing_examples, names(credit_te)]

tidy(impute_rec, number = 1)
tidy(imp_models, number = 1)

## Specifying which variables to imputate with

 impute_rec <- rec \%>\%
  step_bagimpute(Status, Home, Marital, Job, Income, Assets, Debt,
                 impute_with = imp_vars(Time, Age, Expenses),
                 # for quick execution, nbagg lowered
                 options = list(nbagg = 5, keepX = FALSE))

imp_models <- prep(impute_rec, training = credit_tr)

imputed_te <- bake(imp_models, new_data = credit_te, everything())

credit_te[missing_examples,]
imputed_te[missing_examples, names(credit_te)]

tidy(impute_rec, number = 1)
tidy(imp_models, number = 1)
}
}
\references{
Kuhn, M. and Johnson, K. (2013). \emph{Applied Predictive Modeling}.
Springer Verlag.
}
\concept{imputation}
\concept{preprocessing}
\keyword{datagen}
