# Copyright (c) 2017 The Johns Hopkins University/Applied Physics Laboratory
# All Rights Reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License"); you may
# not use this file except in compliance with the License. You may obtain
# a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
# WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
# License for the specific language governing permissions and limitations
# under the License.

import binascii
import logging
import sys

from kmip.core import enums
from kmip.demos import utils
from kmip.pie import client

# Real world example, assuming 'test' is a valid configuration:
#
# $ python kmip/demos/pie/encrypt.py -c test -m "My test message."
# INFO - Successfully created a new encryption key.
# INFO - Secret ID: 470
# INFO - Successfully activated the encryption key.
# INFO - Successfully encrypted the message.
# INFO - Cipher text: b'49cfacbb62659180c20dfbf9f7553488b3ea9ebeecd70ce2e5c4d4
# ece6def0d4'
# INFO - No autogenerated IV expected, since one was provided.
# INFO - Autogenerated IV: None
# $ python kmip/demos/pie/decrypt.py -c test -i 470 -m b'49cfacbb62659180c20df
# bf9f7553488b3ea9ebeecd70ce2e5c4d4ece6def0d4'
# INFO - Successfully decrypted the message.
# INFO - Plain text: 'My test message.'


if __name__ == '__main__':
    logger = utils.build_console_logger(logging.INFO)

    # Build and parse arguments
    parser = utils.build_cli_parser(enums.Operation.DECRYPT)
    opts, args = parser.parse_args(sys.argv[1:])
    config = opts.config
    uuid = opts.uuid
    message = opts.message

    if not message.startswith("b"):
        raise ValueError("The message should be a byte string (e.g., b'...').")
    else:
        message = binascii.unhexlify(message[1:])

    # Build the client and connect to the server
    with client.ProxyKmipClient(
            config=config,
            config_file=opts.config_file
    ) as client:
        # Decrypt the cipher text with the encryption key.
        try:
            plain_text = client.decrypt(
                message,
                uid=uuid,
                cryptographic_parameters={
                    'cryptographic_algorithm':
                        enums.CryptographicAlgorithm.AES,
                    'block_cipher_mode': enums.BlockCipherMode.CBC,
                    'padding_method': enums.PaddingMethod.ANSI_X923
                },
                iv_counter_nonce=(
                    b'\x01\x7D\x45\xA0\x88\x08\x11\x11'
                    b'\xF0\x00\x12\xFF\x7A\x3A\x36\x90'
                )
            )
            logger.info("Successfully decrypted the message.")
            logger.info("Plain text: '{0}'".format(plain_text.decode('utf-8')))
        except Exception as e:
            logger.error(e)
