/*
# PostgreSQL Database Modeler (pgModeler)
#
# Copyright 2006-2019 - Raphael Araújo e Silva <raphael@pgmodeler.io>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation version 3.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# The complete text of GPLv3 is at LICENSE file on source code root directory.
# Also, you can get the complete GNU General Public License at <http://www.gnu.org/licenses/>
*/

/**
\ingroup libpgmodeler_ui
\class ModelDatabaseDiffForm
\brief Implements the operations to compare and generate a diff between model and database via form.
*/

#ifndef MODEL_DATABASE_DIFF_FORM
#define MODEL_DATABASE_DIFF_FORM

#include "ui_modeldatabasediffform.h"
#include "modelsdiffhelper.h"
#include "databaseimporthelper.h"
#include "modelexporthelper.h"
#include "hinttextwidget.h"
#include "syntaxhighlighter.h"
#include "htmlitemdelegate.h"
#include "numberedtexteditor.h"
#include "baseconfigwidget.h"
#include <QThread>

class ModelDatabaseDiffForm: public BaseConfigWidget, public Ui::ModelDatabaseDiffForm {
	private:
		Q_OBJECT

		/*! \brief Indicates if the full output generated during the process should be displayed
		 * When this attribute is true, only errors and some key info messages are displayed. */
		static bool low_verbosity;

		static map<QString, attribs_map> config_params;

		QEventLoop event_loop;

		bool is_adding_new_preset;

		NumberedTextEditor *sqlcode_txt;

		//! \brief Custom delegate used to paint html texts in output tree
		HtmlItemDelegate *htmlitem_del;

		//! \brief Hint texts used on the diff options
		HintTextWidget *apply_on_server_ht, *store_in_file_ht,
		*import_sys_objs_ht, *import_ext_objs_ht, *keep_cluster_objs_ht,
		*trunc_tables_ht, *ignore_errors_ht, *force_recreation_ht,
		*cascade_mode_ht, *pgsql_ver_ht, *recreate_unmod_ht,
		*keep_obj_perms_ht, *ignore_duplic_ht, *reuse_sequences_ht,
		*preserve_db_name_ht, *dont_drop_missing_objs_ht,
		*ignore_error_codes_ht, *drop_missing_cols_constr_ht;

		//! \brief Syntax highlighter used on the diff preview tab
		SyntaxHighlighter *sqlcode_hl;

		//! \brief Helper that will execute the diff between models
		ModelsDiffHelper *diff_helper;

		//! \brief Helper that will execute the database import
		DatabaseImportHelper *import_helper, *src_import_helper;

		//! \brief Helper that will execute the diff export to database
		ModelExportHelper *export_helper;

		//! \brief Threads that will execute each step: import, diff, export
		QThread *import_thread, *diff_thread, *export_thread, *src_import_thread;

		//! \brief Tree items generated in each diff step
		QTreeWidgetItem *import_item, *diff_item, *export_item, *src_import_item;

		/*! \brief This is the model used in the diff process representing the source.
		 * It can be the modelo loaded from file or a representation of the source database (when comparing two dbs) */
		DatabaseModel *source_model,

		//! \brief This is the model loaded from file
		*loaded_model,

		//! \brief This is the model generated by the reverse engineering step
		*imported_model;

		//! \brief Connection used to export the diff to database
		Connection *export_conn;

		//! \brief PostgreSQL version used by the diff process
		QString pgsql_ver;

		int diff_progress, curr_step, total_steps;

		bool process_paused;

		void closeEvent(QCloseEvent *event);
		void showEvent(QShowEvent *);

		//! \brief Creates the helpers and threads
		void createThread(unsigned thread_id);

		//! \brief Destroy the helpers and threads
		void destroyThread(unsigned thread_id);

		//! \brief Destroy the imported model
		void destroyModel(void);

		void clearOutput(void);
		void resetForm(void);
		void resetButtons(void);
		void saveDiffToFile(void);
		void finishDiff(void);

		//! \brief Returns true when one or more threads of the whole diff process are running.
		bool isThreadsRunning(void);

		//! \brief Constants used to reference the thread/helper to be handled in createThread() and destroyThread()
		static constexpr unsigned SrcImportThread=0,
		ImportThread=1,
		DiffThread=2,
		ExportThread=3;

		//! \brief Applies the loaded configurations to the form. In this widget only list the loaded presets
		virtual void applyConfiguration(void);

		//! \brief Loads a set of configurations from a file
		virtual void loadConfiguration(void);

		//! \brief Saves the current settings to a file
		virtual void saveConfiguration(void);

		void togglePresetConfiguration(bool toggle, bool is_edit = false);
		void enablePresetButtons(void);

	public:
		ModelDatabaseDiffForm(QWidget * parent = nullptr, Qt::WindowFlags flags = Qt::Widget);
		~ModelDatabaseDiffForm(void);

		//! \brief Makes the form behaves like a QDialog by running it from an event loop. The event loop is finished when the user clicks close
		void exec(void);

		void setModelWidget(ModelWidget *model_wgt);

		//! \brief Defines if all the output generated during the import process should be displayed
		static void setLowVerbosity(bool value);

	private slots:
		void listDatabases(void);
		void enableDiffMode(void);
		void generateDiff(void);
		void cancelOperation(bool cancel_by_user);
		void updateProgress(int progress, QString msg, ObjectType obj_type, QString cmd=QString());
		void updateDiffInfo(ObjectsDiffInfo diff_info);
		void captureThreadError(Exception e);
		void handleImportFinished(Exception e);
		void handleDiffFinished(void);
		void handleExportFinished(void);
		void handleErrorIgnored(QString err_code, QString err_msg, QString cmd);
		void selectOutputFile(void);
		void importDatabase(unsigned thread_id);
		void diffModels(void);
		void exportDiff(bool confirm=true);
		void filterDiffInfos(void);
		void loadDiffInSQLTool(void);
		void selectPreset(void);
		void removePreset(void);
		void savePreset(void);

		//! \brief Destroy the current configuration file and makes a copy of the default one located at conf/defaults
		virtual void restoreDefaults(void);

	signals:
		/*! \brief This signal is emitted whenever the user changes the connections settings
		within this widget without use the main configurations dialog */
		void s_connectionsUpdateRequest(void);

		/*! \brief This signal is emitted whenever the user wants to load the generated diff in the sql tool
		 * The signal contains the connection id, the database name and the temp filename that is generated containing
		 * the commands to be loaded */
		void s_loadDiffInSQLTool(QString conn_id, QString database, QString sql_file);
};

#endif
