/******************************************************************************
* Copyright (c) 2019, Helix Re Inc. nicolas@helix.re
*
* All rights reserved.
*
* Redistribution and use in source and binary forms, with or without
* modification, are permitted provided that the following
* conditions are met:
*
*     * Redistributions of source code must retain the above copyright
*       notice, this list of conditions and the following disclaimer.
*     * Redistributions in binary form must reproduce the above copyright
*       notice, this list of conditions and the following disclaimer in
*       the documentation and/or other materials provided
*       with the distribution.
*     * Neither the name of Helix Re Inc. nor the
*       names of its contributors may be used to endorse or promote
*       products derived from this software without specific prior
*       written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
* "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
* LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
* FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
* COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
* INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
* BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
* OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
* AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
* OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
* OF SUCH DAMAGE.
****************************************************************************/

#include <pdal/pdal_test_main.hpp>

#include <pdal/PointView.hpp>
#include <io/BufferReader.hpp>
#include <io/TextReader.hpp>
#include <filters/CovarianceFeaturesFilter.hpp>

#include "Support.hpp"

namespace pdal {

TEST(DimensionalityTest, Linearity)
{
    using namespace Dimension;

    PointTable table;
    table.layout()->registerDims({Id::X, Id::Y, Id::Z});

    BufferReader bufferReader;
    CovarianceFeaturesFilter filter;
    Options ops;
    ops.add("knn", 3);
    filter.setInput(bufferReader);
    filter.setOptions(ops);
    filter.prepare(table);

    PointViewPtr view(new PointView(table));
    view->setField(Id::X, 0,0);
    view->setField(Id::X, 1, 0);
    view->setField(Id::X, 2, 0);
    view->setField(Id::Y, 0, 0);
    view->setField(Id::Y, 1, 0);
    view->setField(Id::Y, 2, 0);
    view->setField(Id::Z, 0, 0);
    view->setField(Id::Z, 1, 1);
    view->setField(Id::Z, 2, 2);
    bufferReader.addView(view);

    PointViewSet viewSet = filter.execute(table);
    PointViewPtr outView = *viewSet.begin();

    Dimension::Id planarity = table.layout()->findDim("Planarity");
    Dimension::Id verticality = table.layout()->findDim("Verticality");
    Dimension::Id linearity = table.layout()->findDim("Linearity");
    Dimension::Id scattering = table.layout()->findDim("Scattering");

    for (point_count_t i =0; i < outView->size(); i++)
    {
        ASSERT_FLOAT_EQ(outView->getFieldAs<float>(linearity, i) ,1);
        ASSERT_FLOAT_EQ(outView->getFieldAs<float>(verticality, i) ,1);
        ASSERT_FLOAT_EQ(outView->getFieldAs<float>(planarity, i) ,0);
        ASSERT_FLOAT_EQ(outView->getFieldAs<float>(scattering, i) ,0);
    }
}

TEST(DimensionalityTest, Planarity)
{
    using namespace Dimension;

    PointTable table;
    table.layout()->registerDims({Id::X, Id::Y, Id::Z});

    BufferReader bufferReader;
    CovarianceFeaturesFilter filter;
    Options ops;
    filter.setInput(bufferReader);
    filter.prepare(table);

    PointViewPtr view(new PointView(table));
    view->setField(Id::X, 0,0);
    view->setField(Id::X, 1, 1);
    view->setField(Id::X, 2, 1);
    view->setField(Id::X, 3, 0);
    view->setField(Id::Y, 0, 0);
    view->setField(Id::Y, 1, 0);
    view->setField(Id::Y, 2, 1);
    view->setField(Id::Y, 3, 1);
    view->setField(Id::Z, 0, 0);
    view->setField(Id::Z, 1, 0);
    view->setField(Id::Z, 2, 0);
    view->setField(Id::Z, 3, 0);

    bufferReader.addView(view);


    PointViewSet viewSet = filter.execute(table);
    PointViewPtr outView = *viewSet.begin();

    Dimension::Id planarity = table.layout()->findDim("Planarity");
    Dimension::Id verticality = table.layout()->findDim("Verticality");
    Dimension::Id linearity = table.layout()->findDim("Linearity");
    Dimension::Id scattering = table.layout()->findDim("Scattering");

    for (point_count_t i =0; i < outView->size(); i++)
    {
        ASSERT_LE(outView->getFieldAs<float>(linearity, i) ,0.5);
        ASSERT_FLOAT_EQ(outView->getFieldAs<float>(verticality, i) ,0);
        ASSERT_FLOAT_EQ(outView->getFieldAs<float>(planarity, i) ,1);
        ASSERT_FLOAT_EQ(outView->getFieldAs<float>(scattering, i) ,0);
    }
}

TEST(DimensionalityTest, Scattering)
{
    using namespace Dimension;

    PointTable table;
    table.layout()->registerDims({Id::X, Id::Y, Id::Z});

    BufferReader bufferReader;
    CovarianceFeaturesFilter filter;
    Options ops;
    filter.setInput(bufferReader);
    filter.prepare(table);

    PointViewPtr view(new PointView(table));
    view->setField(Id::X, 0,0);
    view->setField(Id::X, 1, 1);
    view->setField(Id::X, 2, 1);
    view->setField(Id::X, 3, 0);
    view->setField(Id::Y, 0, 0);
    view->setField(Id::Y, 1, 0);
    view->setField(Id::Y, 2, 1);
    view->setField(Id::Y, 3, 1);
    view->setField(Id::Z, 0, 0);
    view->setField(Id::Z, 1, 0);
    view->setField(Id::Z, 2, 1);
    view->setField(Id::Z, 3, 2);

    bufferReader.addView(view);


    PointViewSet viewSet = filter.execute(table);
    PointViewPtr outView = *viewSet.begin();

    Dimension::Id planarity = table.layout()->findDim("Planarity");
    Dimension::Id verticality = table.layout()->findDim("Verticality");
    Dimension::Id linearity = table.layout()->findDim("Linearity");
    Dimension::Id scattering = table.layout()->findDim("Scattering");

    for (point_count_t i =0; i < outView->size(); i++)
    {
        ASSERT_LE(outView->getFieldAs<float>(linearity, i) ,0.5);
        ASSERT_GE(outView->getFieldAs<float>(verticality, i) ,0.5);
        ASSERT_LE(outView->getFieldAs<float>(planarity, i) ,1);
        ASSERT_GE(outView->getFieldAs<float>(scattering, i) ,0.1);
    }
}

}