///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/gui/properties/BooleanPropertyUI.h>
#include <core/undo/UndoManager.h>
#include <core/viewport/input/XFormManager.h>

#include "OrientationDataChannel.h"
#include <atomviz/atoms/AtomsObject.h>

namespace AtomViz {

IMPLEMENT_SERIALIZABLE_PLUGIN_CLASS(OrientationDataChannel, DataChannel)

/******************************************************************************
* Serialization constructor.
******************************************************************************/
OrientationDataChannel::OrientationDataChannel(bool isLoading) : DataChannel(isLoading)
{
}

/******************************************************************************
* Special constructor to create a standard data channel.
******************************************************************************/
OrientationDataChannel::OrientationDataChannel(DataChannelIdentifier which) : DataChannel(which)
{
}

/******************************************************************************
* Render the orientation tripods into the viewport.
******************************************************************************/
void OrientationDataChannel::render(TimeTicks time, Viewport* vp, AtomsObject* atoms, ObjectNode* contextNode)
{
	DataChannel* posChannel = atoms->getStandardDataChannel(DataChannel::PositionChannel);
	if(!posChannel) return;

	// Disable z-buffer.
	vp->setDepthTest(false);
	vp->setLightingEnabled(false);

	const Point3* p = posChannel->constDataPoint3();
	const Quaternion* q = constDataQuaternion();
	for(size_t i = size(); i != 0; --i, ++p, ++q) {
		if(q->X == 0 && q->Y == 0 && q->Z == 0 && q->W == 0) continue;
		AffineTransformation tm = AffineTransformation::rotation(*q);
		tm.setTranslation(*p - ORIGIN);
		XFORM_MANAGER.renderTripod(vp, tm);
	}

	// Re-enable z-buffer.
	vp->setDepthTest(true);
}

/******************************************************************************
* Renders the channel's contents in high-quality mode to an offscreen buffer.
******************************************************************************/
void OrientationDataChannel::renderHQ(TimeTicks time, AtomsObject* atoms, const CameraViewDescription& view, ObjectNode* contextNode, int imageWidth, int imageHeight, Window3D* glcontext)
{
}

IMPLEMENT_PLUGIN_CLASS(OrientationDataChannelEditor, PropertiesEditor)

/******************************************************************************
* Sets up the UI widgets of the editor.
******************************************************************************/
void OrientationDataChannelEditor::createUI(const RolloutInsertionParameters& rolloutParams)
{
	// Create a rollout.
	QWidget* rollout = createRollout(tr("Atomic Orientations"), rolloutParams);

	QVBoxLayout* mainLayout = new QVBoxLayout(rollout);
	mainLayout->setContentsMargins(4,4,4,4);

	BooleanPropertyUI* showOrientationsUI = new BooleanPropertyUI(this, "isVisible", tr("Show orientation tripods"));
	mainLayout->addWidget(showOrientationsUI->checkBox());
}

};	// End of namespace AtomViz
