/*
 *  Copyright (c) 2007 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "Value.h"

#include "Color.h"
#include "Debug.h"
#include "Type.h"
#include "TypesManager_p.h"

#include "SharedPointer.h"

using namespace GTLCore;

struct Value::Private : public SharedPointerData {
  Private() : type(0) {}
  Private(const Private& rhs) : type(rhs.type) {
    if( type->dataType() == Type::ARRAY or type->dataType() == Type::VECTOR )
    {
      value.array = new std::vector< Value >( *rhs.value.array );
    } else {
      value = rhs.value;
    }
  }
  ~Private() {
    cleanup();
  }
  union {
    bool b;
    float f32;
    gtl_int32 i32;
    gtl_uint32 ui32;
    gtl_uint64 ui64;
    gtl_int64 i64;
    Color* color;
    std::vector< Value >* array;
  } value;
  const Type* type;
  void cleanup()
  {
    if( type and (type->dataType() == Type::ARRAY or type->dataType() == Type::VECTOR ) )
    {
      delete value.array;
    } else if ( type == Type::Color )
    {
      delete value.color;
    }
  }
};

Value::Value() : d(new Private)
{
  d->ref();
  d->type = Type::Undefined;
}

Value::Value(float v) : d(new Private)
{
  d->ref();
  d->value.f32 = v;
  d->type = Type::Float32;
}

Value::Value(bool v) : d(new Private)
{
  d->ref();
  d->value.b = v;
  d->type = Type::Boolean;
}

Value::Value(gtl_int32 v) : d(new Private)
{
  d->ref();
  d->value.i32 = v;
  d->type = Type::Integer32;
}

Value::Value(gtl_uint32 v) : d(new Private)
{
  d->ref();
  d->value.ui32 = v;
  d->type = Type::UnsignedInteger32;
}

Value::Value(gtl_uint64 v) : d(new Private)
{
  d->ref();
  d->value.ui64 = v;
  d->type = Type::UnsignedInteger64;
}

Value::Value(const Color& color) : d(new Private)
{
  d->ref();
  d->value.color = new Color(color);
  d->type = Type::Color;
}

Value::Value(const std::vector< Value >& v, const GTLCore::Type* _type ) : d(new Private)
{
  d->ref();
  d->value.array = new std::vector< Value >( v );
  GTL_ASSERT(d->value.array);
  if( _type )
  {
    d->type = _type;
  } else {
    d->type = TypesManager::getArray( v[0].type() );
  }
  GTL_ASSERT(d->type->dataType() != Type::VECTOR or d->type->vectorSize() == v.size() );
}

GTL_SHARED_DATA(Value)

bool Value::operator==(const Value& rhs) const
{
  if(d == rhs.d) return true;
  if( d->type == rhs.d->type )
  {
    switch( d->type->dataType())
    {
      case Type::BOOLEAN:
        return d->value.b == rhs.d->value.b;
      case Type::FLOAT32:
        return d->value.f32 == rhs.d->value.f32;
      case Type::INTEGER32:
        return d->value.i32 == rhs.d->value.i32;
      case Type::UNSIGNED_INTEGER32:
        return d->value.ui32 == rhs.d->value.ui32;
      case Type::ARRAY:
      case Type::VECTOR:
      {
        if( d->value.array->size() == rhs.d->value.array->size() )
        {
          for( std::size_t i = 0; i < d->value.array->size(); ++i)
          {
            if( not( (*d->value.array)[i] == (*rhs.d->value.array)[i] ) )
            {
              return false;
            }
          }
          return true;
        }
      }
      case Type::STRUCTURE:
      {
        if(d->type == Type::Color)
        {
          return *d->value.color == *rhs.d->value.color;
        }
      }
      default:
        return false;
    }
  }
  return false;
}

bool Value::isValid() const
{
  return d->type != Type::Undefined;
}

#define RETURN_AS( _type_ ) \
  switch( d->type->dataType()) \
  { \
    case Type::BOOLEAN: \
      return (_type_)d->value.b; \
    case Type::FLOAT32: \
      return (_type_)d->value.f32; \
    case Type::INTEGER32: \
      return (_type_)d->value.i32; \
    case Type::UNSIGNED_INTEGER32: \
      return (_type_)d->value.ui32; \
    case Type::INTEGER64: \
      return (_type_)d->value.i64; \
    case Type::UNSIGNED_INTEGER64: \
      return (_type_)d->value.ui64; \
    default: \
      GTL_ABORT("Can't convert to that type."); \
      return (_type_)0; \
 }

float Value::asFloat32() const
{
  RETURN_AS(float);
}

void Value::setFloat32( float _v )
{
  deref();
  d->value.f32 = _v;
  d->type = Type::Float32;
}

bool Value::asBoolean() const
{
  RETURN_AS(bool);
}

void Value::setBoolean( bool _v)
{
  deref();
  d->value.b = _v;
  d->type = Type::Boolean;
}

gtl_int32 Value::asInt32() const
{
  RETURN_AS(int);
}

void Value::setInt32(gtl_int32 v)
{
  deref();
  d->value.i32 = v;
  d->type = Type::Integer32;
}

gtl_int64 Value::asInt64() const
{
  RETURN_AS(gtl_int64);
}

void Value::setInt64(gtl_int64 v)
{
  deref();
  d->value.i64 = v;
  d->type = Type::Integer64;
}

gtl_uint32 Value::asUnsignedInt32() const
{
  RETURN_AS(unsigned int);
}

const Type* Value::type() const
{
  return d->type;
}

const std::vector< Value >* Value::asArray() const
{
  if( d->type->dataType() == Type::ARRAY or d->type->dataType() == Type::VECTOR or d->type->isStructure() )
  {
    return d->value.array;
  } else {
    return 0;
  }
}

void Value::setArray( const std::vector< Value >& _array, const GTLCore::Type* _type )
{
  deref();
  d->cleanup();
  d->value.array = new std::vector< Value >( _array );
  if( _type )
  {
    d->type = _type;
  } else {
    d->type = TypesManager::getArray( _array[0].type() );
  }
}

Color Value::asColor() const
{
  if (d->type == Type::Color)
  {
    return *d->value.color;
  }
  return Color();
}

void Value::setColor(const Color& c)
{
  deref();
  if (d->type != Type::Color)
  {
    d->cleanup();
    d->type = Type::Color;
    d->value.color = new Color(c);
  } else {
    *d->value.color = c;
  }
}
