/**
* @license Apache-2.0
*
* Copyright (c) 2018 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

'use strict';

// MODULES //

var isnan = require( '@stdlib/math/base/assert/is-nan' );
var variance = require( './../../../../../base/dists/weibull/variance' );
var gamma = require( '@stdlib/math/base/special/gamma' );
var sqrt = require( '@stdlib/math/base/special/sqrt' );
var mean = require( './../../../../../base/dists/weibull/mean' );
var pow = require( '@stdlib/math/base/special/pow' );


// MAIN //

/**
* Returns the skewness of a Weibull distribution.
*
* @param {PositiveNumber} k - shape parameter
* @param {PositiveNumber} lambda - scale parameter
* @returns {number} skewness
*
* @example
* var v = skewness( 1.0, 1.0 );
* // returns 2.0
*
* @example
* var v = skewness( 4.0, 12.0 );
* // returns ~-0.087
*
* @example
* var v = skewness( 8.0, 2.0 );
* // returns ~-0.534
*
* @example
* var v = skewness( 1.0, -0.1 );
* // returns NaN
*
* @example
* var v = skewness( -0.1, 1.0 );
* // returns NaN
*
* @example
* var v = skewness( 2.0, NaN );
* // returns NaN
*
* @example
* var v = skewness( NaN, 2.0 );
* // returns NaN
*/
function skewness( k, lambda ) {
	var sigma2;
	var sigma;
	var out;
	var mu;
	if (
		isnan( k ) ||
		isnan( lambda ) ||
		k <= 0.0 ||
		lambda <= 0.0
	) {
		return NaN;
	}
	mu = mean( k, lambda );
	sigma2 = variance( k, lambda);
	sigma = sqrt( sigma2 );
	out = gamma( 1.0 + ( 3.0/k ) ) * pow( lambda, 3.0 );
	out -= ( 3.0*mu*sigma2 ) + pow( mu, 3.0 );
	out /= pow( sigma, 3.0 );
	return out;
}


// EXPORTS //

module.exports = skewness;
