/*
* @license Apache-2.0
*
* Copyright (c) 2021 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

// TypeScript Version: 2.0

/* tslint:disable:max-line-length */
/* tslint:disable:max-file-line-count */

import cdf = require( './../../../../../base/dists/geometric/cdf' );
import Geometric = require( './../../../../../base/dists/geometric/ctor' );
import entropy = require( './../../../../../base/dists/geometric/entropy' );
import kurtosis = require( './../../../../../base/dists/geometric/kurtosis' );
import logcdf = require( './../../../../../base/dists/geometric/logcdf' );
import logpmf = require( './../../../../../base/dists/geometric/logpmf' );
import mean = require( './../../../../../base/dists/geometric/mean' );
import median = require( './../../../../../base/dists/geometric/median' );
import mgf = require( './../../../../../base/dists/geometric/mgf' );
import mode = require( './../../../../../base/dists/geometric/mode' );
import pmf = require( './../../../../../base/dists/geometric/pmf' );
import quantile = require( './../../../../../base/dists/geometric/quantile' );
import skewness = require( './../../../../../base/dists/geometric/skewness' );
import stdev = require( './../../../../../base/dists/geometric/stdev' );
import variance = require( './../../../../../base/dists/geometric/variance' );

/**
* Interface describing the `geometric` namespace.
*/
interface Namespace {
	/**
	* Geometric distribution cumulative distribution function (CDF).
	*
	* @param x - input value
	* @param p - success probability
	* @returns evaluated CDF
	*
	* @example
	* var y = ns.cdf( 2.0, 0.5 );
	* // returns 0.875
	*
	* y = ns.cdf( 2.0, 0.1 );
	* // returns ~0.271
	*
	* var mycdf = ns.cdf.factory( 0.5 );
	* y = mycdf( 3.0 );
	* // returns 0.9375
	*
	* y = mycdf( 1.0 );
	* // returns 0.75
	*/
	cdf: typeof cdf;

	/**
	* Geometric distribution.
	*/
	Geometric: typeof Geometric;

	/**
	* Returns the entropy of a geometric distribution.
	*
	* ## Notes
	*
	* -   If `p < 0` or `p > 1`, the function returns `NaN`.
	*
	* @param p - success probability
	* @returns entropy
	*
	* @example
	* var v = ns.entropy( 0.1 );
	* // returns ~3.251
	*
	* @example
	* var v = ns.entropy( 0.5 );
	* // returns ~1.386
	*
	* @example
	* var v = ns.entropy( 0.9 );
	* // returns ~0.361
	*
	* @example
	* var v = ns.entropy( 1.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.entropy( NaN );
	* // returns NaN
	*/
	entropy: typeof entropy;

	/**
	* Returns the excess kurtosis of a geometric distribution.
	*
	* ## Notes
	*
	* -   If `p < 0` or `p > 1`, the function returns `NaN`.
	*
	* @param p - success probability
	* @returns kurtosis
	*
	* @example
	* var v = ns.kurtosis( 0.1 );
	* // returns ~6.011
	*
	* @example
	* var v = ns.kurtosis( 0.5 );
	* // returns 6.5
	*
	* @example
	* var v = ns.kurtosis( 1.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.kurtosis( NaN );
	* // returns NaN
	*/
	kurtosis: typeof kurtosis;

	/**
	* Geometric distribution logarithm of cumulative distribution function (CDF).
	*
	* @param x - input value
	* @param p - success probability
	* @returns evaluated logCDF
	*
	* @example
	* var y = ns.logcdf( 2.0, 0.5 );
	* // returns ~-0.134
	*
	* y = ns.logcdf( 2.0, 0.1 );
	* // returns ~-1.306
	*
	* var mylogcdf = ns.logcdf.factory( 0.5 );
	* y = mylogcdf( 3.0 );
	* // returns ~-0.065
	*
	* y = mylogcdf( 1.0 );
	* // returns ~-0.288
	*/
	logcdf: typeof logcdf;

	/**
	* Geometric distribution natural logarithm of probability mass function (PMF).
	*
	* @param x - input value
	* @param p - success probability
	* @returns evaluated logPMF
	*
	* @example
	* var y = ns.logpmf( 4.0, 0.3 );
	* // returns ~-2.631
	*
	* y = ns.logpmf( 2.0, 0.7 );
	* // returns ~-2.765
	*
	* var mylogpmf = ns.logpmf.factory( 0.5 );
	* y = mylogpmf( 3.0 );
	* // returns ~-2.773
	*
	* y = mylogpmf( 1.0 );
	* // returns ~-1.386
	*/
	logpmf: typeof logpmf;

	/**
	* Returns the expected value of a geometric distribution.
	*
	* ## Notes
	*
	* -   If `p < 0` or `p > 1`, the function returns `NaN`.
	*
	* @param p - success probability
	* @returns expected value
	*
	* @example
	* var v = ns.mean( 0.1 );
	* // returns 9.0
	*
	* @example
	* var v = ns.mean( 0.5 );
	* // returns 1.0
	*
	* @example
	* var v = ns.mean( 1.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.mean( NaN );
	* // returns NaN
	*/
	mean: typeof mean;

	/**
	* Returns the median of a geometric distribution.
	*
	* ## Notes
	*
	* -   If `p < 0` or `p > 1`, the function returns `NaN`.
	*
	* @param p - success probability
	* @returns median
	*
	* @example
	* var v = ns.median( 0.1 );
	* // returns 6
	*
	* @example
	* var v = ns.median( 0.5 );
	* // returns 0
	*
	* @example
	* var v = ns.median( 1.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.median( NaN );
	* // returns NaN
	*/
	median: typeof median;

	/**
	* Geometric distribution moment-generating function (MGF).
	*
	* @param t - input value
	* @param p - success probability
	* @returns evaluated MGF
	*
	* @example
	* var y = ns.mgf( 0.2, 0.5 );
	* // returns ~1.569
	*
	* y = ns.mgf( 0.4, 0.5 );
	* // returns ~2.936
	*
	* var mymgf = ns.mgf.factory( 0.8 );
	* y = mymgf( -0.2 );
	* // returns ~0.783
	*/
	mgf: typeof mgf;

	/**
	* Returns the mode of a geometric distribution.
	*
	* ## Notes
	*
	* -   If `p < 0` or `p > 1`, the function returns `NaN`.
	*
	* @param p - success probability
	* @returns mode
	*
	* @example
	* var v = ns.mode( 0.1 );
	* // returns 0
	*
	* @example
	* var v = ns.mode( 0.5 );
	* // returns 0
	*
	* @example
	* var v = ns.mode( 1.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.mode( NaN );
	* // returns NaN
	*/
	mode: typeof mode;

	/**
	* Geometric distribution probability mass function (PMF).
	*
	* @param x - input value
	* @param p - success probability
	* @returns evaluated PMF
	*
	* @example
	* var y = ns.pmf( 4.0, 0.3 );
	* // returns ~0.072
	*
	* y = ns.pmf( 2.0, 0.7 );
	* // returns ~0.063
	*
	* y = ns.pmf( -1.0, 0.5 );
	* // returns 0.0
	*
	* var mypmf = ns.pmf.factory( 0.5 );
	* y = mypmf( 3.0 );
	* // returns 0.0625
	*
	* y = mypmf( 1.0 );
	* // returns 0.25
	*/
	pmf: typeof pmf;

	/**
	* Geometric distribution quantile function.
	*
	* @param r - input value
	* @param p - success probability
	* @returns evaluated quantile function
	*
	* @example
	* var y = ns.quantile( 0.8, 0.4 );
	* // returns 3
	*
	* y = ns.quantile( 0.5, 0.4 );
	* // returns 1
	*
	* var myquantile = ns.quantile.factory( 0.4 );
	* y = myquantile( 0.4 );
	* // returns 0
	*
	* y = myquantile( 0.8 );
	* // returns 3
	*
	* y = myquantile( 1.0 );
	* // returns Infinity
	*/
	quantile: typeof quantile;

	/**
	* Returns the skewness of a geometric distribution.
	*
	* ## Notes
	*
	* -   If `p < 0` or `p > 1`, the function returns `NaN`.
	*
	* @param p - success probability
	* @returns skewness
	*
	* @example
	* var v = ns.skewness( 0.1 );
	* // returns ~2.003
	*
	* @example
	* var v = ns.skewness( 0.5 );
	* // returns ~2.121
	*
	* @example
	* var v = ns.skewness( 1.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.skewness( NaN );
	* // returns NaN
	*/
	skewness: typeof skewness;

	/**
	* Returns the standard deviation of a geometric distribution.
	*
	* ## Notes
	*
	* -   If `p < 0` or `p > 1`, the function returns `NaN`.
	*
	* @param p - success probability
	* @returns standard deviation
	*
	* @example
	* var v = ns.stdev( 0.1 );
	* // returns ~9.487
	*
	* @example
	* var v = ns.stdev( 0.5 );
	* // returns ~1.414
	*
	* @example
	* var v = ns.stdev( 1.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.stdev( NaN );
	* // returns NaN
	*/
	stdev: typeof stdev;

	/**
	* Returns the variance of a geometric distribution.
	*
	* ## Notes
	*
	* -   If `p < 0` or `p > 1`, the function returns `NaN`.
	*
	* @param p - success probability
	* @returns variance
	*
	* @example
	* var v = ns.variance( 0.1 );
	* // returns ~90.0
	*
	* @example
	* var v = ns.variance( 0.5 );
	* // returns 2.0
	*
	* @example
	* var v = ns.variance( 1.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.variance( NaN );
	* // returns NaN
	*/
	variance: typeof variance;
}

/**
* Geometric distribution.
*/
declare var ns: Namespace;


// EXPORTS //

export = ns;
