# -*- coding: utf-8 -*-
#
#  Copyright (C) 2003-2012 by Shyouzou Sugitani <shy@users.sourceforge.jp>
#  Copyright (C) 2003 by Shun-ichi TAHARA <jado@flowernet.gr.jp>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#

import os
import re

import gtk
import ninix.pix


class Menu(object):

    def __init__(self):
        self.request_parent = lambda *a: None # dummy
        ui_info = '''
        <ui>
          <popup name='popup'>
            <menu action='Recommend'>
            </menu>
            <menu action='Portal'>
            </menu>
            <separator/>
            <menu action='Plugin'>
            </menu>
            <menuitem action='Stick'/>
            <separator/>
            <menu action='Options'>
            <menuitem action='Update'/>
            <menuitem action='Vanish'/>
            <menuitem action='Preferences'/>
            <menuitem action='Console'/>
            <menuitem action='Manager'/>
            </menu>
            <separator/>
            <menu action='Change'>
            </menu>
            <menu action='Summon'>
            </menu>
            <menu action='Shell'>
            </menu>
            <menu action='Costume'>
            </menu>
            <menu action='Balloon'>
            </menu>
            <separator/>
            <menu action='Information'>
            <menuitem action='Usage'/>
            <menuitem action='Version'/>
            </menu>
            <separator/>
            <menu action='Nekodorif'>
            </menu>
            <menu action='Kinoko'>
            </menu>
            <separator/>
            <menuitem action='Close'/>
            <menuitem action='Quit'/>
          </popup>
        </ui>
        '''
        self.__menu_list = {
            'Portal': {
                'entry': ('Portal', None, _('Portal sites(_P)'), None),
                'visible': 1},
            'Recommend': {
                'entry': ('Recommend', None, _('Recommend sites(_R)'), None),
                'visible': 1},
            'Options': {
                'entry': ('Options', None, _('Options(_F)'), None),
                'visible': 1},
            'Options/Update': {
                'entry': ('Update', None, _('Network Update(_U)'), None,
                          '', lambda *a: self.request_parent('NOTIFY', 'network_update')),
                'visible': 1},
            'Options/Vanish': {
                'entry': ('Vanish', None, _('Vanish(_F)'), None,
                          '', lambda *a: self.request_parent('NOTIFY', 'vanish')),
                'visible': 1},
            'Options/Preferences': {
                'entry': ('Preferences', None, _('Preferences...(_O)'), None,
                           '', lambda *a: self.request_parent('NOTIFY', 'edit_preferences')),
                'visible': 1},
            'Options/Console': {
                'entry': ('Console', None, _('Console(_C)'), None,
                          '', lambda *a: self.request_parent('NOTIFY', 'open_console')),
                'visible': 1},
            'Options/Manager': {
                'entry': ('Manager', None, _('Ghost Manager(_M)'), None,
                          '', lambda *a: self.request_parent('NOTIFY', 'open_ghost_manager')),
                'visible': 1},
            'Information': {
                'entry': ('Information', None, _('Information(_I)'), None),
                'visible': 1},
            'Information/Usage': {
                'entry': ('Usage', None, _('Usage graph(_A)'), None,
                          '', lambda *a: self.request_parent('NOTIFY', 'show_usage')),
                'visible': 1},
            'Information/Version': {
                'entry': ('Version', None, _('Version(_V)'), None,
                          '', lambda *a: self.request_parent('NOTIFY', 'about')),
                'visible': 1},
            'Close': {
                'entry': ('Close', None, _('Close(_W)'), None,
                          '', lambda *a: self.request_parent('NOTIFY', 'close_sakura')),
                'visible': 1},
            'Quit': {
                'entry': ('Quit', None, _('Quit(_Q)'), None,
                          '', lambda *a: self.request_parent('NOTIFY', 'close_all')),
                'visible': 1},
            'Change': {
                'entry': ('Change', None, _('Change(_G)'), None),
                'visible': 1},
            'Summon': {
                'entry': ('Summon', None, _('Summon(_X)'), None),
                'visible': 1},
            'Shell': {
                'entry': ('Shell', None, _('Shell(_S)'), None),
                'visible': 1},
            'Balloon': {
                'entry': ('Balloon', None, _('Balloon(_B)'), None),
                'visible': 1},
            'Costume': {
                'entry': ('Costume', None, _('Costume(_C)'), None),
                'visible': 1},
            'Stick': {
                'entry': ('Stick', None, _('Stick(_Y)'), None,
                          '', lambda *a: self.request_parent('NOTIFY', 'stick_window'),
                          False),
                'visible': 1},
            'Nekodorif': {
                'entry': ('Nekodorif', None, _('Nekodorif(_N)'), None),
                'visible': 1},
            'Kinoko': {
                'entry': ('Kinoko', None, _('Kinoko(_K)'), None),
                'visible': 1},
            'Plugin': {
                'entry': ('Plugin', None, _('Plugin(_P)'), None),
                'visible': 1},
            }
        self.__pixmap = None
        self.__pixmap_with_sidebar = None
        self.__pixmap_prelight = None
        self.__pixmap_prelight_with_sidebar = None
        self.__fontcolor = {'normal': '#000000', 'prelight': '#ffffff'} ## FIXME
        self.__sidebar_width = 0
        actions = gtk.ActionGroup('Actions')
        entry = [value['entry'] for key, value in self.__menu_list.items() \
                     if key != 'Stick']
        actions.add_actions(tuple(entry))
        actions.add_toggle_actions(tuple([self.__menu_list['Stick']['entry']]))
        self.ui_manager = gtk.UIManager()
        self.ui_manager.insert_action_group(actions, 0)
        self.ui_manager.add_ui_from_string(ui_info)
        self.__popup_menu = self.ui_manager.get_widget('/popup')

    def set_responsible(self, request_method):
        self.request_parent = request_method

    def set_fontcolor(self, background, foreground):
        self.__fontcolor['normal'] = '#{0:02x}{1:02x}{2:02x}'.format(*background)
        self.__fontcolor['prelight'] = '#{0:02x}{1:02x}{2:02x}'.format(*foreground)

    def set_pixmap(self, path_background, path_sidebar, path_foreground):
        pixbuf = None
        if path_background and os.path.exists(path_background):
            try:
                pixbuf = ninix.pix.create_pixbuf_from_file(
                    path_background, is_pnr=False)
            except:
                pixbuf = None
        if pixbuf is None:
            self.__pixmap = None
            self.__pixmap_with_sidebar = None
            self.__pixmap_prelight = None
            self.__pixmap_prelight_with_sidebar = None
            self.__sidebar_width = 0
            return
        pixmap, mask = pixbuf.render_pixmap_and_mask(255)
        self.__pixmap = pixmap
        pixbuf_sidebar = None
        if path_sidebar and os.path.exists(path_sidebar):
            try:
                pixbuf_sidebar = ninix.pix.create_pixbuf_from_file(
                    path_sidebar, is_pnr=False)
            except:
                pixbuf_sidebar = None
        if pixbuf_sidebar:
            width = pixbuf.get_width()
            height = pixbuf.get_height()
            sidebar_width = pixbuf_sidebar.get_width()
            sidebar_height = pixbuf_sidebar.get_height()
            new_pixbuf = gtk.gdk.Pixbuf(
                pixbuf.get_colorspace(), False,
                pixbuf.get_bits_per_sample(),
                width + sidebar_width, max(height, sidebar_height))
            pixbuf.composite(
                new_pixbuf,
                0, 0, new_pixbuf.get_width(), new_pixbuf.get_height(),
                width - 1, height - 1,
                1.0, 1.0, gtk.gdk.INTERP_BILINEAR, 255)
            pixbuf_sidebar.composite(
                new_pixbuf,
                0, 0, sidebar_width, sidebar_height,
                0, 0,
                1.0, 1.0, gtk.gdk.INTERP_BILINEAR, 255)
            pixbuf.composite(
                new_pixbuf,
                sidebar_width, 0, width, height,
                0, 0,
                1.0, 1.0, gtk.gdk.INTERP_BILINEAR, 255)
            self.__sidebar_width = sidebar_width
            pixmap, mask = new_pixbuf.render_pixmap_and_mask(255)
        else:
            self.__sidebar_width = 0
        self.__pixmap_with_sidebar = pixmap
        pixbuf = None
        if path_foreground and os.path.exists(path_foreground):
            try:
                pixbuf = ninix.pix.create_pixbuf_from_file(
                    path_foreground, is_pnr=False)
            except:
                pixbuf = None
        if pixbuf is None:
            return
        pixmap, mask = pixbuf.render_pixmap_and_mask(255)
        self.__pixmap_prelight = pixmap
        if pixbuf_sidebar:
            width = pixbuf.get_width()
            height = pixbuf.get_height()
            sidebar_width = pixbuf_sidebar.get_width()
            sidebar_height = pixbuf_sidebar.get_height()
            new_pixbuf = gtk.gdk.Pixbuf(
                pixbuf.get_colorspace(), False,
                pixbuf.get_bits_per_sample(),
                width + sidebar_width, max(height, sidebar_height))
            pixbuf.composite(
                new_pixbuf,
                0, 0, new_pixbuf.get_width(), new_pixbuf.get_height(),
                width - 1, height - 1,
                1.0, 1.0, gtk.gdk.INTERP_BILINEAR, 255)
            pixbuf_sidebar.composite(
                new_pixbuf,
                0, 0, sidebar_width, sidebar_height,
                0, 0,
                1.0, 1.0, gtk.gdk.INTERP_BILINEAR, 255)
            pixbuf.composite(
                new_pixbuf,
                sidebar_width, 0, width, height,
                0, 0,
                1.0, 1.0, gtk.gdk.INTERP_BILINEAR, 255)
            pixmap, mask = new_pixbuf.render_pixmap_and_mask(255)
        self.__pixmap_prelight_with_sidebar = pixmap

    def __set_mayuna_menu(self, side):
        if len(self.__mayuna_menu) > side and \
           self.__mayuna_menu[side] is not None:
            menuitem = self.ui_manager.get_widget(
                ''.join(('/popup/', 'Costume')))
            menuitem.set_submenu(self.__mayuna_menu[side])
            self.__set_visible('Costume', 1)
        else:
            self.__set_visible('Costume', 0)

    def create_mayuna_menu(self, mayuna_menu):
        self.__mayuna_menu = []
        for side, index in [('sakura', 0), ('kero', 1)]:
            if mayuna_menu[side]:
                self.__mayuna_menu.append(gtk.Menu())
                item = gtk.TearoffMenuItem()
                item.show()
                self.__mayuna_menu[index].append(item)
                for j in range(len(mayuna_menu[side])):
                    key, name, state = mayuna_menu[side][j]
                    if key != '-':
                        item = gtk.CheckMenuItem(name)
                        item.set_name('popup menu item')
                        item.set_active(bool(state))
                        item.connect(
                            'activate',
                            lambda a, k: self.request_parent(
                                'NOTIFY', 'toggle_bind', k),
                            (index, key))
                    else:
                        item = gtk.SeparatorMenuItem()
                    item.show()
                    self.__mayuna_menu[index].append(item)
            else:
                self.__mayuna_menu.append(None)

    __re_shortcut = re.compile(r'&(?=[\x21-\x7e])')

    def __modify_shortcut(self, caption):
        return self.__re_shortcut.sub('_', caption)

    __re_mnemonic = re.compile(r'\(_.\)|_')

    def __cut_mnemonic(self, caption):
        return self.__re_mnemonic.sub('', caption)

    __ui = {'Options/Update': ([['updatebuttoncaption', 'updatebutton.caption'],
                             ['updatebuttoncaption', 'updatebutton.caption']],
                            '(_U)', [[],[]]),
            'Options/Vanish': ([['vanishbuttoncaption', 'vanishbutton.caption'],
                             ['vanishbuttoncaption', 'vanishbutton.caption']],
                            '(_F)',
                            [['vanishbuttonvisible', 'vanishbutton.visible'],
                             ['vanishbuttonvisible', 'vanishbutton.visible']]),
            'Portal': ([['sakura.portalbuttoncaption',
                         'portalrootbutton.caption'], []], '(_P)', [[], None]),
            'Recommend': ([['sakura.recommendbuttoncaption',
                            'recommendrootbutton.caption'],
                           ['kero.recommendbuttoncaption']], '(_R)', [[], []]),
           }

    def __update_ui(self, side):
        for key in self.__ui:
            assert key in self.__menu_list
            if side > 1:
                if key in ['Options/Update', 'Options/Vanish']:
                    name_list = self.__ui[key][0][1] # same as 'kero'
                elif key == 'Portal':
                    name_list = [] # same as 'kero'
                elif key == 'Recommend':
                    name_list = ['char{0:d}.recommendbuttoncaption'.format(side)]
            else:
                name_list = self.__ui[key][0][side]
            if name_list: # caption
                for name in name_list:
                    caption = self.request_parent('GET', 'getstring', name)
                    if caption:
                        break
                if caption:
                    caption = self.__modify_shortcut(caption)
                    if caption == self.__cut_mnemonic(caption):
                        caption = ''.join((caption, self.__ui[key][1]))
                    self.__set_caption(key, caption)
            if side > 1:
                name_list = self.__ui[key][2][1] # same as 'kero'
            else:
                name_list = self.__ui[key][2][side]
            if name_list: # visible
                for name in name_list:
                    visible = self.request_parent('GET', 'getstring', name)
                    if visible is not None:
                        break
                if visible == '0':
                    self.__set_visible(key, 0)
                else:
                    self.__set_visible(key, 1)
            elif name_list is None:
                self.__set_visible(key, 0)

    def set_submenu_back_pixmap(self, submenu):
        item_list = submenu.get_children()
        if item_list:
            for item in item_list:
                style = item.style.copy()
                style.bg_pixmap[gtk.STATE_PRELIGHT] = self.__pixmap_prelight
                item.set_style(style)
                submenu = item.get_submenu()
                if submenu:
                    style = submenu.style.copy()
                    style.bg_pixmap[gtk.STATE_NORMAL] = self.__pixmap
                    submenu.set_style(style)
                    self.set_submenu_back_pixmap(submenu)

    def set_submenu_fontcolor(self, submenu):
        for item in submenu.get_children():
            # reset label
            children = item.get_children()
            if children:
                label = children[0]
                style = label.style.copy()
                cmap = label.get_colormap()
                colour = cmap.alloc_color(self.__fontcolor['normal'])
                style.fg[gtk.STATE_NORMAL] =  colour
                colour = cmap.alloc_color(self.__fontcolor['prelight'])
                style.fg[gtk.STATE_PRELIGHT] =  colour
                label.set_style(style)
            submenu = item.get_submenu()
            if submenu:
                self.set_submenu_fontcolor(submenu)

    def popup(self, button, side):
        if side > 1:
            string = 'char{0:d}'.format(side)
        else:
            assert side in [0, 1] ## FIXME
            string = ['sakura', 'kero'][side]
        string = ''.join((string, '.popupmenu.visible'))
        if self.request_parent('GET', 'getstring', string) == '0':
            return
        self.__update_ui(side)
        if side == 0:
            portal = self.request_parent(
                'GET', 'getstring', 'sakura.portalsites')
        else:
            portal = None
        self.__set_portal_menu(side, portal)
        if side > 1:
            string = 'char{0:d}'.format(side)
        else:
            assert side in [0, 1] ## FIXME
            string = ['sakura', 'kero'][side]
        string = ''.join((string, '.recommendsites'))
        recommend = self.request_parent('GET', 'getstring', string)
        self.__set_recommend_menu(recommend)
        self.__set_ghost_menu()
        self.__set_shell_menu()
        self.__set_balloon_menu()
        self.__set_plugin_menu()
        self.__set_mayuna_menu(side)
        self.__set_nekodorif_menu()
        self.__set_kinoko_menu()
        style = self.__popup_menu.style.copy()
        style.bg_pixmap[gtk.STATE_NORMAL] = self.__pixmap_with_sidebar
        self.__popup_menu.set_style(style)
        for key in self.__menu_list:
            item = self.ui_manager.get_widget(''.join(('/popup/', key)))
            visible = self.__menu_list[key]['visible']
            if item:
                if visible:
                    if not key.startswith('Options/'): # XXX
                        style = item.style.copy()
                        style.bg_pixmap[gtk.STATE_PRELIGHT] = self.__pixmap_prelight_with_sidebar
                        item.set_style(style)
                    label = item.get_children()[0]
                    style = label.style.copy()
                    cmap = label.get_colormap()
                    colour = cmap.alloc_color(self.__fontcolor['normal'])
                    style.fg[gtk.STATE_NORMAL] =  colour
                    colour = cmap.alloc_color(self.__fontcolor['prelight'])
                    style.fg[gtk.STATE_PRELIGHT] =  colour
                    label.set_style(style)
                    item.show()
                    submenu = item.get_submenu()
                    if submenu:
                        style = submenu.style.copy()
                        style.bg_pixmap[gtk.STATE_NORMAL] = self.__pixmap
                        submenu.set_style(style)
                        self.set_submenu_back_pixmap(submenu)
                        self.set_submenu_fontcolor(submenu)
                else:
                    item.hide()
        self.__popup_menu.popup(None, None, None, button,
                                gtk.get_current_event_time())
        for item in self.__popup_menu.get_children():
            if gtk.REALIZED & item.flags():
                allocation = item.get_allocation()
                allocation.x += self.__sidebar_width
                allocation.width -= self.__sidebar_width
                item.size_allocate(allocation)
        self.__popup_menu.resize_children() ## FIXME: force resize popup_menu

    def __set_caption(self, name, caption):
        assert name in self.__menu_list
        assert isinstance(caption, basestring)
        item = self.ui_manager.get_widget(''.join(('/popup/', name)))
        if item:
            label = item.get_children()[0]
            label.set_text_with_mnemonic(caption)

    def __set_visible(self, name, visible):
        assert name in self.__menu_list
        assert visible in [0, 1]
        self.__menu_list[name]['visible'] = visible

    def __set_portal_menu(self, side, portal):
        if side >= 1:
            self.__set_visible('Portal', 0)
        else:
            if portal:
                menu = gtk.Menu()
                portal_list = portal.split(chr(2))
                for site in portal_list:
                    entry = site.split(chr(1))
                    if not entry:
                        continue
                    title = entry[0]
                    if title == '-':
                        item = gtk.SeparatorMenuItem()
                    else:
                        if len(entry) < 2:
                            continue
                        url = entry[1]
                        ## FIXME
                        #if len(entry) > 2:
                        #    bannar = entry[2]
                        #else:
                        #    bannar = None
                        item = gtk.MenuItem(title)
                        item.connect(
                            'activate',
                            lambda a, i: self.request_parent(
                                'NOTIFY', 'notify_site_selection', i),
                            (title, url))
                    menu.add(item)
                    item.show()
                menuitem = self.ui_manager.get_widget(
                    ''.join(('/popup/', 'Portal')))
                menuitem.set_submenu(menu)
                menu.show()
                self.__set_visible('Portal', 1)
            else:
                self.__set_visible('Portal', 0)

    def __set_recommend_menu(self, recommend):
        if recommend:
            menu = gtk.Menu()
            recommend_list = recommend.split(chr(2))
            for site in recommend_list:
                entry = site.split(chr(1))
                if not entry:
                    continue
                title = entry[0]
                if title == '-':
                    item = gtk.SeparatorMenuItem()
                else:
                    if len(entry) < 2:
                        continue
                    url = entry[1]
                    ## FIXME
                    #if len(entry) > 2:
                    #    bannar = entry[2]
                    #else:
                    #    bannar = None
                    item = gtk.MenuItem(title)
                    item.connect(
                        'activate',
                        lambda a, i: self.request_parent(
                            'NOTIFY', 'notify_site_selection', i),
                        (title, url))
                menu.add(item)
                item.show()
            menuitem =  self.ui_manager.get_widget(
                ''.join(('/popup/', 'Recommend')))
            menuitem.set_submenu(menu)
            menu.show()
            self.__set_visible('Recommend', 1)
        else:
            self.__set_visible('Recommend', 0)

    def create_ghost_menuitem(self, name, icon, key, handler):
        if icon is not None:
            pixbuf = ninix.pix.create_icon_pixbuf(icon)
            if pixbuf is None:
                item = gtk.MenuItem(name)
            else:
                image = gtk.Image()
                image.set_from_pixbuf(pixbuf)
                image.show()
                item = gtk.ImageMenuItem(name)
                item.set_image(image)
                item.set_always_show_image(True) # XXX
        else:
            item = gtk.MenuItem(name)
        item.set_name('popup menu item')
        item.show()
        item.connect('activate', lambda a, v: handler(v), (key))
        return item

    def __set_ghost_menu(self):
        for path in ['Summon', 'Change']:
            ghost_menu = gtk.Menu()
            for items in self.request_parent('GET', 'get_ghost_menus'):
                item = items[path]
                if item.get_parent():
                    item.reparent(ghost_menu)
                else:
                    ghost_menu.append(item)
            menuitem = self.ui_manager.get_widget(''.join(('/popup/', path)))
            menuitem.set_submenu(ghost_menu)

    def __set_shell_menu(self):
        shell_menu = self.request_parent('GET', 'get_shell_menu')
        menuitem = self.ui_manager.get_widget(''.join(('/popup/', 'Shell')))
        if menuitem.get_submenu():
            menuitem.remove_submenu()
        menuitem.set_submenu(shell_menu)

    def __set_balloon_menu(self):
        balloon_menu = self.request_parent('GET', 'get_balloon_menu')
        menuitem = self.ui_manager.get_widget(''.join(('/popup/', 'Balloon')))
        if menuitem.get_submenu():
            menuitem.remove_submenu()
        menuitem.set_submenu(balloon_menu)

    def create_meme_menu(self, menuitem):
        menu = gtk.Menu()
        for item in menuitem.values():
            if item.get_parent():
                item.reparent(menu)
            else:
                menu.append(item)
        return menu

    def create_meme_menuitem(self, name, value, handler):
        item = gtk.MenuItem(name)
        item.set_name('popup menu item')
        item.show()
        item.connect('activate', lambda a, v: handler(v), (value))
        return item

    def __set_plugin_menu(self):
        plugin_list = self.request_parent('GET', 'get_plugin_list')
        plugin_menu = gtk.Menu()
        for i in range(len(plugin_list)):
            name = plugin_list[i]['name']
            item = gtk.MenuItem(name)
            item.set_name('popup menu item')
            item.show()
            plugin_menu.append(item)
            item_list = plugin_list[i]['items']
            if len(item_list) <= 1:
                label, value = item_list[0]
                item.connect(
                    'activate',
                    lambda a, v: self.request_parent(
                        'NOTIFY', 'select_plugin', v),
                    (value))
                ##if working:
                ##    item.set_sensitive(False)
            else:
                submenu = gtk.Menu()
                submenu.set_name('popup menu')
                item.set_submenu(submenu)
                for label, value in item_list:
                    item = gtk.MenuItem(label)
                    item.set_name('popup menu item')
                    item.connect(
                        'activate',
                        lambda a, v: self.request_parent(
                            'NOTIFY', 'select_plugin', v),
                        (value))
                    item.show()
                    ##if working:
                    ##    item.set_sensitive(False)
                    submenu.append(item)
        menuitem = self.ui_manager.get_widget(''.join(('/popup/', 'Plugin')))
        menuitem.set_submenu(plugin_menu)

    def __set_nekodorif_menu(self):
        nekodorif_list = self.request_parent('GET', 'get_nekodorif_list')
        nekodorif_menu = gtk.Menu()
        for i in range(len(nekodorif_list)):
            name = nekodorif_list[i]['name']
            item = gtk.MenuItem(name)
            item.set_name('popup menu item')
            item.show()
            nekodorif_menu.append(item)
            item.connect(
                'activate',
                lambda a, n: self.request_parent(
                    'NOTIFY', 'select_nekodorif', n),
                (nekodorif_list[i]['dir']))
            ##if working:
            ##    item.set_sensitive(False)
        menuitem = self.ui_manager.get_widget(
            ''.join(('/popup/', 'Nekodorif')))
        menuitem.set_submenu(nekodorif_menu)

    def __set_kinoko_menu(self):
        kinoko_list = self.request_parent('GET', 'get_kinoko_list')
        kinoko_menu = gtk.Menu()
        for i in range(len(kinoko_list)):
            name = kinoko_list[i]['title']
            item = gtk.MenuItem(name)
            item.set_name('popup menu item')
            item.show()
            kinoko_menu.append(item)
            item.connect(
                'activate',
                lambda a, k: self.request_parent(
                    'NOTIFY', 'select_kinoko', k),
                (kinoko_list[i]))
            ##if working:
            ##    item.set_sensitive(False)
        menuitem = self.ui_manager.get_widget(''.join(('/popup/', 'Kinoko')))
        menuitem.set_submenu(kinoko_menu)

    def get_stick(self):
        item =  self.ui_manager.get_widget(''.join(('/popup/', 'Stick')))
        return 1 if item and item.get_active() else 0


def test():
    pass


if __name__ == '__main__':
    test()
