/*-------------------------------------------------------------------------
 * Copyright (c) 2000-2002 Kenneth W. Sodemann (stufflehead@bigfoot.com)
 *-------------------------------------------------------------------------
 * mainwin
 *
 * Synopsis:
 *   The main MathWars window
 *
 * $Id: mainwin.c,v 1.10 2002/11/18 16:42:55 stuffle Exp $
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to
 * Free Software Foundation, Inc.
 * 59 Temple Place, Suite 330 
 * Boston, MA  02111-1307  USA
 *
 *-------------------------------------------------------------------------
 */
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <ctype.h>
#include <gnome.h>

#include "about_dlg.h"
#include "draw.h"
#include "game_engine.h"
#include "mainwin.h"
#include "operators.h"
#include "prop_dlg.h"
#include "support.h"

#define MAINWIN        "main_window"
#define DRAWING_GC     "drawing_gc"
#define DRAWING_AREA   "drawing_area"
#define MAIN_APPBAR    "appbar1"
#define TIMER_DIAL     "timer_dial"
#define GAME_ENGINE    "game_eng"
#define ANSWER_ENTRY   "answer_entry"
#define SUBMIT_BUTTON  "submit_button"
#define PAUSE_BUTTON   "pause_button"
#define YES_BUTTON     "yes_button"
#define NO_BUTTON      "no_button"
#define NEW_GAME_MENU  "new_game_menu"

static GtkDrawingArea *main_drawing_area = NULL;

GtkDrawingArea *
get_main_drawing_area (void)
{
   /* 
    * This function should never be called before we have setup the
    * main window.
    */
   g_assert (main_drawing_area != NULL);
   return main_drawing_area;
}

static void
on_new_game_activate (GtkMenuItem  *menuitem,
                      gpointer      user_data)
{
   GtkWidget   *mainwin;
   game_engine *game;

   mainwin = lookup_widget (GTK_WIDGET (menuitem), MAINWIN);
   g_assert (mainwin);
   game = gtk_object_get_data (GTK_OBJECT (mainwin), GAME_ENGINE);
   g_assert (game);
   begin_game (game);
   begin_round (game);
}


static void
on_exit_activate (GtkMenuItem  *menuitem,
                  gpointer      user_data)
{
   GtkWidget  *mainwin;

   mainwin = lookup_widget (GTK_WIDGET (menuitem), MAINWIN);
   g_assert (mainwin);

   gtk_widget_destroy (GTK_WIDGET (mainwin));
}


static void
on_preferences1_activate (GtkMenuItem  *menuitem,
                          gpointer      user_data)
{
   GtkWidget  *mainwin;

   mainwin = lookup_widget (GTK_WIDGET (menuitem), MAINWIN);
   g_assert (mainwin);

   show_prop_dlg(GTK_WIDGET (mainwin));
}


static void
on_about_activate (GtkMenuItem  *menuitem,
                   gpointer      user_data)
{
   GtkWidget *parent = get_widget (GTK_WIDGET (menuitem), MAINWIN);

   g_assert (parent != NULL);
   show_about_dlg (parent);
}


static void
on_mainwin_show (GtkWidget     *widget,
                 gpointer       user_data)
{
   game_engine *game;

   game = create_game_engine (widget);
   gtk_object_set_data (GTK_OBJECT (widget), GAME_ENGINE, game);

   /*
    * init_draw_objects() will initialize certain objects that
    * determine the size of the drawing area.  Thus, set the
    * size here instead of in the create function.
    */
   init_draw_objects();
   gtk_widget_set_usize (GTK_WIDGET (main_drawing_area),
                         board_width(), board_height());
   draw_cards_down();

   refresh_board(get_player1(game), get_player2(game));
}


static void
on_mainwin_destroy (GtkWidget  *widget,
                    gpointer    user_data)
{
   game_engine *game;

   game = gtk_object_get_data (GTK_OBJECT (widget), GAME_ENGINE);
   g_assert (game);
   destroy_game_engine (game);

   gtk_main_quit();
}


/* 
 * Example code: http://www.gtk.org/faq/#AEN812
 */
static void
limit_input_to_nums (GtkEntry    *entry,
                     const gchar *text,
                     gint         length,
                     gint        *position,
                     gpointer     data)
{
   GtkEditable *edit = GTK_EDITABLE (entry);
   gint         idx;
   gint         cnt = 0;
   gchar       *res = g_new (gchar, length);

   for (idx = 0; idx < length; idx++)
      {
      if (text[idx] == '-' || (isdigit (text[idx])))
         {
         res[cnt++] = text[idx];
         }
      }

   if (cnt > 0)
      {
      gtk_signal_handler_block_by_func (GTK_OBJECT (edit),
                                        GTK_SIGNAL_FUNC (limit_input_to_nums),
                                        data);
      gtk_editable_insert_text (edit, res, cnt, position);
      gtk_signal_handler_unblock_by_func (GTK_OBJECT (edit),
                                          GTK_SIGNAL_FUNC (limit_input_to_nums),
                                          data);
     }
   gtk_signal_emit_stop_by_name (GTK_OBJECT (edit), "insert_text");

   g_free (res);
}


static gboolean
on_answer_entry_key_press_event (GtkWidget       *widget,
                                 GdkEventKey     *event,
                                 gpointer         user_data)
{
   GtkWidget  *submit_btn;

   if (event->keyval == GDK_Return || event->keyval == GDK_KP_Enter)
      {
      submit_btn = get_widget (widget, SUBMIT_BUTTON);
      g_assert (submit_btn);

      gtk_signal_emit_by_name (GTK_OBJECT (submit_btn), "clicked");
      }

   return FALSE;
}


static void
on_yes_clicked (GtkWidget  *widget,
                gpointer    user_data)
{
   GtkWidget   *other_button;
   GtkWidget   *mainwin;
   game_engine *game;

   /*
    * Right now, the only time this button is active is if the 
    * computer player has made a guess, and the user is answering
    * the bonus question.  The game engine will handle most of the
    * actual logic.  Just let the game engine know that the user
    * pressed the button, and then disable both the yes button and
    * the no button.
    */
   mainwin = lookup_widget (widget, MAINWIN);
   g_assert (mainwin != NULL);
   other_button = lookup_widget (mainwin, NO_BUTTON);
   g_assert (other_button != NULL);
   game = gtk_object_get_data (GTK_OBJECT (mainwin), GAME_ENGINE);
   g_assert (game != NULL);
   
   gtk_widget_set_sensitive (other_button, FALSE);
   gtk_widget_set_sensitive (widget, FALSE);
   handle_yes_answer (game);
}


static void
on_no_clicked (GtkWidget  *widget,
               gpointer    user_data)
{
   GtkWidget   *other_button;
   GtkWidget   *mainwin;
   game_engine *game;

   /*
    * Right now, the only time this button is active is if the 
    * computer player has made a guess, and the user is answering
    * the bonus question.  The game engine will handle most of the
    * actual logic.  Just let the game engine know that the user
    * pressed the button, and then disable both the yes button and
    * the no button.
    */
   mainwin = lookup_widget (widget, MAINWIN);
   g_assert (mainwin != NULL);
   other_button = lookup_widget (mainwin, YES_BUTTON);
   g_assert (other_button != NULL);
   game = gtk_object_get_data (GTK_OBJECT (mainwin), GAME_ENGINE);
   g_assert (game != NULL);

   gtk_widget_set_sensitive (other_button, FALSE);
   gtk_widget_set_sensitive (widget, FALSE);
   handle_no_answer (game);
}


static void
on_submit_clicked (GtkWidget  *widget,
                   gpointer    user_data)
{
   game_engine *game;
   GtkWidget   *answer_entry;
   GtkWidget   *mainwin;
   GtkWidget   *pause_btn;

   mainwin = lookup_widget (widget, MAINWIN);
   g_assert (mainwin);
   answer_entry = lookup_widget (mainwin, ANSWER_ENTRY);
   g_assert (answer_entry);
   pause_btn = lookup_widget (mainwin, PAUSE_BUTTON);
   g_assert (pause_btn);
   game = gtk_object_get_data (GTK_OBJECT (mainwin), GAME_ENGINE);
   g_assert (game);

   gtk_widget_set_sensitive (widget, FALSE);

   handle_player_answer (game, gtk_entry_get_text (GTK_ENTRY (answer_entry)));
}


static void
on_pause_toggled (GtkWidget *pause_btn,
                  gpointer   user_data)
{
   game_engine  *game;
   GtkWidget    *mainwin;
   GtkWidget    *widget;

   /*
    * static vars used to store the current state of the buttons.
    * The current state should be saved before pausing, and 
    * restored when the game is no longer paused.
    *
    * There should not be any event that occurs during a pause
    * that will validly change the states.
    */
   static gboolean submit_active = TRUE;
   static gboolean yes_active    = FALSE;
   static gboolean no_active     = FALSE;

   mainwin = lookup_widget (pause_btn, MAINWIN);
   g_assert (mainwin);
   game = gtk_object_get_data (GTK_OBJECT (mainwin), GAME_ENGINE);
   g_assert (game);

   if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (pause_btn)))
      {
      /*
       * Save the state of the submit, yes, and no buttons, then
       * disable them.
       */
      widget = lookup_widget (pause_btn, SUBMIT_BUTTON);
      g_assert (widget != NULL);
      submit_active = GTK_WIDGET_SENSITIVE (widget);
      gtk_widget_set_sensitive (widget, FALSE);
      widget = lookup_widget (pause_btn, YES_BUTTON);
      g_assert (widget != NULL);
      yes_active = GTK_WIDGET_SENSITIVE (widget);
      gtk_widget_set_sensitive (widget, FALSE);
      widget = lookup_widget (pause_btn, NO_BUTTON);
      g_assert (widget != NULL);
      no_active = GTK_WIDGET_SENSITIVE (widget);
      gtk_widget_set_sensitive (widget, FALSE);

      /*
       * Disable the "New Game" menu item.
       */
      widget = lookup_widget (pause_btn, NEW_GAME_MENU);
      g_assert (widget != NULL);
      gtk_widget_set_sensitive (widget, FALSE);

      pause_game (game);
      }
   else
      {
      /*
       * Restore the submit, yes, and no buttons before resuming the
       * game.
       */
      widget = lookup_widget (pause_btn, SUBMIT_BUTTON);
      g_assert (widget != NULL);
      gtk_widget_set_sensitive (widget, submit_active);
      widget = lookup_widget (pause_btn, YES_BUTTON);
      g_assert (widget != NULL);
      gtk_widget_set_sensitive (widget, yes_active);
      widget = lookup_widget (pause_btn, NO_BUTTON);
      g_assert (widget != NULL);
      gtk_widget_set_sensitive (widget, no_active);

      /*
       * Enable the "New Game" menu item.
       */
      widget = lookup_widget (pause_btn, NEW_GAME_MENU);
      g_assert (widget != NULL);
      gtk_widget_set_sensitive (widget, TRUE);

      restore_game (game);
      }
}


static GnomeUIInfo game_menu_uiinfo[] =
{
  GNOMEUIINFO_MENU_NEW_ITEM (N_("_New Game"), 
                             N_("Start a new game."), 
                             on_new_game_activate, NULL),
  GNOMEUIINFO_SEPARATOR,
  GNOMEUIINFO_MENU_EXIT_ITEM (on_exit_activate, NULL),
  GNOMEUIINFO_END
};

static GnomeUIInfo settings1_menu_uiinfo[] =
{
  GNOMEUIINFO_MENU_PREFERENCES_ITEM (on_preferences1_activate, NULL),
  GNOMEUIINFO_END
};

static GnomeUIInfo help1_menu_uiinfo[] =
{
  GNOMEUIINFO_HELP("MathWar"),
  GNOMEUIINFO_SEPARATOR,    
  GNOMEUIINFO_MENU_ABOUT_ITEM (on_about_activate, NULL),
  GNOMEUIINFO_END
};

static GnomeUIInfo menubar1_uiinfo[] =
{
  GNOMEUIINFO_MENU_FILE_TREE (game_menu_uiinfo),
  GNOMEUIINFO_MENU_SETTINGS_TREE (settings1_menu_uiinfo),
  GNOMEUIINFO_MENU_HELP_TREE (help1_menu_uiinfo),
  GNOMEUIINFO_END
};


GtkWidget *
create_mainwin (void)
{
  GtkWidget   *app1;
  GtkWidget   *dock1;
  GtkWidget   *vbox1;
  GtkWidget   *drawingarea;
  GtkWidget   *hbox1;
  GtkWidget   *hbox2;
  GtkWidget   *entry1;
  GtkWidget   *submit_btn;
  GtkWidget   *yes_btn;
  GtkWidget   *no_btn;
  GtkWidget   *pause_btn;
  GtkWidget   *appbar1;
  GtkWidget   *hsep;
  GtkTooltips *tooltips;

  tooltips = gtk_tooltips_new();

  app1 = gnome_app_new (PACKAGE, _(PACKAGE));
  gtk_object_set_data (GTK_OBJECT (app1), MAINWIN, app1);
  gtk_signal_connect (GTK_OBJECT (app1), "show",
                      GTK_SIGNAL_FUNC (on_mainwin_show),
                      NULL);
  gtk_signal_connect (GTK_OBJECT (app1), "destroy",
                      GTK_SIGNAL_FUNC (on_mainwin_destroy),
                      NULL);
  gtk_window_set_policy (GTK_WINDOW (app1), FALSE, FALSE, TRUE);

  dock1 = GNOME_APP (app1)->dock;
  gtk_widget_ref (dock1);
  gtk_object_set_data_full (GTK_OBJECT (app1), "dock1", dock1,
                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (dock1);

  gnome_app_create_menus (GNOME_APP (app1), menubar1_uiinfo);

  gtk_widget_ref (menubar1_uiinfo[0].widget);
  gtk_object_set_data_full (GTK_OBJECT (app1), "game",
                            menubar1_uiinfo[0].widget,
                            (GtkDestroyNotify) gtk_widget_unref);

  gtk_widget_ref (game_menu_uiinfo[0].widget);
  gtk_object_set_data_full (GTK_OBJECT (app1), NEW_GAME_MENU,
                            game_menu_uiinfo[0].widget,
                            (GtkDestroyNotify) gtk_widget_unref);

  gtk_widget_ref (game_menu_uiinfo[1].widget);
  gtk_object_set_data_full (GTK_OBJECT (app1), "separator1",
                            game_menu_uiinfo[1].widget,
                            (GtkDestroyNotify) gtk_widget_unref);

  gtk_widget_ref (game_menu_uiinfo[2].widget);
  gtk_object_set_data_full (GTK_OBJECT (app1), "exit1",
                            game_menu_uiinfo[2].widget,
                            (GtkDestroyNotify) gtk_widget_unref);

  gtk_widget_ref (menubar1_uiinfo[1].widget);
  gtk_object_set_data_full (GTK_OBJECT (app1), "settings1",
                            menubar1_uiinfo[1].widget,
                            (GtkDestroyNotify) gtk_widget_unref);

  gtk_widget_ref (settings1_menu_uiinfo[0].widget);
  gtk_object_set_data_full (GTK_OBJECT (app1), "preferences1",
                            settings1_menu_uiinfo[0].widget,
                            (GtkDestroyNotify) gtk_widget_unref);

  gtk_widget_ref (menubar1_uiinfo[2].widget);
  gtk_object_set_data_full (GTK_OBJECT (app1), "help1",
                            menubar1_uiinfo[2].widget,
                            (GtkDestroyNotify) gtk_widget_unref);

  gtk_widget_ref (help1_menu_uiinfo[2].widget);
  gtk_object_set_data_full (GTK_OBJECT (app1), "about1",
                            help1_menu_uiinfo[2].widget,
                            (GtkDestroyNotify) gtk_widget_unref);

  vbox1 = gtk_vbox_new (FALSE, 5);
  gtk_widget_ref (vbox1);
  gtk_object_set_data_full (GTK_OBJECT (app1), "vbox1", vbox1,
                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (vbox1);
  gnome_app_set_contents (GNOME_APP (app1), vbox1);
  gtk_container_set_border_width (GTK_CONTAINER (vbox1), 0);

  drawingarea = gtk_drawing_area_new ();
  gtk_widget_ref (drawingarea);
  gtk_object_set_data_full (GTK_OBJECT (app1), DRAWING_AREA, drawingarea,
                            (GtkDestroyNotify)gtk_widget_unref);
  gtk_widget_show (drawingarea);
  gtk_box_pack_start (GTK_BOX (vbox1), drawingarea, TRUE, TRUE, 0);
  main_drawing_area = GTK_DRAWING_AREA (drawingarea);
  gtk_widget_set_usize (GTK_WIDGET (main_drawing_area),
                        board_width(), board_height());
  
  hsep = gtk_hseparator_new ();
  gtk_widget_ref (hsep);
  gtk_object_set_data_full (GTK_OBJECT (hsep), "hsep2", hsep,
                            (GtkDestroyNotify)gtk_widget_unref);
  gtk_widget_show (hsep);
  gtk_box_pack_start (GTK_BOX (vbox1), hsep, FALSE, FALSE, 0);

  hbox1 = gtk_hbox_new (FALSE, 10);
  gtk_widget_ref (hbox1);
  gtk_object_set_data_full (GTK_OBJECT (app1), "hbox1", hbox1,
                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (hbox1);
  gtk_box_pack_start (GTK_BOX (vbox1), hbox1, FALSE, FALSE, 0);
  gtk_container_set_border_width (GTK_CONTAINER (hbox1), 5);

  entry1 = gtk_entry_new ();
  gtk_widget_ref (entry1);
  gtk_object_set_data_full (GTK_OBJECT (app1), ANSWER_ENTRY, entry1,
                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (entry1);
  gtk_box_pack_start (GTK_BOX (hbox1), entry1, TRUE, TRUE, 0);
  gtk_signal_connect (GTK_OBJECT (entry1), "insert_text",
                      GTK_SIGNAL_FUNC (limit_input_to_nums),
                      NULL);
  gtk_signal_connect (GTK_OBJECT (entry1), "key_press_event",
                      GTK_SIGNAL_FUNC (on_answer_entry_key_press_event),
                      NULL);

  hbox2 = gtk_hbox_new (TRUE, 0);
  gtk_widget_ref (hbox2);
  gtk_object_set_data_full (GTK_OBJECT (app1), "hbox2", hbox2,
                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (hbox2);
  gtk_box_pack_start (GTK_BOX (hbox1), hbox2, FALSE, TRUE, 0);

  submit_btn = gtk_button_new_with_label (_("Submit"));
  gtk_widget_ref (submit_btn);
  gtk_object_set_data_full (GTK_OBJECT (app1), SUBMIT_BUTTON, submit_btn,
                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (submit_btn);
  gtk_box_pack_start (GTK_BOX (hbox2), submit_btn, FALSE, TRUE, 0);
  gtk_widget_set_sensitive (submit_btn, FALSE);
  GTK_WIDGET_SET_FLAGS (submit_btn, GTK_CAN_DEFAULT);
  gtk_signal_connect (GTK_OBJECT (submit_btn), "clicked",
                      GTK_SIGNAL_FUNC (on_submit_clicked),
                      NULL);

  pause_btn = gtk_toggle_button_new_with_label (_("Pause"));
  gtk_widget_ref (pause_btn);
  gtk_object_set_data_full (GTK_OBJECT (app1), PAUSE_BUTTON, pause_btn,
                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (pause_btn);
  gtk_box_pack_start (GTK_BOX (hbox2), pause_btn, FALSE, TRUE, 0);
  GTK_WIDGET_SET_FLAGS (pause_btn, GTK_CAN_DEFAULT);
  gtk_widget_set_sensitive (pause_btn, FALSE);
  gtk_signal_connect (GTK_OBJECT (pause_btn), "toggled",
                      GTK_SIGNAL_FUNC (on_pause_toggled),
                      NULL);

  yes_btn = gnome_stock_button (GNOME_STOCK_BUTTON_YES);
  gtk_widget_ref (yes_btn);
  gtk_object_set_data_full (GTK_OBJECT (app1), YES_BUTTON, yes_btn,
                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (yes_btn);
  gtk_box_pack_start (GTK_BOX (hbox2), yes_btn, FALSE, TRUE, 0);
  GTK_WIDGET_SET_FLAGS (yes_btn, GTK_CAN_DEFAULT);
  gtk_widget_set_sensitive (yes_btn, FALSE);
  gtk_signal_connect (GTK_OBJECT (yes_btn), "clicked",
                      GTK_SIGNAL_FUNC (on_yes_clicked),
                      NULL);

  no_btn = gnome_stock_button (GNOME_STOCK_BUTTON_NO);
  gtk_widget_ref (no_btn);
  gtk_object_set_data_full (GTK_OBJECT (app1), NO_BUTTON, no_btn,
                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (no_btn);
  gtk_box_pack_start (GTK_BOX (hbox2), no_btn, FALSE, TRUE, 0);
  GTK_WIDGET_SET_FLAGS (no_btn, GTK_CAN_DEFAULT);
  gtk_widget_set_sensitive (no_btn, FALSE);
  gtk_signal_connect (GTK_OBJECT (no_btn), "clicked",
                      GTK_SIGNAL_FUNC (on_no_clicked),
                      NULL);

  appbar1 = gnome_appbar_new (TRUE, TRUE, GNOME_PREFERENCES_NEVER);
  gtk_widget_ref (appbar1);
  gtk_object_set_data_full (GTK_OBJECT (app1), MAIN_APPBAR, appbar1,
                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (appbar1);
  gnome_app_set_statusbar (GNOME_APP (app1), appbar1);

  gnome_app_install_menu_hints (GNOME_APP (app1), menubar1_uiinfo);

  gtk_widget_grab_default (submit_btn);

  gtk_object_set_data (GTK_OBJECT (app1), "tooltips", tooltips);

  return app1;
}


void 
set_timer_pct (GtkWidget *mainwin,
               gfloat     pct)
{
   GnomeAppBar *bar;

   bar = GNOME_APPBAR (lookup_widget (mainwin, MAIN_APPBAR));
   g_assert (bar != NULL);

   gnome_appbar_set_progress (bar, pct);

   return;
}


void
disable_all_input_widgets (GtkWidget *mainwin)
{
   GtkWidget  *widget;

   widget = lookup_widget (mainwin, SUBMIT_BUTTON);
   g_assert (widget != NULL);
   gtk_widget_set_sensitive (widget, FALSE);
   gtk_widget_grab_default (widget);

   widget = lookup_widget (mainwin, PAUSE_BUTTON);
   g_assert (widget != NULL);
   gtk_widget_set_sensitive (widget, FALSE);
   gtk_widget_grab_focus (widget);

   widget = lookup_widget (mainwin, YES_BUTTON);
   g_assert (widget != NULL);
   gtk_widget_set_sensitive (widget, FALSE);

   widget = lookup_widget (mainwin, NO_BUTTON);
   g_assert (widget != NULL);
   gtk_widget_set_sensitive (widget, FALSE);

   widget = lookup_widget (mainwin, ANSWER_ENTRY);
   g_assert (widget != NULL);
   gtk_entry_set_text (GTK_ENTRY (widget), "");
   gtk_widget_grab_focus (widget);
}


void
reset_input_widgets (GtkWidget *mainwin)
{
   GtkWidget  *widget;

   widget = lookup_widget (mainwin, SUBMIT_BUTTON);
   g_assert (widget != NULL);
   gtk_widget_set_sensitive (widget, TRUE);
   gtk_widget_grab_default (widget);

   widget = lookup_widget (mainwin, PAUSE_BUTTON);
   g_assert (widget != NULL);
   gtk_widget_set_sensitive (widget, TRUE);
   gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (widget), FALSE);
   gtk_widget_grab_focus (widget);

   widget = lookup_widget (mainwin, YES_BUTTON);
   g_assert (widget != NULL);
   gtk_widget_set_sensitive (widget, FALSE);

   widget = lookup_widget (mainwin, NO_BUTTON);
   g_assert (widget != NULL);
   gtk_widget_set_sensitive (widget, FALSE);

   widget = lookup_widget (mainwin, ANSWER_ENTRY);
   g_assert (widget != NULL);
   gtk_entry_set_text (GTK_ENTRY (widget), "");
   gtk_widget_grab_focus (widget);
}


void
display_rounds_left (GtkWidget *mainwin,
                     gint       rounds)
{
   GnomeAppBar *bar;
   GString     *str;

   bar = GNOME_APPBAR (lookup_widget (mainwin, MAIN_APPBAR));
   g_return_if_fail (bar != NULL);

   str = g_string_new ("");
   g_string_sprintf (str, _("Rounds Left: %d"), rounds);
   gnome_appbar_set_default (bar, str->str);

   g_string_free (str, TRUE);
}


void
yesno_activate (GtkWidget *mainwin)
{
   GtkWidget *widget;

   widget = lookup_widget (mainwin, SUBMIT_BUTTON);
   g_assert (widget != NULL);
   gtk_widget_set_sensitive (widget, FALSE);

   widget = lookup_widget (mainwin, YES_BUTTON);
   g_assert (widget != NULL);
   gtk_widget_set_sensitive (widget, TRUE);

   widget = lookup_widget (mainwin, NO_BUTTON);
   g_assert (widget != NULL);
   gtk_widget_set_sensitive (widget, TRUE);
}
