use select::{
    document::Document,
    node::Node,
    predicate::{Attr, Name},
};

use crate::data::{KbParsedEntry, PageProcess, QueryResponse};

pub fn get_pages() -> Vec<PageProcess<'static>> {
    vec![
        PageProcess {
            url: "https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/AuroraMySQL.Reference.ParameterGroups.html#AuroraMySQL.Reference.GlobalStatusVars".to_string(),
            name: "aws-rds-aurora-mysql".to_string(),
            data_type: "variables",
        }
    ]
}

fn process_table_row(tr: Node) -> KbParsedEntry {
    let mut tr_nodes = tr.find(Name("td"));
    let row_value: Node = tr_nodes.next().expect("Node to exist");
    let row_description: Node = tr_nodes.next().expect("Node to exist");
    let row_description_text: String = row_description.text().trim().to_owned();

    let variable_id: String = row_value.text().trim().to_owned();

    KbParsedEntry {
        has_description: row_description_text.len() > 0,
        is_removed: false,
        cli: None,
        default: None,
        dynamic: None,
        id: None,
        name: Some(row_value.text().trim().to_string()),
        scope: None,
        r#type: match row_description_text {
            _ if variable_id.ends_with("_request_cnt") => Some("integer".to_string()),
            _ if variable_id.ends_with("_response_cnt") => Some("integer".to_string()),
            _ if variable_id.ends_with("_hit_cnt") => Some("integer".to_string()),
            _ if variable_id == "Aurora_tmz_version" => Some("string".to_string()),
            _ if variable_id == "Aurora_enhanced_binlog" => Some("boolean".to_string()),
            _ if variable_id == "server_aurora_das_running" => Some("boolean".to_string()),
            d if d.contains("latency since") => Some("integer".to_string()),
            d if d.contains("A counter that's incremented") => Some("integer".to_string()),
            d if d.contains("The amount of memory in bytes") => Some("byte".to_string()),
            d if d.contains("The number of bytes") => Some("byte".to_string()),
            d if d.contains("Number of bytes replicated") => Some("byte".to_string()),
            d if d.contains("The maximum number of parallel query sessions") => {
                Some("integer".to_string())
            }
            d if d.contains("The number of times") => Some("integer".to_string()),
            d if d.contains("The number of parallel query requests") => Some("integer".to_string()),
            d if d.contains("The number of parallel query sessions") => Some("integer".to_string()),
            d if d.contains("The number of data pages") => Some("integer".to_string()),
            d if d.contains("The number of read requests") => Some("integer".to_string()),
            d if d.contains("The number of database connections") => Some("integer".to_string()),
            d if d.contains("The total number of DML statements") => Some("integer".to_string()),
            d if d.contains("The total duration of DML statements") => Some("integer".to_string()),
            d if d.contains("The total number of commits") => Some("integer".to_string()),
            d if d.contains("The number of forwarded queries") => Some("integer".to_string()),
            d if d.contains("The number of forwarded sessions") => Some("integer".to_string()),
            d if d.contains("The total number of SELECT statements") => Some("integer".to_string()),
            d if d.contains("The total duration of SELECT statements") => {
                Some("integer".to_string())
            }
            d if d.contains("The current number of threads") => Some("integer".to_string()),
            d if d.contains("enabled or disabled on this DB instance") => {
                Some("boolean".to_string())
            }
            _ => None,
        },
        valid_values: None,
        range: None,
    }
}

fn process_table_container(table_container: Node) -> Vec<KbParsedEntry> {
    table_container
        .find(Name("tr"))
        .into_iter()
        .skip(1) // Skip header row
        .map(|table_row| process_table_row(table_row))
        .collect()
}

pub fn extract_aurora_mysql_from_text(qr: QueryResponse) -> Vec<KbParsedEntry> {
    let document = Document::from(qr.body.as_str());

    let node: Node = document
        .find(Attr("id", "AuroraMySQL.Reference.GlobalStatusVars"))
        .next()
        .unwrap();

    let mut found_table_container: Option<Node> = None;
    let mut node_count = 10;
    let mut node_cur: Option<Node> = Some(node);

    loop {
        // Current node is None exit
        if node_cur.is_none() {
            break;
        }
        // Move cursor to next and bump count
        node_cur = node_cur.unwrap().next();
        node_count = node_count - 1;
        // If still is None or count too low exit
        if node_cur.is_none() || node_count < 1 {
            break;
        }

        let n = node_cur.unwrap();
        if n.is(Name("div")) && n.attr("class") == Some("table-container") {
            found_table_container = Some(n);
        }
    }

    match found_table_container {
        Some(node) => process_table_container(node),
        None => vec![],
    }
}

#[cfg(test)]
mod tests {

    // Note this useful idiom: importing names from outer (for mod tests) scope.
    use super::*;
    use pretty_assertions::assert_eq;
    use std::env;
    use std::fs;

    fn get_test_data(file_name: &str) -> String {
        let test_dir = env::current_dir().unwrap();
        fs::read_to_string(test_dir.to_str().unwrap().to_owned() + "/src/rust/data/" + file_name)
            .expect("Should have been able to read the test data file")
    }

    #[test]
    fn test_case_1() {
        let entries = extract_aurora_mysql_from_text(QueryResponse {
            body: get_test_data("mysql_aurora_test_case_1.html"),
            url: "https://example.com".to_string(),
        });
        assert_eq!(
            vec![
                KbParsedEntry {
                    has_description: true,
                    is_removed: false,
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("AuroraDb_commits".to_string()),
                    id: None,
                    scope: None,
                    r#type: Some("integer".to_string()),
                    valid_values: None,
                    range: None,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("AuroraDb_commit_latency".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("integer".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("AuroraDb_ddl_stmt_duration".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("integer".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("AuroraDb_select_stmt_duration".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("integer".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("AuroraDb_insert_stmt_duration".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("integer".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("AuroraDb_update_stmt_duration".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("integer".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("AuroraDb_delete_stmt_duration".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("integer".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("Aurora_binlog_io_cache_allocated".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("byte".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("Aurora_binlog_io_cache_read_requests".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("integer".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("Aurora_binlog_io_cache_reads".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("integer".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("Aurora_enhanced_binlog".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("boolean".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("Aurora_external_connection_count".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("integer".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("Aurora_fast_insert_cache_hits".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("integer".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("Aurora_fast_insert_cache_misses".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("integer".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("Aurora_fwd_master_dml_stmt_count".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("integer".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("Aurora_fwd_master_dml_stmt_duration".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("integer".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("Aurora_fwd_master_errors_rpc_timeout".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("integer".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("Aurora_fwd_master_errors_session_limit".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("integer".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("Aurora_fwd_master_errors_session_timeout".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("integer".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("Aurora_fwd_master_open_sessions".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("integer".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("Aurora_fwd_master_select_stmt_count".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("integer".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("Aurora_fwd_master_select_stmt_duration".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("integer".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("Aurora_fwd_writer_dml_stmt_count".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("integer".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("Aurora_fwd_writer_dml_stmt_duration".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("integer".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("Aurora_fwd_writer_errors_rpc_timeout".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("integer".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("Aurora_fwd_writer_errors_session_limit".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("integer".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("Aurora_fwd_writer_errors_session_timeout".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("integer".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("Aurora_fwd_writer_open_sessions".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("integer".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("Aurora_fwd_writer_select_stmt_count".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("integer".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("Aurora_fwd_writer_select_stmt_duration".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("integer".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("Aurora_lockmgr_memory_used".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("byte".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("Aurora_ml_actual_request_cnt".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("integer".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("Aurora_ml_actual_response_cnt".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("integer".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("Aurora_ml_cache_hit_cnt".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("integer".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("Aurora_ml_logical_request_cnt".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("integer".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("Aurora_ml_logical_response_cnt".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("integer".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("Aurora_ml_retry_request_cnt".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("integer".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("Aurora_ml_single_request_cnt".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("integer".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("Aurora_pq_bytes_returned".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("byte".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("Aurora_pq_max_concurrent_requests".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("integer".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("Aurora_pq_pages_pushed_down".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("integer".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("Aurora_pq_request_attempted".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("integer".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("Aurora_pq_request_executed".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("integer".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("Aurora_pq_request_failed".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("integer".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("Aurora_pq_request_in_progress".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("integer".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("Aurora_pq_request_not_chosen".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("integer".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("Aurora_pq_request_not_chosen_below_min_rows".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("integer".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("Aurora_pq_request_not_chosen_column_bit".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("integer".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("Aurora_pq_request_not_chosen_column_geometry".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("integer".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("Aurora_pq_request_not_chosen_column_lob".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("integer".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("Aurora_pq_request_not_chosen_column_virtual".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("integer".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("Aurora_pq_request_not_chosen_custom_charset".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("integer".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("Aurora_pq_request_not_chosen_fast_ddl".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("integer".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some(
                        "Aurora_pq_request_not_chosen_few_pages_outside_buffer_pool".to_string()
                    ),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("integer".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("Aurora_pq_request_not_chosen_full_text_index".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("integer".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("Aurora_pq_request_not_chosen_high_buffer_pool_pct".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("integer".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("Aurora_pq_request_not_chosen_index_hint".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("integer".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("Aurora_pq_request_not_chosen_innodb_table_format".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("integer".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("Aurora_pq_request_not_chosen_long_trx".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("integer".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("Aurora_pq_request_not_chosen_no_where_clause".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("integer".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("Aurora_pq_request_not_chosen_range_scan".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("integer".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("Aurora_pq_request_not_chosen_row_length_too_long".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("integer".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("Aurora_pq_request_not_chosen_small_table".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("integer".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("Aurora_pq_request_not_chosen_temporary_table".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("integer".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("Aurora_pq_request_not_chosen_tx_isolation".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("integer".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("Aurora_pq_request_not_chosen_update_delete_stmts".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("integer".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("Aurora_pq_request_not_chosen_unsupported_access".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("integer".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("Aurora_pq_request_not_chosen_unsupported_storage_type".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("integer".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("Aurora_pq_request_throttled".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("integer".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("Aurora_repl_bytes_received".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("byte".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("Aurora_reserved_mem_exceeded_incidents".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("integer".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("Aurora_thread_pool_thread_count".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("integer".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("Aurora_tmz_version".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("string".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
                KbParsedEntry {
                    cli: None,
                    default: None,
                    dynamic: None,
                    name: Some("server_aurora_das_running".to_string()),
                    id: None,
                    range: None,
                    scope: None,
                    r#type: Some("boolean".to_string()),
                    valid_values: None,
                    has_description: true,
                    is_removed: false,
                },
            ],
            entries
        );
    }
}
