/*
 * libsyncml - A syncml protocol implementation
 * Copyright (C) 2005  Armin Bauer <armin.bauer@opensync.org>
 * Copyright (C) 2008  Michael Bell <michael.bell@opensync.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 */

#ifndef _SML_PARSE_H_
#define _SML_PARSE_H_

typedef enum {
	SML_PARSER_RESULT_ERROR = 0,
	SML_PARSER_RESULT_OPEN = 1,
	SML_PARSER_RESULT_CLOSE = 2,
	SML_PARSER_RESULT_NORMAL = 3,
	SML_PARSER_RESULT_STATUS = 4,
	SML_PARSER_RESULT_OTHER = 5
} SmlParserResult;

typedef void (* SmlParserFreeFunction) (void *userdata);
typedef SmlBool (* SmlParserStartFunction) (void *userdata, const char *data, unsigned int size, SmlError **error);
typedef SmlBool (* SmlParserEndFunction) (void *userdata, SmlBool *final, SmlBool *end, SmlError **error);
typedef SmlBool (* SmlParserHeaderFunction) (void *userdata, SmlHeader **header, SmlCred **cred, SmlError **error);
typedef SmlBool (* SmlParserStatusFunction) (void *userdata, SmlStatus **status, SmlError **error);
typedef SmlParserResult (* SmlParserCommandFunction) (void *userdata, SmlCommand **cmd, SmlError **error);

typedef struct SmlParserFunctions {
	SmlParserFreeFunction free;
	SmlParserStartFunction start;
	SmlParserEndFunction end;
	SmlParserHeaderFunction get_header;
	SmlParserStatusFunction get_status;
	SmlParserCommandFunction get_cmd;
} SmlParserFunctions;

typedef void (* SmlAssemblerFreeFunction) (void *userdata);
typedef SmlBool (* SmlAssemblerStartFunction) (void *userdata, SmlSession *session, SmlError **error);
typedef SmlBool (* SmlAssemblerEndFunction) (void *userdata, SmlError **error);
typedef SmlBool (* SmlAssemblerRunFunction) (void *userdata, char **data, unsigned int *size, SmlBool *end, SmlBool final, unsigned int maxsize, SmlError **error);
typedef unsigned int (* SmlAssemblerFlushFunction) (void *userdata);
typedef SmlBool (* SmlAssemblerStatusFunction) (void *userdata, SmlStatus *status, SmlError **error);
typedef SmlBool (* SmlAssemblerRemStatusFunction) (void *userdata, SmlError **error);
typedef SmlBool (* SmlAssemblerReserveStatusFunction) (void *userdata, unsigned int cmdRef, unsigned int msgRef, unsigned int cmdID, SmlError **error);
typedef SmlBool (* SmlAssemblerStatusMissingFunction) (void *userdata);
typedef SmlBool (* SmlAssemblerStartCommandFunction) (void *userdata, unsigned int parentID, SmlCommand *cmd, SmlError **error);
typedef SmlBool (* SmlAssemblerEndCommandFunction) (void *userdata, unsigned int parentID, SmlError **error);
typedef SmlBool (* SmlAssemblerRemCommandFunction) (void *userdata, unsigned int parentID, SmlError **error);
typedef SmlBool (* SmlAssemblerHeaderFunction) (void *userdata, SmlSession *session, SmlError **error);
typedef unsigned int (* SmlAssemblerCheckFunction) (void *userdata, SmlBool headeronly, SmlError **error);
typedef SmlBool (* SmlAssemblerNextCmdRefFunction) (void *userdata, unsigned int *cmdRef, unsigned int *msgRef);
typedef void (* SmlAssemblerRestoreCommandsFunction) (void *userdata);

typedef struct SmlAssemblerFunctions {
	SmlAssemblerFreeFunction free;
	SmlAssemblerStartFunction start;
	SmlAssemblerEndFunction end;
	SmlAssemblerRunFunction run;
	SmlAssemblerFlushFunction flush;
	SmlAssemblerStatusFunction add_status;
	SmlAssemblerRemStatusFunction rem_status;
	SmlAssemblerReserveStatusFunction reserve_status;
	SmlAssemblerStatusMissingFunction missing_status;
	SmlAssemblerStartCommandFunction start_cmd;
	SmlAssemblerEndCommandFunction end_cmd;
	SmlAssemblerRemCommandFunction rem_cmd;
	SmlAssemblerHeaderFunction add_header;
	SmlAssemblerCheckFunction check_size;
	SmlAssemblerNextCmdRefFunction next_cmdref;
	SmlAssemblerRestoreCommandsFunction restore_cmds;
} SmlAssemblerFunctions;

typedef enum {
	SML_ASSEMBLER_RESULT_ERROR = 0,
	SML_ASSEMBLER_RESULT_OK = 1,
	SML_ASSEMBLER_RESULT_MISMATCH =2
} SmlAssemblerResult;

SmlParser *smlParserNew(SmlMimeType type, unsigned int limit, SmlError **error);
void smlParserFree(SmlParser *parser);
SmlBool smlParserStart(SmlParser *parser, const char *data, unsigned int size, SmlError **error);
SmlBool smlParserGetHeader(SmlParser *parser, SmlHeader **header, SmlCred **cred, SmlError **error);
SmlParserResult smlParserGetCommand(SmlParser *parser, SmlCommand **cmd, SmlError **error);
SmlBool smlParserGetStatus(SmlParser *parser, SmlStatus **status, SmlError **error);
SmlBool smlParserEnd(SmlParser *parser, SmlBool *final, SmlBool *end, SmlError **error);

SmlAssembler *smlAssemblerNew(SmlMimeType type, unsigned int limit, SmlError **error);
void smlAssemblerFree(SmlAssembler *assm);
SmlBool smlAssemblerStart(SmlAssembler *assm, SmlSession *session, SmlError **error);
unsigned int smlAssemblerFlush(SmlAssembler *assm);
void smlAssemblerRestoreCommands(SmlAssembler *assm);
SmlBool smlAssemblerRun(SmlAssembler *assm, char **data, unsigned int *size, SmlBool *end, SmlBool final, SmlError **error);
SmlAssemblerResult smlAssemblerAddStatus(SmlAssembler *assm, SmlStatus *status, SmlError **error);
SmlAssemblerResult smlAssemblerAddStatusFull(SmlAssembler *assm, SmlStatus *status, SmlBool force, SmlError **error);
SmlAssemblerResult smlAssemblerReserveStatus(SmlAssembler *assm, unsigned int cmdRef, unsigned int msgRef, unsigned int cmdID, SmlError **error);
SmlBool smlAssemblerAddHeader(SmlAssembler *assm, SmlSession *session, SmlError **error);
SmlAssemblerResult smlAssemblerStartCommand(SmlAssembler *assm, SmlCommand *parent, SmlCommand *cmd, SmlError **error);
SmlBool smlAssemblerEndCommand(SmlAssembler *assm, SmlCommand *parent, SmlError **error);
unsigned int smlAssemblerCheckSize(SmlAssembler *assm, SmlBool headeronly, SmlError **error);
void smlAssemblerSetOption(SmlAssembler *assm, const char *optionname, const char *value);
const char *smlAssemblerGetOption(SmlAssembler *assm, const char *optionname);

unsigned int smlAssemblerSetRemoteMaxMsgSize(SmlAssembler *assm, unsigned int limit);
unsigned int smlAssemblerGetRemoteMaxMsgSize(SmlAssembler *assm);
SmlBool smlAssemblerGetSpace(SmlAssembler *assm, int *space, SmlCommand *parent, SmlCommand *cmd, SmlError **error);

unsigned int smlAssemblerGetRemoteMaxObjSize(SmlAssembler *assm);
unsigned int smlAssemblerSetRemoteMaxObjSize(SmlAssembler *assm, unsigned int limit);

SmlBool smlAssemblerIsEmpty(SmlAssembler *assm);
SmlBool smlAssemblerIsStatusMissing(SmlAssembler *assm);
SmlBool smlAssemblerGetNextCmdRef(SmlAssembler *assm, unsigned int *cmdRef, unsigned int *msgRef);

/* expire date: 20090626 */
void smlAssemblerSetRequestedLimit(SmlAssembler *assm, unsigned int limit) LIBSYNCML_DEPRECATED;
void smlAssemblerSetLimit(SmlAssembler *assm, unsigned int limit) LIBSYNCML_DEPRECATED;
unsigned int smlAssemblerGetLimit(SmlAssembler *assm) LIBSYNCML_DEPRECATED;
void smlAssemblerSetRequestedMaxObjSize(SmlAssembler *assm, int limit) LIBSYNCML_DEPRECATED;
void smlAssemblerSetSendingMaxObjSize(SmlAssembler *assm, int limit) LIBSYNCML_DEPRECATED;
int smlAssemblerGetSendingMaxObjSize(SmlAssembler *assm) LIBSYNCML_DEPRECATED;
int smlAssemblerGetRequestedMaxObjSize(SmlAssembler *assm) LIBSYNCML_DEPRECATED;

/* If you want to parse OMA DS 1.2 SANs then the manager is required. */
void smlParserSetManager(SmlParser *parser, SmlManager *manager);

#endif //_SML_PARSE_H_
