/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * version 2 of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include <QtGui>
#include "QtShiva.h"

#include <QTextEdit>
#include <QTextStream>
#include <QCloseEvent>
#include <QFileDialog>

#include "ui_MainWindow.h"

#include <OpenShiva/Kernel.h>
#include <GTLCore/PixelDescription.h>
#include <GTLCore/Region.h>
#include <GTLCore/Region.h>
#include <GTLCore/Type.h>
#include <GTLCore/Version.h>

#include "QtGTL/QImageImage.h"
#include <GTLCore/CompilationMessages.h>

qShiva::qShiva()
{
  m_mainWindow = new Ui_MainWindow;
  m_kernel = new OpenShiva::Kernel( 4 );
  QWidget* widget = new QWidget;
  m_mainWindow->setupUi(widget);
  setCentralWidget(widget);
  connect(m_mainWindow->buttonAdd, SIGNAL(released()), SLOT(add()));
  connect(m_mainWindow->buttonRemove, SIGNAL(released()), SLOT(remove()));
  connect(m_mainWindow->buttonApply, SIGNAL(released()), SLOT(apply()));
  connect(m_mainWindow->buttonRefresh, SIGNAL(released()), SLOT(refreshImageViewer()));
  connect(m_mainWindow->ctlCodeEditor, SIGNAL(textChanged()), SLOT(sourceChanged()));
  sourceChanged();
}

qShiva::~qShiva()
{
  delete m_mainWindow;
}

void qShiva::refreshImageViewer()
{
  QPixmap pm = QPixmap::fromImage(currentImage);
  m_mainWindow->imageViewer->setPixmap(pm.scaled( m_mainWindow->imageViewer->size(), Qt::KeepAspectRatio));
}

void qShiva::add()
{
  QString fileName = QFileDialog::getOpenFileName(this);
  if (fileName.isEmpty()) return;
  QListWidgetItem* lwi = new QListWidgetItem( QFileInfo(fileName).fileName() );
  lwi->setData(Qt::UserRole, fileName );
  m_mainWindow->listImages->addItem( lwi );
}

void qShiva::remove()
{
  delete m_mainWindow->listImages->currentItem();
}

int clamp255( int v)
{
  if( v > 255 ) return 255;
  if( v < 0 ) return 0;
  return v;
}

void qShiva::sourceChanged()
{
  m_kernel->setSource( m_mainWindow->ctlCodeEditor->toPlainText().toAscii().data());
  m_mainWindow->parametersWidget->setKernel( m_kernel );
}

void qShiva::apply()
{
  GTLCore::PixelDescription pixel( GTLCore::Type::Float32, 3 );
  m_kernel->compile();
  if(m_kernel->isCompiled())
  {
#if GTL_CORE_VERSION_REVISION < 13 && GTL_CORE_VERSION_MINOR == 9 && GTL_CORE_VERSION_MAJOR == 0
    std::list<GTLCore::AbstractImage*> inputImages;
#else
    std::list<const GTLCore::AbstractImage*> inputImages;
#endif
    for( int k = 0; k < m_mainWindow->listImages->count(); ++k)
    {
      QString fileName = m_mainWindow->listImages->item( k )->data( Qt::UserRole ).toString();
      QImage img;
      if (not img.load(fileName)) {
        QMessageBox::warning(this, tr("qShiva"),
                              tr("Cannot read image %1..")
                              .arg(fileName));
        return;
      }
      GTLCore::AbstractImage* iImage = new QtGTL::QImageImage( img );
      inputImages.push_back( iImage );
    }
    QtGTL::QImageImage image( QImage( 1000, 1000, QImage::Format_RGB32) );
    m_kernel->evaluatePixels( GTLCore::RegionI(0,0, 1000,1000), inputImages, &image );
    currentImage = image.image();
    for( std::list<const GTLCore::AbstractImage*>::iterator it = inputImages.begin();
         it != inputImages.end(); ++it)
    {
      delete *it;
    }
    refreshImageViewer();
  } else {
    QMessageBox::critical(this, tr("qShiva"), tr("Compilation error: %1").arg( m_kernel->compilationMessages().toString().c_str() ));
  }
}

#include "QtShiva.moc"
