package tests::LrTimegroupTest;

use strict;

use base qw/Lire::Test::TestCase/;

use Lire::SQLExt::LrTimegroup;

use POSIX qw/strftime tzset/;
sub test_lr_timegroup_sec {
    my $self = $_[0];

    my @tests =
      (
       [ '2003-10-13 14:00 EDT', 1066068720, 3600, 1066068000 ],
       [ '2003-10-13 14:00 EDT', 1066068720.100, 3600, 1066068000 ],
       [ '2003-10-13 14:12 EDT', 1066069440, 3600, 1066068000 ],
       [ '2003-10-13 14:59:59 EDT', 1066071599, 3600, 1066068000 ],
       [ '2003-10-13 15:59:59 EDT', 1066075199, 3600, 1066071600 ],
       [ '2003-10-13 20:12 EDT', 1066090320, 4*3600,  1066089600 ],
       [ '2003-10-13 23:15 EDT', 1066101300, 4*3600, 1066089600 ],
       [ '2003-10-14 00:15 EDT', 1066104900, 4*3600, 1066104000 ],
       );

    $self->assert_null( Lire::SQLExt::LrTimegroup::lr_timegroup_sec( undef, 3600 ),
                        "lr_timegroup_sec( undef ) should return undef" );
    foreach my $t ( @tests ) {
        my ( $date, $epoch, $period_size, $result ) = @$t;
        my $r =
          Lire::SQLExt::LrTimegroup::lr_timegroup_sec( $epoch, $period_size);
        my $result_fmt = strftime( '%Y-%m-%d %H:%M', localtime $result );
        my $r_fmt = strftime( '%Y-%m-%d %H:%M', localtime $r );
        $self->assert_equals( $result, $r, 
                              "lr_timegroup_sec( '$date', $period_size ) returned $r_fmt, expected $result_fmt" );
    }
}

sub test_lr_timegroup_day {
    my $self = $_[0];

    my @tests = ( [ '2003-10-14 00:15 EST', 'EST', 1066104900, 1066107600 ],
                  [ '2003-10-14 00:15 EST', 'EST', 1066104900.367, 1066107600],
                  [ '2003-07-14 23:15 CET', 'CET', 1058217300, 1058133600 ],
                  [ '2003-12-12 23:15 CET', 'CET', 1071267300, 1071183600 ],
                  [ '2003-10-14 00:15 UTC', 'UTC', 1066090500, 1066089600 ] );
    $self->assert_null( Lire::SQLExt::LrTimegroup::lr_timegroup_day( undef ),
                        "lr_timegroup_day( undef ) should return undef" );
    my $old_tz = $ENV{'TZ'};
    foreach my $t ( @tests ) {
        my ( $date, $tz, $epoch, $result ) = @$t;
        @Timezone::tz_local = ();
        $ENV{'TZ'} = $tz;
        tzset();
        my $r = Lire::SQLExt::LrTimegroup::lr_timegroup_day( $epoch );
        my $result_fmt = strftime( '%Y-%m-%d %H:%M %Z', localtime $result );
        my $r_fmt = strftime( '%Y-%m-%d %H:%M %Z', localtime $r );
        $ENV{'TZ'} =  $old_tz || '';
        tzset();
        $self->assert_equals( $result, $r,
                              "lr_timegroup_day( '$date' ) returned $r_fmt($r), expected $result_fmt($result)" );
    }
}

sub test_lr_timegroup_week {
    my $self = $_[0];

    my @tests =
      ( [ '2003-01-01 00:15 CET', 'CET', 'ISO', 1, 1041376500, 1041202800 ],
        [ '2003-01-01 00:15 CET', 'CET', 'ISO', 1, 1041376500.444, 1041202800],
        [ '2003-01-01 00:15 CET', 'CET', 'ISO', 4, 1041376500, 1041202800 ],
        [ '2002-12-31 23:15 CET', 'CET', 'ISO', 1, 1041372900, 1041202800 ],
        [ '2003-10-14 12:00 UTC', 'UTC', 'ISO', 1, 1066125600, 1066003200 ],
        [ '2003-10-14 12:00 UTC', 'UTC', 'ISO', 4, 1066125600, 1065398400 ],
        [ '2003-01-22 00:00 EST', 'EST', 'ISO', 4, 1043211600, 1041224400 ],
        [ '2003-01-29 00:00 EST', 'EST', 'ISO', 4, 1043816400, 1043643600 ],
        [ '2003-02-19 00:00 EST', 'EST', 'ISO', 4, 1045630800, 1043643600 ],

        # Weeks start on sunday
        [ '2003-01-01 00:15 CET', 'CET', 'U', 1, 1041376500, 1041116400 ],
        # Week 0 is grouped in Week 1
        [ '2003-01-01 00:15 CET', 'CET', 'U', 4, 1041376500, 1041721200 ],
        [ '2002-12-31 23:15 CET', 'CET', 'U', 1, 1041372900, 1041116400 ],
        [ '2003-10-14 12:00 UTC', 'UTC', 'U', 1, 1066125600, 1065916800 ],
        [ '2003-10-14 12:00 UTC', 'UTC', 'U', 4, 1066125600, 1065916800 ],
        [ '2003-01-29 00:00 EST', 'EST', 'U', 4, 1043816400, 1041742800 ],
        [ '2003-02-04 00:00 EST', 'EST', 'U', 4, 1044334800, 1044162000 ],
      );
    $self->assert_null( Lire::SQLExt::LrTimegroup::lr_timegroup_week( undef ),
                        "lr_timegroup_week( undef ) should return undef" );
    my $old_tz = $ENV{'TZ'};
    foreach my $t ( @tests ) {
        my ( $date, $tz, $style, $mult, $epoch, $result ) = @$t;
        $ENV{'TZ'} = $tz;
        tzset();
        $self->{'cfg'}{'lr_week_numbering'} = $style;
        $Lire::SQLExt::LrTimegroup::calc = undef;
        my $r = Lire::SQLExt::LrTimegroup::lr_timegroup_week( $epoch, $mult );
        my $result_fmt = strftime( '%Y-%m-%d (%G-%V) %Z', localtime $result );
        my $r_fmt = strftime( '%Y-%m-%d (%G-%V) %Z', localtime $r );
        $ENV{'TZ'} =  $old_tz || '';
        tzset();
        $self->assert_equals( $result, $r,
                              "lr_timegroup_week( '$date', $mult ) using $style returned $r_fmt($r), expected $result_fmt($result)" );
    }
}

sub test_lr_timegroup_month {
    my $self = $_[0];

    my @tests =
      ( [ '2002-12-31 00:00 CET', 'CET',  1, 1041289200, 1038697200 ],
        [ '2002-12-31 00:00 CET', 'CET',  3, 1041289200, 1033423200 ],
        [ '2003-01-01 00:00 UTC', 'UTC',  1, 1041379200, 1041379200 ],
        [ '2003-01-01 00:00 UTC', 'UTC',  1, 1041379200.346, 1041379200 ],
        [ '2003-01-01 00:00 UTC', 'UTC',  3, 1041379200, 1041379200 ],
        [ '2003-03-01 00:00 UTC', 'UTC',  3, 1046476800, 1041379200 ],
        [ '2003-04-05 00:00 EST', 'EST',  1, 1049518800, 1049173200 ],
        [ '2003-04-05 00:00 EST', 'EST',  3, 1049518800, 1049173200 ],
      );
    $self->assert_null( Lire::SQLExt::LrTimegroup::lr_timegroup_month( undef ),
                        "lr_timegroup_month( undef ) should return undef" );
    my $old_tz = $ENV{'TZ'};
    foreach my $t ( @tests ) {
        my ( $date, $tz, $mult, $epoch, $result ) = @$t;
        $ENV{'TZ'} = $tz;
        tzset();

        my $r = Lire::SQLExt::LrTimegroup::lr_timegroup_month( $epoch, $mult );
        my $result_fmt = strftime( '%Y-%m-%d %Z', localtime $result );
        my $r_fmt = strftime( '%Y-%m-%d %Z', localtime $r );
        $ENV{'TZ'} =  $old_tz || '';
        tzset();
        $self->assert_equals( $result, $r,
                              "lr_timegroup_month( '$date', $mult ) returned $r_fmt($r), expected $result_fmt($result)" );
    }
}

sub test_lr_timegroup_year {
    my $self = $_[0];

    my @tests =
      ( [ '2002-12-31 00:00 CET', 'CET',  1, 1041289200, 1009839600 ],
        [ '2002-12-31 00:00 CET', 'CET',  1, 1041289200.765, 1009839600 ],
        [ '2000-01-01 00:00 UTC', 'UTC',  10, 946684800, 662688000 ],
        [ '2001-01-01 00:00 UTC', 'UTC',  10, 978307200, 978307200 ],
      );
    $self->assert_null( Lire::SQLExt::LrTimegroup::lr_timegroup_year( undef ),
                        "lr_timegroup_year( undef ) should return undef" );
    my $old_tz = $ENV{'TZ'};
    foreach my $t ( @tests ) {
        my ( $date, $tz, $mult, $epoch, $result ) = @$t;
        $ENV{'TZ'} = $tz;
        tzset();

        my $r = Lire::SQLExt::LrTimegroup::lr_timegroup_year( $epoch, $mult );
        my $result_fmt = strftime( '%Y-%m-%d %Z', localtime $result );
        my $r_fmt = strftime( '%Y-%m-%d %Z', localtime $r );
        $ENV{'TZ'} =  $old_tz || '';
        tzset();
        $self->assert_equals( $result, $r,
                              "lr_timegroup_year( '$date', $mult ) returned $r_fmt($r), expected $result_fmt($result)" );
    }
}

1;
