package tests::ConfigXMLSpecListSpecTest;

use strict;

use tests::ConfigSpecTest;

use base qw/ tests::ConfigTypeSpecBase tests::TestStoreFixture
             tests::ChartTypesFixture /;

use Lire::ReportSpec;
use Lire::Config::XMLSpecListSpec;
use Lire::Utils qw/ create_file tempdir /;
use Lire::DlfSchema;

sub new {
    my $self = shift->SUPER::new( @_ );

    $self->{'tmpdir'} = tempdir( __PACKAGE__ . '_XXXXXX',
                                 'CLEANUP' => 1 );
    $self->init();

    return $self;
}

sub set_up {
    my $self = $_[0];

    $self->SUPER::set_up();

    $self->{'_old_schemas'} = \%Lire::DlfSchema::SCHEMA_CACHE;
    %Lire::DlfSchema::SCHEMA_CACHE = ();
    $self->{'cfg'}{'lr_reports_path'} = [ $self->{'tmpdir'} ];
    $self->{'cfg'}{'lr_schemas_path'} = [];

    $self->set_up_test1();
    $self->set_up_test2();
    return;
}

sub set_up_test1 {
    my $self = $_[0];

    $Lire::DlfSchema::SCHEMA_CACHE{'test1'} = 1;
    mkdir "$self->{'tmpdir'}/test1", 0755
      or $self->error( "mkdir failed: $!" );

    create_file( "$self->{'tmpdir'}/test1/report1.xml", '' );
    create_file( "$self->{'tmpdir'}/test1/report2.xml", '' );
}

sub set_up_test2 {
    my $self = $_[0];

    $Lire::DlfSchema::SCHEMA_CACHE{'test2'} = 1;
    mkdir "$self->{'tmpdir'}/test2", 0755
      or $self->error( "mkdir failed: $!" );
    create_file( "$self->{'tmpdir'}/test2/report2.xml", '' );
}

sub tear_down {
    my $self = $_[0];
    $self->SUPER::tear_down();

    *Lire::DlfSchema::SCHEMA_CACHE = $self->{'_old_schemas'};

    return;
}

sub additional_new_params {
    return ( 'type' => 'reports' );
}

sub type {
    return 'Lire::Config::XMLSpecListSpec';
}

sub test_new {
    my $self = $_[0];

    $self->SUPER::test_new();

    $self->assert_str_equals( 'Lire::ReportSpec', $self->{'spec'}{'_module'} );
    $self->assert_str_equals( 'reports', $self->{'spec'}{'_type'} );
    $self->assert_str_equals( 'reports', $self->{'spec'}->type() );
}

sub test_has_component {
    my $self = $_[0];

    my $spec = $self->{'spec'};
    $self->assert( $spec->has_component( 'test1:report1' ) ? 1 : 0,
                   'has_component( "test1:report1" )' );
    $self->assert( $spec->has_component( 'test1:report2' ) ? 1 : 0,
                   'has_component( "test1:report2" )' );
    $self->assert( $spec->has_component( 'test2:report2' ) ? 1 : 0,
                   'has_component( "test2:report2" )' );
    $self->assert( !$spec->has_component( 'test2:report1' ) ? 1 : 0,
                   '! has_component( "test2:report1" )' );
    $self->assert( !$spec->has_component( 'test:another_report' ) ? 1 : 0,
                   '! has_component( "test:report1" )' );

    $self->assert( !$spec->has_component( 'missing' ) ? 1 : 0,
                   '! has_component( "missing" )' );
}

sub test_component_names {
    my $self = $_[0];

    $self->assert_deep_equals( [ sort qw/ test1:report1 test1:report2
                                          test2:report2 / ],
                               [ sort $self->{'spec'}->component_names() ] );
}

sub test_get {
    my $self = $_[0];

    $self->set_up_chart_types();
    $self->set_up_test_schema();
    $self->{'cfg'}{'lr_reports_path'} = [ "$self->{'testdir'}/reports" ];
    $self->assert_dies( qr/no component named \'param\'/,
                        sub { $self->{'spec'}->get( 'param' ) } );

    my $spec = $self->{'spec'}->get( 'test:top-dirs2' );
    $self->assert_str_equals( $spec, $self->{'spec'}->get( 'test:top-dirs2' ) );
    $self->assert_isa( 'Lire::Config::ObjectSpec', $spec );
    $self->assert_str_equals( 'Lire::ReportSpec', $spec->class() );
    $self->assert_str_equals( 'test:top-dirs2', $spec->name() );
    $self->assert_str_equals( 'Top Directories Report Specification',
                              $spec->summary() );
    $self->assert_str_equals( ' <para>Test report specification using an extended schema.</para> ',
                              $spec->description() );
    $self->assert_str_equals( 'lire-test', $spec->{'i18n_domain'} );
    $self->assert_num_equals( 4, scalar $spec->components() );
    $self->assert_str_equals( 'id', $spec->label_component() );

    my $id = $spec->get( 'id' );
    $self->assert_isa( 'Lire::Config::StringSpec', $id );
    $self->assert_str_equals( '^[\w.:-]+$', $id->valid_re() );
    $self->assert_not_null( $id->text_description() );

    my $title = $spec->get( 'title' );
    $self->assert_isa( 'Lire::Config::StringSpec', $title );
    $self->assert_num_equals( 0, $title->required() );
    $self->assert_not_null( $title->text_description()  );
    $self->assert_not_null( $title->default() );
    $self->assert_str_equals( 'Downloads by Directory, Top $dirs_to_show',
                              $title->default()->get() );

    $self->assert_isa( 'Lire::Config::IntegerSpec',
                       $spec->get( 'dirs_to_show' ) );

    my $charts = $spec->get( 'charts' );
    $self->assert_isa( 'Lire::Config::ListSpec', $charts );
    $self->assert_not_null( $title->text_description()  );
    $self->assert_not_null( $charts->default() );
    $self->assert_num_equals( 1, scalar $charts->component_names() );
    $self->assert_isa( 'Lire::Config::ChartSpec', $charts->get( 'chart' ) );

    my $chart_configs = $charts->default()->as_value();
    $chart_configs->[0]->spec()->summary( undef ); # For the comparison
    $chart_configs->[0]->spec()->description( undef );
    $self->assert_deep_equals( Lire::ReportSpec->load( 'test', 'top-dirs2' )->chart_configs(), $chart_configs );
}

sub test_components {
    my $self = $_[0];

    $self->set_up_chart_types();
    $self->set_up_test_schema();
    $self->{'cfg'}{'lr_reports_path'} = [ "$self->{'testdir'}/reports" ];

    my @names = $self->{'spec'}->component_names();
    my @comps = $self->{'spec'}->components();
    $self->assert( scalar @names > 1 );
    $self->assert_num_equals( scalar @names, scalar @comps );
}

1;

