package tests::ConfigSpecParserTest;

use strict;

use base qw/ Lire::Test::TestCase /;

use Lire::Config::SpecParser;
use Lire::Config::TypeSpec;

use Lire::Utils qw/ tempdir create_file/;

sub new {
    my $self = shift()->SUPER::new( @_ );

    $self->{'specdir'} = tempdir( "config-spec_XXXXXX", 'CLEANUP' => 1 );
    $self->{'expected_spec'} = $self->build_expected_spec();
    $self->_make_config_spec();

    return $self;
}

sub set_up {
    my $self = $_[0];
    $self->SUPER::set_up();

    return;
}

sub tear_down {
    my $self = $_[0];
    $self->SUPER::tear_down();

    return;
}

sub build_expected_spec {
    my $self = $_[0];

    $self->{'bool_desc'} = "<para>This is an experimental option</para>";
    my $cfg = new Lire::Config::ConfigSpec();
    my $bool = new Lire::Config::BooleanSpec( 'name' => "bool_param" );
    $bool->summary( "This is a boolean parameter" );
    $bool->description( $self->{'bool_desc'} );
    $bool->default( $bool->instance( 'value' => 1 ) );
    $cfg->add( $bool );
    $cfg->add( new Lire::Config::IntegerSpec( 'name' => "int_param" ) );
    $cfg->add( new Lire::Config::DlfConverterSpec( 'name' => "dlfconverter_param" ) );
    $cfg->add( new Lire::Config::CommandSpec( 'name' => "command_param" ) );
    $cfg->add( new Lire::Config::DirectorySpec( 'name' => "dir_param" ) );
    $cfg->add( new Lire::Config::FileSpec( 'name' => "file_param" ) );
    $cfg->add( new Lire::Config::ExecutableSpec( 'name' => "executable_param" ) );

    my $select = new Lire::Config::SelectSpec( 'name' => "select_param" );
    $cfg->add( $select );
    $select->add( new Lire::Config::OptionSpec( 'name' => "option_1" ) );
    $select->add( new Lire::Config::OptionSpec( 'name' => "option_2" ) );

    my $list = new Lire::Config::ListSpec( 'name' => "list_param" );
    $cfg->add( $list );
    $list->add( new Lire::Config::StringSpec( 'name' => "string_param" ));

    my $rec = new Lire::Config::RecordSpec( 'name' => 'myrec' );
    $rec->add( new Lire::Config::StringSpec( 'name' => 'first' ) );
    $rec->add( new Lire::Config::StringSpec( 'name' => 'second' ) );

    my $rec_dflt = $rec->instance();
    $rec_dflt->get( 'first' )->set( 1 );
    $rec_dflt->get( 'second' )->set( 2 );
    $rec->default( $rec_dflt );
    $cfg->add( $rec );

    $cfg->add( new Lire::Config::ReportSpec( 'name' => 'my_report' ) );

    return $cfg;
}

sub _make_config_spec {
    my $self = $_[0];

    $self->{'spec_file'} = "$self->{'specdir'}/spec.xml";
    create_file( $self->{'spec_file'}, <<SPEC );
<lrcsml:config-spec 
  xmlns:lrcsml="http://www.logreport.org/LRCSML/"
  xmlns:lrcml="http://www.logreport.org/LRCML/">
 <lrcsml:boolean name="bool_param">
  <lrcsml:summary>This is a boolean parameter</lrcsml:summary>
  <lrcsml:description>$self->{'bool_desc'}</lrcsml:description>
  <lrcml:param name="bool_param" value="1"/>
 </lrcsml:boolean>
 <lrcsml:integer name="int_param"/>
 <lrcsml:dlf-converter name="dlfconverter_param"/>
 <lrcsml:command name="command_param"/>
 <lrcsml:directory name="dir_param"/>
 <lrcsml:file name="file_param"/>
 <lrcsml:executable name="executable_param"/>
 <lrcsml:select name="select_param">
  <lrcsml:option name="option_1"/>
  <lrcsml:option name="option_2"/>
 </lrcsml:select>
 <lrcsml:list name="list_param">
  <lrcsml:string name="string_param"/>
 </lrcsml:list>
 <lrcsml:record name="myrec">
  <lrcsml:string name="first"/>
  <lrcsml:string name="second"/>
  <lrcml:param name="myrec">
    <lrcml:param name="first" value="1"/>
    <lrcml:param name="second" value="2"/>
  </lrcml:param>
 </lrcsml:record>
 <lrcsml:report-config name="my_report"/>
</lrcsml:config-spec>
SPEC

    return;
}

sub test_new {
    my $self = $_[0];

    my $parser = new Lire::Config::SpecParser();
    $self->assert_isa( 'Lire::Config::SpecParser', $parser );
}

sub test_merge_specifications_dir {
    my $self = $_[0];

    my $parser = new Lire::Config::SpecParser();
    $parser->merge_specifications_dir( $self->{'specdir'} );

    $self->assert_isa( 'Lire::Config::ConfigSpec', $parser->configspec() );
    $self->assert_deep_equals( $self->{'expected_spec'},
                               $parser->configspec() );
}

sub test_merge_specification {
    my $self = $_[0];

    my $parser = new Lire::Config::SpecParser();
    $parser->merge_specification( "$self->{'spec_file'}" );

    $self->assert_isa( 'Lire::Config::ConfigSpec', $parser->configspec() );
    $self->assert_deep_equals( $self->{'expected_spec'},
                               $parser->configspec() );
}

1;
