/* Copyright (c) 2011, Code Aurora Forum. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 and
 * only version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */


/*
[SENSOR]
Sensor Model:   MI1040
Camera Module:
Lens Model:
Driver IC:
PV Size         = 1280 x 960
Cap Size        = 1280 x 960
Output Format   = YCbCr
MCLK Speed      = 24M
PV Frame Rate   = 30fps
Cap Frame Rate  = 30fps
I2C Slave ID    = 0x90
I2C Mode        = 16Addr, 16Data
*/

#ifndef CAMSENSOR_MI1040
#define CAMSENSOR_MI1040

/* Sysctl registers */
#define MI1040_COMMAND_REGISTER                0x0080
#define MI1040_COMMAND_REGISTER_APPLY_PATCH    (1 << 0)
#define MI1040_COMMAND_REGISTER_SET_STATE      (1 << 1)
#define MI1040_COMMAND_REGISTER_REFRESH        (1 << 2)
#define MI1040_COMMAND_REGISTER_WAIT_FOR_EVENT (1 << 3)
#define MI1040_COMMAND_REGISTER_OK             (1 << 15)

enum Vendor_ID{
    A,
    B,
};

static uint16_t mi1040_flicker_detect_table[] = {
0xdc10, 0xc0f1, 0x0cda, 0x0580, 0x76cf, 0xff00, 0x2184, 0x9624, 0x218c, 0x8fc3, 0x75cf, 0xffff, 0xe058, 0xf686, 0x1550, 0x1080, 0xe001, 0x1d50, 0x1002, 0x1552, 0x1100, 0x6038, 0x1d52, 0x1004, 0x1540,
0xdc40, 0x1080, 0x081b, 0x00d1, 0x8512, 0x1000, 0x00c0, 0x7822, 0x2089, 0x0fc1, 0x2008, 0x0f81, 0xffff, 0xff80, 0x8512, 0x1801, 0x0052, 0xa512, 0x1544, 0x1080, 0xb861, 0x262f, 0xf007, 0x1d44, 0x1002,
0xdc70, 0x20ca, 0x0021, 0x20cf, 0x04e1, 0x0850, 0x04a1, 0x21ca, 0x0021, 0x1542, 0x1140, 0x8d2c, 0x6038, 0x1d42, 0x1004, 0x1542, 0x1140, 0xb601, 0x046d, 0x0580, 0x78e0, 0xd800, 0xb893, 0x002d, 0x04a0,
0xdca0, 0xd900, 0x78e0, 0x72cf, 0xffff, 0xe058, 0x2240, 0x0340, 0xa212, 0x208a, 0x0fff, 0x1a42, 0x0004, 0xd830, 0x1a44, 0x0002, 0xd800, 0x1a50, 0x0002, 0x1a52, 0x0004, 0x1242, 0x0140, 0x8a2c, 0x6038,
0xdcd0, 0x1a42, 0x0004, 0x1242, 0x0141, 0x70cf, 0xff00, 0x2184, 0xb021, 0xd800, 0xb893, 0x07e5, 0x0460, 0xd901, 0x78e0, 0xc0f1, 0x0bfa, 0x05a0, 0x216f, 0x0043, 0xc1a4, 0x220a, 0x1f80, 0xffff, 0xe058,
0xdd00, 0x2240, 0x134f, 0x1a48, 0x13c0, 0x1248, 0x1002, 0x70cf, 0x7fff, 0xffff, 0xe230, 0xc240, 0xda00, 0xf00c, 0x1248, 0x1003, 0x1301, 0x04cb, 0x7261, 0x2108, 0x0081, 0x2009, 0x0080, 0x1a48, 0x10c0,
0xdd30, 0x1248, 0x100b, 0xc300, 0x0be7, 0x90c4, 0x2102, 0x0003, 0x238c, 0x8fc3, 0xf6c7, 0xdaff, 0x1a05, 0x1082, 0xc241, 0xf005, 0x7a6f, 0xc241, 0x1a05, 0x10c2, 0x2000, 0x8040, 0xda00, 0x20c0, 0x0064,
0xdd60, 0x781c, 0xc042, 0x1c0e, 0x3082, 0x1a48, 0x13c0, 0x7548, 0x7348, 0x7148, 0x7648, 0xf002, 0x7608, 0x1248, 0x1000, 0x1400, 0x300b, 0x084d, 0x02c5, 0x1248, 0x1000, 0xe101, 0x1001, 0x04cb, 0x1a48,
0xdd90, 0x1000, 0x7361, 0x1408, 0x300b, 0x2302, 0x02c0, 0x780d, 0x2607, 0x903e, 0x07d6, 0xffe3, 0x792f, 0x09cf, 0x8152, 0x1248, 0x100e, 0x2400, 0x334b, 0xe501, 0x7ee2, 0x0dbf, 0x90f2, 0x1b0c, 0x1382,
0xddc0, 0xc123, 0x140e, 0x3080, 0x7822, 0x1a07, 0x1002, 0x124c, 0x1000, 0x120b, 0x1081, 0x1207, 0x1083, 0x2142, 0x004b, 0x781b, 0x0b21, 0x02e2, 0x1a4c, 0x1000, 0xe101, 0x0915, 0x00c2, 0xc101, 0x1204,
0xddf0, 0x1083, 0x090d, 0x00c2, 0xe001, 0x1a4c, 0x1000, 0x1a06, 0x1002, 0x234a, 0x1000, 0x7169, 0xf008, 0x2053, 0x0003, 0x6179, 0x781c, 0x2340, 0x104b, 0x1203, 0x1083, 0x0bf1, 0x90c2, 0x1202, 0x1080,
0xde20, 0x091d, 0x0004, 0x70cf, 0xffff, 0xc644, 0x881b, 0xe0b2, 0xd83c, 0x20ca, 0x0ca2, 0x1a01, 0x1002, 0x1a4c, 0x1080, 0x02b9, 0x05a0, 0xc0a4, 0x78e0, 0xc0f1, 0xff95, 0xd800, 0x71cf, 0xff00, 0x1fe0,
0xde50, 0x19d0, 0x001c, 0x19d1, 0x001c, 0x70cf, 0xffff, 0xe058, 0x901f, 0xb861, 0x19d2, 0x001c, 0xc0d1, 0x7ee0, 0x78e0, 0xc0f1, 0x0a7a, 0x0580, 0x70cf, 0xffff, 0xc5d4, 0x9041, 0x9023, 0x75cf, 0xffff,
0xde80, 0xe058, 0x7942, 0xb967, 0x7f30, 0xb53f, 0x71cf, 0xffff, 0xc84c, 0x91d3, 0x108b, 0x0081, 0x2615, 0x1380, 0x090f, 0x0c91, 0x0a8e, 0x05a0, 0xd906, 0x7e10, 0x2615, 0x1380, 0x0a82, 0x05a0, 0xd960,
0xdeb0, 0x790f, 0x090d, 0x0133, 0xad0c, 0xd904, 0xad2c, 0x79ec, 0x2941, 0x7402, 0x71cf, 0xff00, 0x2184, 0xb142, 0x1906, 0x0e44, 0xffde, 0x70c9, 0x0a5a, 0x05a0, 0x8d2c, 0xad0b, 0xd800, 0xad01, 0x0219,
0xdee0, 0x05a0, 0xa513, 0xc0f1, 0x71cf, 0xffff, 0xc644, 0xa91b, 0xd902, 0x70cf, 0xffff, 0xc84c, 0x093e, 0x03a0, 0xa826, 0xffdc, 0xf1b5, 0xc0f1, 0x09ea, 0x0580, 0x75cf, 0xffff, 0xe058, 0x1540, 0x1080,
0xdf10, 0x08a7, 0x0010, 0x8d00, 0x0813, 0x009e, 0x1540, 0x1081, 0xe181, 0x20ca, 0x00a1, 0xf24b, 0x1540, 0x1081, 0x090f, 0x0050, 0x1540, 0x1081, 0x0927, 0x0091, 0x1550, 0x1081, 0xde00, 0xad2a, 0x1d50,
0xdf40, 0x1382, 0x1552, 0x1101, 0x1d52, 0x1384, 0xb524, 0x082d, 0x015f, 0xff55, 0xd803, 0xf033, 0x1540, 0x1081, 0x0967, 0x00d1, 0x1550, 0x1081, 0xde00, 0xad2a, 0x1d50, 0x1382, 0x1552, 0x1101, 0x1d52,
0xdf70, 0x1384, 0xb524, 0x0811, 0x019e, 0xb8a0, 0xad00, 0xff47, 0x1d40, 0x1382, 0xf01f, 0xff5a, 0x8d01, 0x8d40, 0xe812, 0x71cf, 0xffff, 0xc644, 0x893b, 0x7030, 0x22d1, 0x8062, 0xf20a, 0x0a0f, 0x009e,
0xdfa0, 0x71cf, 0xffff, 0xc84c, 0x893b, 0xe902, 0xffcf, 0x8d00, 0xb8e7, 0x26ca, 0x1022, 0xf5e2, 0xff3c, 0xd801, 0x1d40, 0x1002, 0x0141, 0x0580, 0x78e0, 0xc0f1, 0xc5e1, 0xff34, 0xdd00, 0x70cf, 0xffff,
0xdfd0, 0xe090, 0xa8a8, 0xd800, 0xb893, 0x0c8a, 0x0460, 0xd901, 0x71cf, 0xffff, 0xdc10, 0xd813, 0x0b96, 0x0460, 0x72a9, 0x0119, 0x0580, 0xc0f1, 0x71cf, 0x0000, 0x5bae, 0x7940, 0xff9d, 0xf135, 0x78e0,
0xe000, 0xc0f1, 0x70cf, 0x0000, 0x5cba, 0x7840, 0x70cf, 0xffff, 0xe058, 0x8800, 0x0815, 0x001e, 0x70cf, 0xffff, 0xc84c, 0x881a, 0xe080, 0x0ee0, 0xffc1, 0xf121, 0x78e0, 0xc0f1, 0xd900, 0xf009, 0x70cf,
0xe030, 0xffff, 0xe0ac, 0x7835, 0x8041, 0x8000, 0xe102, 0xa040, 0x09f3, 0x8114, 0x71cf, 0xffff, 0xe058, 0x70cf, 0xffff, 0xc594, 0xb030, 0xffdd, 0xd800, 0xf109, 0x0000, 0x0300, 0x0204, 0x0700, 0x0000,
0xe060, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
0xe090, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xffff, 0xcb68, 0xffff, 0xdff0, 0xffff, 0xcb6c, 0xffff, 0xe000
};

static struct msm_camera_i2c_reg_conf mi1040_720p_settings[] = {
	{0xdc00, 0x50, MSM_CAMERA_I2C_BYTE_DATA, MSM_CAMERA_I2C_CMD_WRITE},
	{MI1040_COMMAND_REGISTER, MI1040_COMMAND_REGISTER_SET_STATE,
		MSM_CAMERA_I2C_UNSET_WORD_MASK, MSM_CAMERA_I2C_CMD_POLL},
	{MI1040_COMMAND_REGISTER, (MI1040_COMMAND_REGISTER_OK |
		MI1040_COMMAND_REGISTER_SET_STATE), MSM_CAMERA_I2C_WORD_DATA,
		MSM_CAMERA_I2C_CMD_WRITE},
	{MI1040_COMMAND_REGISTER, MI1040_COMMAND_REGISTER_SET_STATE,
		MSM_CAMERA_I2C_UNSET_WORD_MASK, MSM_CAMERA_I2C_CMD_POLL},
	{0xDC01, 0x52, MSM_CAMERA_I2C_BYTE_DATA, MSM_CAMERA_I2C_CMD_POLL},

	{0x098E, 0, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC868, 0x0500,},/*output_width = 1280*/
	{0xC86A, 0x02D0,},/*output_height = 720*/
	{0xC85C, 0x03, MSM_CAMERA_I2C_BYTE_DATA},  /*aet_aemode = 0*/
	{0xC854, 0x0000,},// CAM_CROP_WINDOW_XOFFSET
	{0xC856, 0x0078,},// CAM_CROP_WINDOW_YOFFSET
	{0xC858, 0x0500,},// CAM_CROP_WINDOW_WIDTH
	{0xC85A, 0x02D0,},// CAM_CROP_WINDOW_HEIGHT
	{0xC88C, 0x1E02,},// CAM_AET_MAX_FRAME_RATE
	{0xC88E, 0x0F00,},// CAM_AET_MIN_FRAME_RATE
	{0xC914, 0x0000,},// CAM_STAT_AWB_CLIP_WINDOW_XSTART
	{0xC916, 0x0000,},// CAM_STAT_AWB_CLIP_WINDOW_YSTART
	{0xC918, 0x04FF,},// CAM_STAT_AWB_CLIP_WINDOW_XEND
	{0xC91A, 0x02CF,},// CAM_STAT_AWB_CLIP_WINDOW_YEND
	{0xC91C, 0x0000,},// CAM_STAT_AE_INITIAL_WINDOW_XSTART
	{0xC91E, 0x0000,},// CAM_STAT_AE_INITIAL_WINDOW_YSTART
	{0xC920, 0x00FF,},// CAM_STAT_AE_INITIAL_WINDOW_XEND
	{0xC922, 0x008F,},// CAM_STAT_AE_INITIAL_WINDOW_YEND
};

static struct msm_camera_i2c_reg_conf mi1040_960p_settings[] = {
	{0xdc00, 0x50, MSM_CAMERA_I2C_BYTE_DATA, MSM_CAMERA_I2C_CMD_WRITE},
	{MI1040_COMMAND_REGISTER, MI1040_COMMAND_REGISTER_SET_STATE,
		MSM_CAMERA_I2C_UNSET_WORD_MASK, MSM_CAMERA_I2C_CMD_POLL},
	{MI1040_COMMAND_REGISTER, (MI1040_COMMAND_REGISTER_OK |
		MI1040_COMMAND_REGISTER_SET_STATE), MSM_CAMERA_I2C_WORD_DATA,
		MSM_CAMERA_I2C_CMD_WRITE},
	{MI1040_COMMAND_REGISTER, MI1040_COMMAND_REGISTER_SET_STATE,
		MSM_CAMERA_I2C_UNSET_WORD_MASK, MSM_CAMERA_I2C_CMD_POLL},
	{0xDC01, 0x52, MSM_CAMERA_I2C_BYTE_DATA, MSM_CAMERA_I2C_CMD_POLL},

	{0x098E, 0, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC868, 0x0500,},/*output_width = 1280*/
	{0xC86A, 0x03C0,},/*output_height = 960*/
	{0xC85C, 0x03, MSM_CAMERA_I2C_BYTE_DATA},  /*aet_aemode = 0*/
	{0xC854, 0x0000,},// CAM_CROP_WINDOW_XOFFSET
	{0xC856, 0x0000,},// CAM_CROP_WINDOW_YOFFSET
	{0xC858, 0x0500,},// CAM_CROP_WINDOW_WIDTH
	{0xC85A, 0x03C0,},// CAM_CROP_WINDOW_HEIGHT
	{0xC88C, 0x1E02,},// CAM_AET_MAX_FRAME_RATE
	{0xC88E, 0x0F00,},// CAM_AET_MIN_FRAME_RATE
	{0xC914, 0x0000,},// CAM_STAT_AWB_CLIP_WINDOW_XSTART
	{0xC916, 0x0000,},// CAM_STAT_AWB_CLIP_WINDOW_YSTART
	{0xC918, 0x04FF,},// CAM_STAT_AWB_CLIP_WINDOW_XEND
	{0xC91A, 0x03BF,},// CAM_STAT_AWB_CLIP_WINDOW_YEND
	{0xC91C, 0x0000,},// CAM_STAT_AE_INITIAL_WINDOW_XSTART
	{0xC91E, 0x0000,},// CAM_STAT_AE_INITIAL_WINDOW_YSTART
	{0xC920, 0x00FF,},// CAM_STAT_AE_INITIAL_WINDOW_XEND
	{0xC922, 0x00BF,},// CAM_STAT_AE_INITIAL_WINDOW_YEND
};

static struct msm_camera_i2c_reg_conf mi1040_recommend_settings[] = {
	{0x301A, 0x0234, MSM_CAMERA_I2C_SET_WORD_MASK},
	{0x098E, 0, MSM_CAMERA_I2C_BYTE_DATA},
	/*cam_sysctl_pll_enable = 1*/
	{0xC97E, 0x01, MSM_CAMERA_I2C_BYTE_DATA},
	/*cam_sysctl_pll_divider_m_n = 288*/
	{0xC980, 0x0120,},
	/*cam_sysctl_pll_divider_p = 1792*/
	{0xC982, 0x0700,},

	{ 0x98E, 0x071D ,},	// set XDMA to logical addressing
	{ 0xC800, 0x0004,}, 	// CAM_SENSOR_CFG_Y_ADDR_START
	{ 0xC802, 0x0004,}, 	// CAM_SENSOR_CFG_X_ADDR_START
	{ 0xC804, 0x03CB,}, 	// CAM_SENSOR_CFG_Y_ADDR_END
	{ 0xC806, 0x050B,}, 	// CAM_SENSOR_CFG_X_ADDR_END
	// data length is double words
	{0xC808, 0x02DC,},/*pixclk = 48000000*/
	{0xC80A, 0x6C00,},/*pixclk = 48000000*/
	//
	{ 0xC80C, 0x0001,}, 	// CAM_SENSOR_CFG_ROW_SPEED
	{ 0xC80E, 0x00DB,}, 	// CAM_SENSOR_CFG_FINE_INTEG_TIME_MIN
	{ 0xC810, 0x05B3,}, 	// CAM_SENSOR_CFG_FINE_INTEG_TIME_MAX
	{ 0xC812, 0x03EE,}, 	// CAM_SENSOR_CFG_FRAME_LENGTH_LINES
	{ 0xC814, 0x0636,}, 	// CAM_SENSOR_CFG_LINE_LENGTH_PCK
	{ 0xC816, 0x0060,}, 	// CAM_SENSOR_CFG_FINE_CORRECTION
	{ 0xC818, 0x03C3,}, 	// CAM_SENSOR_CFG_CPIPE_LAST_ROW
	{ 0xC834, 0x0000,}, 	// CAM_SENSOR_CONTROL_READ_MODE
	{ 0xC854, 0x0000,}, 	// CAM_CROP_WINDOW_XOFFSET
	{ 0xC856, 0x0000,}, 	// CAM_CROP_WINDOW_YOFFSET
	{ 0xC858, 0x0500,}, 	// CAM_CROP_WINDOW_WIDTH
	{ 0xC85A, 0x03C0,}, 	// CAM_CROP_WINDOW_HEIGHT
	{ 0xC85C, 0x03, MSM_CAMERA_I2C_BYTE_DATA,}, 	// CAM_CROP_CROPMODE
	{ 0xC868, 0x0500,}, 	// CAM_OUTPUT_WIDTH
	{ 0xC86A, 0x03C0,}, 	// CAM_OUTPUT_HEIGHT
	{ 0xC88C, 0x1E02,}, 	// CAM_AET_MAX_FRAME_RATE
	// Changed to varied frame rate to 7.5~30fps
	{ 0xC88E, 0x0780,}, 	// CAM_AET_MIN_FRAME_RATE
	//
	{ 0xC914, 0x0000,}, 	// CAM_STAT_AWB_CLIP_WINDOW_XSTART
	{ 0xC916, 0x0000,}, 	// CAM_STAT_AWB_CLIP_WINDOW_YSTART
	{ 0xC918, 0x04FF,}, 	// CAM_STAT_AWB_CLIP_WINDOW_XEND
	{ 0xC91A, 0x03BF,}, 	// CAM_STAT_AWB_CLIP_WINDOW_YEND
	{ 0xC91C, 0x0000,}, 	// CAM_STAT_AE_INITIAL_WINDOW_XSTART
	{ 0xC91E, 0x0000,}, 	// CAM_STAT_AE_INITIAL_WINDOW_YSTART
	{ 0xC920, 0x00FF,}, 	// CAM_STAT_AE_INITIAL_WINDOW_XEND
	{ 0xC922, 0x00BF,}, 	// CAM_STAT_AE_INITIAL_WINDOW_YEND
	{ 0xE801, 0x00, MSM_CAMERA_I2C_BYTE_DATA ,},	// AUTO_BINNING_MODE

	/*Sensor optimization*/
	{0x316A, 0x8270,},
	{0x316C, 0x8270,},
	{0x3ED0, 0x2305,},
	{0x3ED2, 0x77CF,},
	{0x316E, 0x8202,},
	{0x3180, 0x87FF,},
	{0x30D4, 0x6080,},
	{0xA802, 0x0008,},
	{0x3E14, 0xFF39,},

	// PGA parameter and APGA
	{ 0x098E, 0x495E,}, 	// LOGICAL_ADDRESS_ACCESS [CAM_PGA_PGA_CONTROL]
	{ 0xC95E, 0x0002,}, 	// CAM_PGA_PGA_CONTROL
	{ 0x3640, 0x0230,}, 	//  P_G1_P0Q0
	{ 0x3642, 0xEF67,}, 	//  P_G1_P0Q1
	{ 0x3644, 0x6590,}, 	//  P_G1_P0Q2
	{ 0x3646, 0x178D,}, 	//  P_G1_P0Q3
	{ 0x3648, 0x2031,}, 	//  P_G1_P0Q4
	{ 0x364A, 0x00B0,}, 	//  P_R_P0Q0
	{ 0x364C, 0xDF48,}, 	//  P_R_P0Q1
	{ 0x364E, 0x11B1,}, 	//  P_R_P0Q2
	{ 0x3650, 0x2E6E,}, 	//  P_R_P0Q3
	{ 0x3652, 0x37B1,}, 	//  P_R_P0Q4
	{ 0x3654, 0x00F0,}, 	//  P_B_P0Q0
	{ 0x3656, 0x67AB,}, 	//  P_B_P0Q1
	{ 0x3658, 0x37B0,}, 	//  P_B_P0Q2
	{ 0x365A, 0x778D,}, 	//  P_B_P0Q3
	{ 0x365C, 0x7FB0,}, 	//  P_B_P0Q4
	{ 0x365E, 0x00D0,}, 	//  P_G2_P0Q0
	{ 0x3660, 0xA8C9,}, 	//  P_G2_P0Q1
	{ 0x3662, 0x7270,}, 	//  P_G2_P0Q2
	{ 0x3664, 0x04ED,}, 	//  P_G2_P0Q3
	{ 0x3666, 0x14F1,}, 	//  P_G2_P0Q4
	{ 0x3680, 0x54EB,}, 	//  P_G1_P1Q0
	{ 0x3682, 0x244C,}, 	//  P_G1_P1Q1
	{ 0x3684, 0x47AE,}, 	//  P_G1_P1Q2
	{ 0x3686, 0x34CC,}, 	//  P_G1_P1Q3
	{ 0x3688, 0xD4CC,}, 	//  P_G1_P1Q4
	{ 0x368A, 0x1F0B,}, 	//  P_R_P1Q0
	{ 0x368C, 0x6CEC,}, 	//  P_R_P1Q1
	{ 0x368E, 0x67CE,}, 	//  P_R_P1Q2
	{ 0x3690, 0x282A,}, 	//  P_R_P1Q3
	{ 0x3692, 0xCDAD,}, 	//  P_R_P1Q4
	{ 0x3694, 0x200C,}, 	//  P_B_P1Q0
	{ 0x3696, 0x858A,}, 	//  P_B_P1Q1
	{ 0x3698, 0x124C,}, 	//  P_B_P1Q2
	{ 0x369A, 0xD46D,}, 	//  P_B_P1Q3
	{ 0x369C, 0xA60F,}, 	//  P_B_P1Q4
	{ 0x369E, 0x152C,}, 	//  P_G2_P1Q0
	{ 0x36A0, 0x0DAB,}, 	//  P_G2_P1Q1
	{ 0x36A2, 0xA96A,}, 	//  P_G2_P1Q2
	{ 0x36A4, 0xB26C,}, 	//  P_G2_P1Q3
	{ 0x36A6, 0xA40F,}, 	//  P_G2_P1Q4
	{ 0x36C0, 0x77D0,}, 	//  P_G1_P2Q0
	{ 0x36C2, 0x2E8B,}, 	//  P_G1_P2Q1
	{ 0x36C4, 0x7E92,}, 	//  P_G1_P2Q2
	{ 0x36C6, 0x4C4E,}, 	//  P_G1_P2Q3
	{ 0x36C8, 0xEA71,}, 	//  P_G1_P2Q4
	{ 0x36CA, 0x12F1,}, 	//  P_R_P2Q0
	{ 0x36CC, 0x1F8D,}, 	//  P_R_P2Q1
	{ 0x36CE, 0x2013,}, 	//  P_R_P2Q2
	{ 0x36D0, 0x1BB0,}, 	//  P_R_P2Q3
	{ 0x36D2, 0xE072,}, 	//  P_R_P2Q4
	{ 0x36D4, 0x4010,}, 	//  P_B_P2Q0
	{ 0x36D6, 0x060A,}, 	//  P_B_P2Q1
	{ 0x36D8, 0x6F92,}, 	//  P_B_P2Q2
	{ 0x36DA, 0x2CCF,}, 	//  P_B_P2Q3
	{ 0x36DC, 0x9F32,}, 	//  P_B_P2Q4
	{ 0x36DE, 0x6FD0,}, 	//  P_G2_P2Q0
	{ 0x36E0, 0x0B6C,}, 	//  P_G2_P2Q1
	{ 0x36E2, 0x7672,}, 	//  P_G2_P2Q2
	{ 0x36E4, 0x10CE,}, 	//  P_G2_P2Q3
	{ 0x36E6, 0x8092,}, 	//  P_G2_P2Q4
	{ 0x3700, 0x662A,}, 	//  P_G1_P3Q0
	{ 0x3702, 0xA44C,}, 	//  P_G1_P3Q1
	{ 0x3704, 0xFA6F,}, 	//  P_G1_P3Q2
	{ 0x3706, 0xA10F,}, 	//  P_G1_P3Q3
	{ 0x3708, 0x3690,}, 	//  P_G1_P3Q4
	{ 0x370A, 0x696C,}, 	//  P_R_P3Q0
	{ 0x370C, 0xA5CE,}, 	//  P_R_P3Q1
	{ 0x370E, 0xCC4E,}, 	//  P_R_P3Q2
	{ 0x3710, 0xD989,}, 	//  P_R_P3Q3
	{ 0x3712, 0x534F,}, 	//  P_R_P3Q4
	{ 0x3714, 0x68AD,}, 	//  P_B_P3Q0
	{ 0x3716, 0xE48C,}, 	//  P_B_P3Q1
	{ 0x3718, 0xBE90,}, 	//  P_B_P3Q2
	{ 0x371A, 0x0AEF,}, 	//  P_B_P3Q3
	{ 0x371C, 0x4012,}, 	//  P_B_P3Q4
	{ 0x371E, 0x20ED,}, 	//  P_G2_P3Q0
	{ 0x3720, 0xF76D,}, 	//  P_G2_P3Q1
	{ 0x3722, 0xBD91,}, 	//  P_G2_P3Q2
	{ 0x3724, 0x186E,}, 	//  P_G2_P3Q3
	{ 0x3726, 0x0C73,}, 	//  P_G2_P3Q4
	{ 0x3740, 0x0E51,}, 	//  P_G1_P4Q0
	{ 0x3742, 0x31EF,}, 	//  P_G1_P4Q1
	{ 0x3744, 0x5E73,}, 	//  P_G1_P4Q2
	{ 0x3746, 0x8D70,}, 	//  P_G1_P4Q3
	{ 0x3748, 0xEC56,}, 	//  P_G1_P4Q4
	{ 0x374A, 0x23B1,}, 	//  P_R_P4Q0
	{ 0x374C, 0x2290,}, 	//  P_R_P4Q1
	{ 0x374E, 0x79B3,},	//  P_R_P4Q2
	{ 0x3750, 0xE5F2,}, 	//  P_R_P4Q3
	{ 0x3752, 0x9057,}, 	//  P_R_P4Q4
	{ 0x3754, 0x6550,}, 	//  P_B_P4Q0
	{ 0x3756, 0x1470,}, 	//  P_B_P4Q1
	{ 0x3758, 0x0673,}, 	//  P_B_P4Q2
	{ 0x375A, 0x92F2,}, 	//  P_B_P4Q3
	{ 0x375C, 0xAF76,}, 	//  P_B_P4Q4
	{ 0x375E, 0x1031,}, 	//  P_G2_P4Q0
	{ 0x3760, 0x062F,}, 	//  P_G2_P4Q1
	{ 0x3762, 0x72B3,}, 	//  P_G2_P4Q2
	{ 0x3764, 0x15CC,}, 	//  P_G2_P4Q3
	{ 0x3766, 0xEBB6,}, 	//  P_G2_P4Q4
	{ 0x3784, 0x026C,}, 	//  CENTER_COLUMN
	{ 0x3782, 0x01CC,}, 	//  CENTER_ROW
	{ 0x37C0, 0xC00A,}, 	//  P_GR_Q5
	{ 0x37C2, 0xE7E9,}, 	//  P_RD_Q5
	{ 0x37C4, 0x806A,}, 	//  P_BL_Q5
	{ 0x37C6, 0xDAE9,}, 	//  P_GB_Q5
	{ 0x098E, 0x0000,}, 	//  LOGICAL addressing

	// -- 0518, Alias
	{ 0xC960, 0x0A8C,}, 	// CAM_PGA_L_CONFIG_COLOUR_TEMP
	{ 0xC962, 0x7DFC,}, 	// CAM_PGA_L_CONFIG_GREEN_RED_Q14
	{ 0xC964, 0x67A8,}, 	// CAM_PGA_L_CONFIG_RED_Q14
	{ 0xC966, 0x7E28,}, 	// CAM_PGA_L_CONFIG_GREEN_BLUE_Q14
	{ 0xC968, 0x7E42,}, 	// CAM_PGA_L_CONFIG_BLUE_Q14
	//
	{ 0xC96A, 0x0FA0,}, 	// CAM_PGA_M_CONFIG_COLOUR_TEMP
	{ 0xC96C, 0x7E80,}, 	// CAM_PGA_M_CONFIG_GREEN_RED_Q14
	{ 0xC96E, 0x8798,}, 	// CAM_PGA_M_CONFIG_RED_Q14
	{ 0xC970, 0x7F25,}, 	// CAM_PGA_M_CONFIG_GREEN_BLUE_Q14
	{ 0xC972, 0x7EFF,}, 	// CAM_PGA_M_CONFIG_BLUE_Q14
	{ 0xC974, 0x1964,},	// CAM_PGA_R_CONFIG_COLOUR_TEMP
	{ 0xC976, 0x7EF8,}, 	// CAM_PGA_R_CONFIG_GREEN_RED_Q14
	{ 0xC978, 0x78E4,}, 	// CAM_PGA_R_CONFIG_RED_Q14
	{ 0xC97A, 0x80EC,}, 	// CAM_PGA_R_CONFIG_GREEN_BLUE_Q14
	{ 0xC97C, 0x79AE,}, 	// CAM_PGA_R_CONFIG_BLUE_Q14
	{ 0xC95E, 0x0003,}, 	// CAM_PGA_PGA_CONTROL


	/*[CCM]*/
	{0xC892, 0x0267,},/*CAM_AWB_CCM_L_0*/
	{0xC894, 0xFF1A,},/*CAM_AWB_CCM_L_1*/
	{0xC896, 0xFFB3,},/*CAM_AWB_CCM_L_2*/
	{0xC898, 0xFF80,},/*CAM_AWB_CCM_L_3*/
	{0xC89A, 0x0166,},/*CAM_AWB_CCM_L_4*/
	{0xC89C, 0x0003,},/*CAM_AWB_CCM_L_5*/
	{0xC89E, 0xFF9A,},/*CAM_AWB_CCM_L_6*/
	{0xC8A0, 0xFEB4,},/*CAM_AWB_CCM_L_7*/
	{0xC8A2, 0x024D,},/*CAM_AWB_CCM_L_8*/
	{0xC8A4, 0x01BF,},/*CAM_AWB_CCM_M_0*/
	{0xC8A6, 0xFF01,},/*CAM_AWB_CCM_M_1*/
	{0xC8A8, 0xFFF3,},/*CAM_AWB_CCM_M_2*/
	{0xC8AA, 0xFF75,},/*CAM_AWB_CCM_M_3*/
	{0xC8AC, 0x0198,},/*CAM_AWB_CCM_M_4*/
	{0xC8AE, 0xFFFD,},/*CAM_AWB_CCM_M_5*/
	{0xC8B0, 0xFF9A,},/*CAM_AWB_CCM_M_6*/
	{0xC8B2, 0xFEE7,},/*CAM_AWB_CCM_M_7*/
	{0xC8B4, 0x02A8,},/*CAM_AWB_CCM_M_8*/

	{0xC8B6, 0x0100,},/*CAM_AWB_CCM_R_0*/
	{0xC8B8, 0xFF72,},/*CAM_AWB_CCM_R_1*/
	{0xC8BA, 0xFFF8,},/*CAM_AWB_CCM_R_2*/
	{0xC8BC, 0xFFA8,},/*CAM_AWB_CCM_R_3*/
	{0xC8BE, 0x011D,},/*CAM_AWB_CCM_R_4*/
	{0xC8C0, 0xFFFB,},/*CAM_AWB_CCM_R_5*/
	{0xC8C2, 0xFFE3,},/*CAM_AWB_CCM_R_6*/
	{0xC8C4, 0xFE48,},/*CAM_AWB_CCM_R_7*/
	{0xC8C6, 0x0356,},/*CAM_AWB_CCM_R_8*/

	{0xC8DA, 0x004D,},// CAM_AWB_LL_CCM_0
	{0xC8DC, 0x0096,},// CAM_AWB_LL_CCM_1
	{0xC8DE, 0x001D,},// CAM_AWB_LL_CCM_2
	{0xC8E0, 0x004D,},// CAM_AWB_LL_CCM_3
	{0xC8E2, 0x0096,},// CAM_AWB_LL_CCM_4
	{0xC8E4, 0x001D,},// CAM_AWB_LL_CCM_5
	{0xC8E6, 0x004D,},// CAM_AWB_LL_CCM_6
	{0xC8E8, 0x0096,},// CAM_AWB_LL_CCM_7
	{0xC8EA, 0x001D,},// CAM_AWB_LL_CCM_8
	{0xC8C8, 0x0075,},// CAM_AWB_CCM_L_RG_GAIN
	{0xC8CA, 0x011C,},// CAM_AWB_CCM_L_BG_GAIN
	{0xC8CC, 0x009A,},// CAM_AWB_CCM_M_RG_GAIN
	{0xC8CE, 0x0105,},// CAM_AWB_CCM_M_BG_GAIN
	{0xC8D0, 0x00A4,},// CAM_AWB_CCM_R_RG_GAIN
	{0xC8D2, 0x00AC,},// CAM_AWB_CCM_R_BG_GAIN
	{0xC8D4, 0x0A8C,},// CAM_AWB_CCM_L_CTEMP
	{0xC8D6, 0x0F0A,},// CAM_AWB_CCM_M_CTEMP
	{0xC8D8, 0x1964,},// CAM_AWB_CCM_R_CTEMP

	/*[AWB]*/
	{0xC914, 0x0000,},/*CAM_STAT_AWB_CLIP_WINDOW_XSTART*/
	{0xC916, 0x0000,},/*CAM_STAT_AWB_CLIP_WINDOW_YSTART*/
	{0xC918, 0x04FF,},/*CAM_STAT_AWB_CLIP_WINDOW_XEND*/
	{0xC91A, 0x03BF,},/*CAM_STAT_AWB_CLIP_WINDOW_YEND*/
	{0xC8F2, 0x04, MSM_CAMERA_I2C_BYTE_DATA},/*CAM_AWB_AWB_XSCALE*/
	{0xC8F3, 0x02, MSM_CAMERA_I2C_BYTE_DATA},/*CAM_AWB_AWB_YSCALE*/
	{0xC904, 0x0036,},/*CAM_AWB_AWB_YSHIFT_PRE_ADJ*/
	{0xC906, 0x0040,},/*CAM_AWB_AWB_YSHIFT_PRE_ADJ*/
	{0xC8F4, 0x0000,},// CAM_AWB_AWB_WEIGHTS_0
	{0xC8F6, 0x0000,},// CAM_AWB_AWB_WEIGHTS_1
	{0xC8F8, 0x0000,},// CAM_AWB_AWB_WEIGHTS_2
	{0xC8FA, 0xE724,},// CAM_AWB_AWB_WEIGHTS_3
	{0xC8FC, 0x1583,},// CAM_AWB_AWB_WEIGHTS_4
	{0xC8FE, 0x2045,},// CAM_AWB_AWB_WEIGHTS_5
	{0xC900, 0x05DC,},// CAM_AWB_AWB_WEIGHTS_6
	{0xC902, 0x007C,},// CAM_AWB_AWB_WEIGHTS_7
	{0xC90A, 0x1388,},// CAM_AWB_TINTS_CTEMP_THRESHOLD
	{0xC90C, 0x80, MSM_CAMERA_I2C_BYTE_DATA},// CAM_AWB_K_R_L
	{0xC90D, 0x90, MSM_CAMERA_I2C_BYTE_DATA},// CAM_AWB_K_G_L
	{0xC90E, 0x90, MSM_CAMERA_I2C_BYTE_DATA},// CAM_AWB_K_B_L
	{0xC90F, 0x80, MSM_CAMERA_I2C_BYTE_DATA},// CAM_AWB_K_R_R
	{0xC910, 0x88, MSM_CAMERA_I2C_BYTE_DATA},// CAM_AWB_K_G_R
	{0xC911, 0x80, MSM_CAMERA_I2C_BYTE_DATA},// CAM_AWB_K_B_R
	{0x098E, 0xAC06,}, // LOGICAL_ADDRESS_ACCESS [AWB_R_RATIO_LOWER]
	{0xAC06, 0x62, MSM_CAMERA_I2C_BYTE_DATA},// AWB_R_RATIO_LOWER
	{0xAC07, 0x66, MSM_CAMERA_I2C_BYTE_DATA},// AWB_R_RATIO_UPPER
	{0xAC08, 0x64, MSM_CAMERA_I2C_BYTE_DATA},// AWB_B_RATIO_LOWER
	{0xAC09, 0x66, MSM_CAMERA_I2C_BYTE_DATA},// AWB_B_RATIO_UPPER

	/*[Step7-CPIPE_Preference]*/
	{0xC926, 0x0060,},// CAM_LL_START_BRIGHTNESS
	{0xC928, 0x009A,},// CAM_LL_STOP_BRIGHTNESS
	{0xC946, 0x0070,},// CAM_LL_START_GAIN_METRIC
	{0xC948, 0x00F3,},// CAM_LL_STOP_GAIN_METRIC
	{0xC952, 0x0060,},// CAM_LL_START_TARGET_LUMA_BM
	{0xC954, 0x009A,},// CAM_LL_STOP_TARGET_LUMA_BM
	{0xC95A, 0x04, MSM_CAMERA_I2C_BYTE_DATA},// CAM_SEQ_UV_COLOR_BOOST
	{0xC92A, 0x5A, MSM_CAMERA_I2C_BYTE_DATA},// CAM_LL_START_SATURATION
	{0xC92B, 0x3C, MSM_CAMERA_I2C_BYTE_DATA},// CAM_LL_END_SATURATION
	{0xC92C, 0x00, MSM_CAMERA_I2C_BYTE_DATA},// CAM_LL_START_DESATURATION
	{0xC92D, 0xFF, MSM_CAMERA_I2C_BYTE_DATA},// CAM_LL_END_DESATURATION
	{0xC92E, 0x32, MSM_CAMERA_I2C_BYTE_DATA},   // CAM_LL_START_DEMOSAIC
	{0xC92F, 0x04, MSM_CAMERA_I2C_BYTE_DATA},   // CAM_LL_START_AP_GAIN
	{0xC930, 0x02, MSM_CAMERA_I2C_BYTE_DATA},   // CAM_LL_START_AP_THRESH
	{0xC931, 0x78, MSM_CAMERA_I2C_BYTE_DATA},   // CAM_LL_STOP_DEMOSAIC
	{0xC932, 0x01, MSM_CAMERA_I2C_BYTE_DATA},   // CAM_LL_STOP_AP_GAIN
	{0xC933, 0x09, MSM_CAMERA_I2C_BYTE_DATA},   // CAM_LL_STOP_AP_THRESH
	{0xC934, 0x32, MSM_CAMERA_I2C_BYTE_DATA},   // CAM_LL_START_NR_RED
	{0xC935, 0x14, MSM_CAMERA_I2C_BYTE_DATA},   // CAM_LL_START_NR_GREEN
	{0xC936, 0x32, MSM_CAMERA_I2C_BYTE_DATA},   // CAM_LL_START_NR_BLUE
	{0xC937, 0x14, MSM_CAMERA_I2C_BYTE_DATA},   // CAM_LL_START_NR_THRESH
	{0xC938, 0x50, MSM_CAMERA_I2C_BYTE_DATA},   // CAM_LL_STOP_NR_RED
	{0xC939, 0x50, MSM_CAMERA_I2C_BYTE_DATA},   // CAM_LL_STOP_NR_GREEN
	{0xC93A, 0x50, MSM_CAMERA_I2C_BYTE_DATA},  // CAM_LL_STOP_NR_BLUE
	{0xC93B, 0x50, MSM_CAMERA_I2C_BYTE_DATA},   // CAM_LL_STOP_NR_THRESH
	{0xC93C, 0x0005 ,},/*CAM_LL_START_CONTRAST_BM*/
	{0xC93E, 0x0358 ,},/*CAM_LL_STOP_CONTRAST_BM*/
	{0xC940, 0x00DC ,},/*CAM_LL_GAMMA*/
	/*CAM_LL_START_CONTRAST_GRADIENT*/
	{0xC942, 0x4B, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC943, 0x3C, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC944, 0x22, MSM_CAMERA_I2C_BYTE_DATA},/*CAM_LL_START_CONTRAST_LUMA*/
	{0xC945, 0x19, MSM_CAMERA_I2C_BYTE_DATA},/*CAM_LL_STOP_CONTRAST_LUMA*/
	{0x098E, 0xC942,},// LOGICAL_ADDRESS_ACCESS [CAM_LL_START_CONTRAST_GRADIENT]
	{0x098E, 0xC92A,},// LOGICAL_ADDRESS_ACCESS [CAM_LL_START_SATURATION]
	{0xC94A, 0x00F0,},// CAM_LL_START_FADE_TO_BLACK_LUMA
	{0xC94C, 0x0010,},// CAM_LL_STOP_FADE_TO_BLACK_LUMA
	{0xC94E, 0x01CD,},// CAM_LL_CLUSTER_DC_TH_BM
	{0xC950, 0x05, MSM_CAMERA_I2C_BYTE_DATA},// CAM_LL_CLUSTER_DC_GATE_PERCENTAGE
	{0xC951, 0x40, MSM_CAMERA_I2C_BYTE_DATA},// CAM_LL_SUMMING_SENSITIVITY_FACTOR
	{0xC87A, 0x3C, MSM_CAMERA_I2C_BYTE_DATA},// CAM_AET_TARGET_AVERAGE_LUMA
	{0xC87B, 0x1E, MSM_CAMERA_I2C_BYTE_DATA},// CAM_AET_TARGET_AVERAGE_LUMA_DARK
	{0xC878, 0x0E, MSM_CAMERA_I2C_BYTE_DATA},// CAM_AET_AEMODE  -- for DR test
	{0xC890, 0x0080 ,},/*CAM_AET_TARGET_GAIN*/
	{0xC882, 0x0100 ,},/*CAM_AET_AE_MAX_VIRT_AGAIN*/
	{0xC886, 0x0100 ,},/*CAM_AET_BLACK_CLIPPING_TARGET*/
	{0xA404, 0x02, MSM_CAMERA_I2C_BYTE_DATA},/*CCM_DELTA_GAIN*/
	/*AE_TRACK_AE_TRACKING_DAMPENING*/
	{0xC87C, 0x0016,}, // CAM_AET_BLACK_CLIPPING_TARGET
	{0xB42A, 0x05, MSM_CAMERA_I2C_BYTE_DATA}, // CCM_DELTA_GAIN
	{0xA80A, 0x10, MSM_CAMERA_I2C_BYTE_DATA}, // AE_TRACK_AE_TRACKING_DAMPENING_SPEED
	{0x326C, 0x0A08,},// APERTURE_PARAMETERS_2D

	/*Step8-Features*/
	{ 0x098E, 0x0000,}, 	// LOGICAL_ADDRESS_ACCESS
	{ 0xC984, 0x8040,}, 	// CAM_PORT_OUTPUT_CONTROL
	{ 0x001E, 0x0777,}, 	// PAD_SLEW

	// LOAD=MIPI setting for SOC1040
	{ 0xC984, 0x8041,}, 	// CAM_PORT_OUTPUT_CONTROL
	{ 0xC988, 0x0F00,}, 	// CAM_PORT_MIPI_TIMING_T_HS_ZERO
	{ 0xC98A, 0x0B07,}, 	// CAM_PORT_MIPI_TIMING_T_HS_EXIT_HS_TRAIL
	{ 0xC98C, 0x0D01,}, 	// CAM_PORT_MIPI_TIMING_T_CLK_POST_CLK_PRE
	{ 0xC98E, 0x071D,}, 	// CAM_PORT_MIPI_TIMING_T_CLK_TRAIL_CLK_ZERO
	{ 0xC990, 0x0006,}, 	// CAM_PORT_MIPI_TIMING_T_LPX
	{ 0xC992, 0x0A0C,}, 	// CAM_PORT_MIPI_TIMING_INIT_TIMING
	{ 0x3C5A, 0x0009,}, 	// MIPI_DELAY_TRIM

	// Improve AE Dampening speed
	//[Speed up AE/AWB]
	{ 0x098E, 0x2802,}, 	// LOGICAL_ADDRESS_ACCESS
	{ 0xA802, 0x0008,}, 	// AE_TRACK_MODE
	{ 0xC908, 0x01, MSM_CAMERA_I2C_BYTE_DATA}, 	// CAM_AWB_SKIP_FRAMES
	{ 0xC879, 0x01, MSM_CAMERA_I2C_BYTE_DATA}, 	// CAM_AET_SKIP_FRAMES
	{ 0xC909, 0x02, MSM_CAMERA_I2C_BYTE_DATA}, 	// CAM_AWB_AWBMODE
	{ 0xA80A, 0x18, MSM_CAMERA_I2C_BYTE_DATA}, 	// AE_TRACK_AE_TRACKING_DAMPENING_SPEED
	{ 0xA80B, 0x18, MSM_CAMERA_I2C_BYTE_DATA}, 	// AE_TRACK_AE_DAMPENING_SPEED
	{ 0xAC16, 0x18, MSM_CAMERA_I2C_BYTE_DATA}, 	// AWB_PRE_AWB_RATIOS_TRACKING_SPEED
	{ 0xC878, 0x0E, MSM_CAMERA_I2C_BYTE_DATA}, 	// CAM_AET_AEMODE

	{ 0x31E0, 0x01, MSM_CAMERA_I2C_BYTE_DATA}, 	// COLOR DOT

};

static void mi1040_stop_stream(struct msm_sensor_ctrl_t *s_ctrl) {}

static struct msm_camera_i2c_reg_conf mi1040_config_change_settings[] = {
	{0xdc00, 0x28, MSM_CAMERA_I2C_BYTE_DATA, MSM_CAMERA_I2C_CMD_WRITE},
	{MI1040_COMMAND_REGISTER, MI1040_COMMAND_REGISTER_SET_STATE,
		MSM_CAMERA_I2C_UNSET_WORD_MASK, MSM_CAMERA_I2C_CMD_POLL},
	{MI1040_COMMAND_REGISTER, (MI1040_COMMAND_REGISTER_OK |
		MI1040_COMMAND_REGISTER_SET_STATE), MSM_CAMERA_I2C_WORD_DATA,
		MSM_CAMERA_I2C_CMD_WRITE},
	{MI1040_COMMAND_REGISTER, MI1040_COMMAND_REGISTER_SET_STATE,
		MSM_CAMERA_I2C_UNSET_WORD_MASK, MSM_CAMERA_I2C_CMD_POLL},
	{0xDC01, 0x31, MSM_CAMERA_I2C_BYTE_DATA},
};

static int mi1040_saturation_enum_map[] = {
	MSM_V4L2_SATURATION_L0,
	MSM_V4L2_SATURATION_L1,
	MSM_V4L2_SATURATION_L2,
	MSM_V4L2_SATURATION_L3,
	MSM_V4L2_SATURATION_L4,
	MSM_V4L2_SATURATION_L5,
	MSM_V4L2_SATURATION_L6,
	MSM_V4L2_SATURATION_L7,
	MSM_V4L2_SATURATION_L8,
	MSM_V4L2_SATURATION_L9,
	MSM_V4L2_SATURATION_L10,
};

static struct msm_camera_i2c_reg_conf mi1040_saturation[][1] = {
	{{0xCC12, 0x00},},
	{{0xCC12, 0x1A},},
	{{0xCC12, 0x34},},
	{{0xCC12, 0x4E},},
	{{0xCC12, 0x68},},
	{{0xCC12, 0x80},},
	{{0xCC12, 0x9A},},
	{{0xCC12, 0xB4},},
	{{0xCC12, 0xCE},},
	{{0xCC12, 0xE8},},
	{{0xCC12, 0xFF},},
};

static struct msm_camera_i2c_reg_conf mi1040_refresh[] = {
	{MI1040_COMMAND_REGISTER, MI1040_COMMAND_REGISTER_REFRESH,
		MSM_CAMERA_I2C_UNSET_WORD_MASK, MSM_CAMERA_I2C_CMD_POLL},
	{MI1040_COMMAND_REGISTER, (MI1040_COMMAND_REGISTER_OK |
		MI1040_COMMAND_REGISTER_REFRESH), MSM_CAMERA_I2C_WORD_DATA,
		MSM_CAMERA_I2C_CMD_WRITE},
	{MI1040_COMMAND_REGISTER, MI1040_COMMAND_REGISTER_REFRESH,
		MSM_CAMERA_I2C_UNSET_WORD_MASK, MSM_CAMERA_I2C_CMD_POLL},
	{MI1040_COMMAND_REGISTER, MI1040_COMMAND_REGISTER_OK,
		MSM_CAMERA_I2C_SET_WORD_MASK, MSM_CAMERA_I2C_CMD_POLL},
};

static struct msm_camera_i2c_reg_conf ColorEffect_None[] = {
	//[2.1 Normal -- default]
	{0x098E, 0xC874} , 	// LOGICAL_ADDRESS_ACCESS [CAM_SFX_CONTROL]
	{0xC874, 0x00, MSM_CAMERA_I2C_BYTE_DATA} , 	// CAM_SFX_CONTROL
	{0xDC00, 0x28, MSM_CAMERA_I2C_BYTE_DATA} , 	// SYSMGR_NEXT_STATE
	{0x0080, 0x8004} , 	// COMMAND_REGISTER
};

static struct msm_camera_i2c_reg_conf ColorEffect_Mono[] = {
	//[2.2 Mono color effect]
	{0x098E, 0xC874} , 	// LOGICAL_ADDRESS_ACCESS [CAM_SFX_CONTROL]
	{0xC874, 0x01, MSM_CAMERA_I2C_BYTE_DATA} , 	// CAM_SFX_CONTROL
	{0xDC00, 0x28, MSM_CAMERA_I2C_BYTE_DATA} , 	// SYSMGR_NEXT_STATE
	{0x0080, 0x8004} , 	// COMMAND_REGISTER
};

static struct msm_camera_i2c_reg_conf ColorEffect_Sepia[] = {
	//[2.3 Sepia effect]
	{0x098E, 0xC874} , 	// LOGICAL_ADDRESS_ACCESS [CAM_SFX_CONTROL]
	{0xC874, 0x02, MSM_CAMERA_I2C_BYTE_DATA} , 	// CAM_SFX_CONTROL
	{0xDC00, 0x28, MSM_CAMERA_I2C_BYTE_DATA} , 	// SYSMGR_NEXT_STATE
	{0x0080, 0x8004} , 	// COMMAND_REGISTER
};

static struct msm_camera_i2c_reg_conf ColorEffect_Negative[] = {
	//[2.4 Negative effect]
	{0x098E, 0xC874} , 	// LOGICAL_ADDRESS_ACCESS [CAM_SFX_CONTROL]
	{0xC874, 0x03, MSM_CAMERA_I2C_BYTE_DATA} , 	// CAM_SFX_CONTROL
	{0xDC00, 0x28, MSM_CAMERA_I2C_BYTE_DATA} , 	// SYSMGR_NEXT_STATE
	{0x0080, 0x8004} , 	// COMMAND_REGISTER
};

static struct msm_camera_i2c_reg_conf ColorEffect_Solarize[] = {
	//[2.5 Solarize 1]
	{0x098E, 0xC874} , 	// LOGICAL_ADDRESS_ACCESS [CAM_SFX_CONTROL]
	{0xC874, 0x04, MSM_CAMERA_I2C_BYTE_DATA} , 	// CAM_SFX_CONTROL
	{0xDC00, 0x28, MSM_CAMERA_I2C_BYTE_DATA} , 	// SYSMGR_NEXT_STATE
	{0x0080, 0x8004} , 	// COMMAND_REGISTER
};

static struct msm_camera_i2c_reg_conf Whitebalance_Auto[] = {
	//[4.1 AWB -- default]
	{0x098E, 0x0000} , 	// LOGICAL_ADDRESS_ACCESS
	{0xC909, 0x02, MSM_CAMERA_I2C_BYTE_DATA} , 	// CAM_AWB_AWBMODE
};

static struct msm_camera_i2c_reg_conf Whitebalance_Incandescent[] = {
	//[4.5 MWB: A Light]
	{0x098E, 0x0000} , 	// LOGICAL_ADDRESS_ACCESS
	{0xC909, 0x00, MSM_CAMERA_I2C_BYTE_DATA} , 	// CAM_AWB_AWBMODE
	{0xC8F0, 0x0A8C} , 	// CAM_AWB_COLOR_TEMPERATURE
};

static struct msm_camera_i2c_reg_conf Whitebalance_Daylight[] = {
	//[4.2 MWB: D65]
	{0x098E, 0x0000} , 	// LOGICAL_ADDRESS_ACCESS
	{0xC909, 0x00, MSM_CAMERA_I2C_BYTE_DATA} , 	// CAM_AWB_AWBMODE
	{0xC8F0, 0x1964} , 	// CAM_AWB_COLOR_TEMPERATURE
};

static struct msm_camera_i2c_reg_conf Whitebalance_Fluorescent[] = {
	//[4.4 MWB: TL84]
	{0x098E, 0x0000} , 	// LOGICAL_ADDRESS_ACCESS
	{0xC909, 0x00, MSM_CAMERA_I2C_BYTE_DATA} , 	// CAM_AWB_AWBMODE
	{0xC8F0, 0x0E74} , 	// CAM_AWB_COLOR_TEMPERATURE
};

static struct msm_camera_i2c_reg_conf EV_zero[] = {
	//[3.3 EV0: 128 -- default]
	{0x098E, 0xC87A} , 	// LOGICAL_ADDRESS_ACCESS [CAM_AET_TARGET_AVERAGE_LUMA]
	{0xC87A, 0x3C, MSM_CAMERA_I2C_BYTE_DATA} , 	// CAM_AET_TARGET_AVERAGE_LUMA
	{0xC87B, 0x1E, MSM_CAMERA_I2C_BYTE_DATA} , 	// CAM_AET_TARGET_AVERAGE_LUMA_DARK
};

static struct msm_camera_i2c_reg_conf EV_plus_1[] = {
	//[3.4 EV+1: 138]
	{0x098E, 0xC87A} , 	// LOGICAL_ADDRESS_ACCESS [CAM_AET_TARGET_AVERAGE_LUMA]
	{0xC87A, 0x42, MSM_CAMERA_I2C_BYTE_DATA} , 	// CAM_AET_TARGET_AVERAGE_LUMA
	{0xC87B, 0x21, MSM_CAMERA_I2C_BYTE_DATA} , 	// CAM_AET_TARGET_AVERAGE_LUMA_DARK
};

static struct msm_camera_i2c_reg_conf EV_plus_2[] = {
	//[3.5 EV+2: 148]
	{0x098E, 0xC87A} , 	// LOGICAL_ADDRESS_ACCESS [CAM_AET_TARGET_AVERAGE_LUMA]
	{0xC87A, 0x48, MSM_CAMERA_I2C_BYTE_DATA} , 	// CAM_AET_TARGET_AVERAGE_LUMA
	{0xC87B, 0x24, MSM_CAMERA_I2C_BYTE_DATA} , 	// CAM_AET_TARGET_AVERAGE_LUMA_DARK
};

static struct msm_camera_i2c_reg_conf EV_minus_1[] = {
	//[3.2 EV-1: 118]
	{0x098E, 0xC87A} , 	// LOGICAL_ADDRESS_ACCESS [CAM_AET_TARGET_AVERAGE_LUMA]
	{0xC87A, 0x36, MSM_CAMERA_I2C_BYTE_DATA} , 	// CAM_AET_TARGET_AVERAGE_LUMA
	{0xC87B, 0x1B, MSM_CAMERA_I2C_BYTE_DATA} , 	// CAM_AET_TARGET_AVERAGE_LUMA_DARK
};

static struct msm_camera_i2c_reg_conf EV_minus_2[] = {
	//[3.1 EV-2: 108]
	{0x098E, 0xC87A} , 	// LOGICAL_ADDRESS_ACCESS [CAM_AET_TARGET_AVERAGE_LUMA]
	{0xC87A, 0x32, MSM_CAMERA_I2C_BYTE_DATA} , 	// CAM_AET_TARGET_AVERAGE_LUMA
	{0xC87B, 0x19, MSM_CAMERA_I2C_BYTE_DATA} , 	// CAM_AET_TARGET_AVERAGE_LUMA_DARK
};

static struct msm_camera_i2c_reg_conf mi1040_B_recommend_settings[] = {
	{0x098E, 0x1000, MSM_CAMERA_I2C_BYTE_DATA},
	/*cam_sysctl_pll_enable = 1*/
	{0xC97E, 0x01, MSM_CAMERA_I2C_BYTE_DATA},
	/*cam_sysctl_pll_divider_m_n = 288*/
	{0xC980, 0x0120,},
	/*cam_sysctl_pll_divider_p = 1792*/
	{0xC982, 0x0700,},

	{ 0xC800, 0x0004,}, 	// CAM_SENSOR_CFG_Y_ADDR_START
	{ 0xC802, 0x0004,}, 	// CAM_SENSOR_CFG_X_ADDR_START
	{ 0xC804, 0x03CB,}, 	// CAM_SENSOR_CFG_Y_ADDR_END
	{ 0xC806, 0x050B,}, 	// CAM_SENSOR_CFG_X_ADDR_END
	// data length is double words
	{0xC808, 0x02DC,},/*pixclk = 48000000*/
	{0xC80A, 0x6C00,},/*pixclk = 48000000*/
	//
	{ 0xC80C, 0x0001,}, 	// CAM_SENSOR_CFG_ROW_SPEED
	{ 0xC80E, 0x00DB,}, 	// CAM_SENSOR_CFG_FINE_INTEG_TIME_MIN
	{ 0xC810, 0x05C1,}, 	// CAM_SENSOR_CFG_FINE_INTEG_TIME_MAX
	{ 0xC812, 0x03F3,}, 	// CAM_SENSOR_CFG_FRAME_LENGTH_LINES
	{ 0xC814, 0x0644,}, 	// CAM_SENSOR_CFG_LINE_LENGTH_PCK
	{ 0xC816, 0x0060,}, 	// CAM_SENSOR_CFG_FINE_CORRECTION
	{ 0xC818, 0x03C3,}, 	// CAM_SENSOR_CFG_CPIPE_LAST_ROW
	{ 0xC826, 0x0020,}, 	// cam_sensor_cfg_reg_0_data = 32
	{ 0xC834, 0x0000,}, 	// CAM_SENSOR_CONTROL_READ_MODE
	{ 0xC854, 0x0000,}, 	// CAM_CROP_WINDOW_XOFFSET
	{ 0xC856, 0x0000,}, 	// CAM_CROP_WINDOW_YOFFSET
	{ 0xC858, 0x0500,}, 	// CAM_CROP_WINDOW_WIDTH
	{ 0xC85A, 0x03C0,}, 	// CAM_CROP_WINDOW_HEIGHT
	{ 0xC85C, 0x03, MSM_CAMERA_I2C_BYTE_DATA,}, 	// CAM_CROP_CROPMODE
	{ 0xC868, 0x0500,}, 	// CAM_OUTPUT_WIDTH
	{ 0xC86A, 0x03C0,}, 	// CAM_OUTPUT_HEIGHT
	{ 0xC878, 0x00, MSM_CAMERA_I2C_BYTE_DATA,}, 	// cam_aet_aemode = 0
	{ 0xC88C, 0x1D99,}, 	// CAM_AET_MAX_FRAME_RATE
	// Changed to varied frame rate to 7.5~30fps
	{ 0xC88E, 0x0766,}, 	// CAM_AET_MIN_FRAME_RATE
	//
	{ 0xC914, 0x0000,}, 	// CAM_STAT_AWB_CLIP_WINDOW_XSTART
	{ 0xC916, 0x0000,}, 	// CAM_STAT_AWB_CLIP_WINDOW_YSTART
	{ 0xC918, 0x04FF,}, 	// CAM_STAT_AWB_CLIP_WINDOW_XEND
	{ 0xC91A, 0x03BF,}, 	// CAM_STAT_AWB_CLIP_WINDOW_YEND
	{ 0xC91C, 0x0000,}, 	// CAM_STAT_AE_INITIAL_WINDOW_XSTART
	{ 0xC91E, 0x0000,}, 	// CAM_STAT_AE_INITIAL_WINDOW_YSTART
	{ 0xC920, 0x00FF,}, 	// CAM_STAT_AE_INITIAL_WINDOW_XEND
	{ 0xC922, 0x00BF,}, 	// CAM_STAT_AE_INITIAL_WINDOW_YEND
	{ 0xE801, 0x00, MSM_CAMERA_I2C_BYTE_DATA ,},	// AUTO_BINNING_MODE

	// Enable FW patch of anti-flicker
	{ 0x098E, 0xDC00,}, 	// LOGICAL_ADDRESS_ACCESS [SYSMGR_NEXT_STATE]
	{ 0xDC00, 0x28, MSM_CAMERA_I2C_BYTE_DATA ,}, 	// SYSMGR_NEXT_STATE
	{ 0x0080, 0x8002,}, 	// SYSMGR_NEXT_STATE
	{MI1040_COMMAND_REGISTER, MI1040_COMMAND_REGISTER_SET_STATE,
		MSM_CAMERA_I2C_UNSET_WORD_MASK, MSM_CAMERA_I2C_CMD_POLL},

	/*Sensor optimization*/
	{0x316A, 0x8270,},
	{0x316C, 0x8270,},
	{0x3ED0, 0x2305,},
	{0x3ED2, 0x77CF,},
	{0x316E, 0x8202,},
	{0x3180, 0x87FF,},
	{0x30D4, 0x6080,},
	{0xA802, 0x0008,},
	{0x3E14, 0xFF39,},
	{0x31E0, 0x0001,},

	// Flicker Detect
	{ 0x0982, 0x0001,}, 	// ACCESS_CTL_STAT
	{ 0xC98A, 0x5C10,}, 	// PHYSICAL_ADDRESS_ACCESS
	{ 0x5C10, 0xDC10, MSM_CAMERA_I2C_WORD_DATA, MSM_CAMERA_I2C_CMD_SEQ_WRITE,
		0, mi1040_flicker_detect_table, ARRAY_SIZE(mi1040_flicker_detect_table)},
	{ 0x098E, 0x0000,}, 	// LOGICAL_ADDRESS_ACCESS [SYSMGR_NEXT_STATE]
	{ 0xE000, 0x1028,}, 	// PATCHLDR_LOADER_ADDRESS
	{ 0xE002, 0x1004,}, 	// PATCHLDR_PATCH_ID
	{ 0xE004, 0x4103,}, 	// PATCHLDR_FIRMWARE_ID
	{ 0xE006, 0x0202,}, 	// PATCHLDR_FIRMWARE_ID
	{MI1040_COMMAND_REGISTER, MI1040_COMMAND_REGISTER_APPLY_PATCH,
		MSM_CAMERA_I2C_UNSET_WORD_MASK, MSM_CAMERA_I2C_CMD_POLL},
	{MI1040_COMMAND_REGISTER, MI1040_COMMAND_REGISTER_APPLY_PATCH,
		MSM_CAMERA_I2C_UNSET_WORD_MASK, MSM_CAMERA_I2C_CMD_POLL},

	// PGA parameter and APGA
	{ 0x3640, 0x0290,}, 	//  P_G1_P0Q0
	{ 0x3642, 0xAFEA,}, 	//  P_G1_P0Q1
	{ 0x3644, 0x04D1,}, 	//  P_G1_P0Q2
	{ 0x3646, 0x85CD,}, 	//  P_G1_P0Q3
	{ 0x3648, 0xA46E,}, 	//  P_G1_P0Q4
	{ 0x364A, 0x00D0,}, 	//  P_R_P0Q0
	{ 0x364C, 0x8EEB,}, 	//  P_R_P0Q1
	{ 0x364E, 0x1C71,}, 	//  P_R_P0Q2
	{ 0x3650, 0xAF8C,}, 	//  P_R_P0Q3
	{ 0x3652, 0xF40D,}, 	//  P_R_P0Q4
	{ 0x3654, 0x0170,}, 	//  P_B_P0Q0
	{ 0x3656, 0x16AB,}, 	//  P_B_P0Q1
	{ 0x3658, 0x4870,}, 	//  P_B_P0Q2
	{ 0x365A, 0xFC2B,}, 	//  P_B_P0Q3
	{ 0x365C, 0x8DED,}, 	//  P_B_P0Q4
	{ 0x365E, 0x0130,}, 	//  P_G2_P0Q0
	{ 0x3660, 0xBFAB,}, 	//  P_G2_P0Q1
	{ 0x3662, 0x0CB1,}, 	//  P_G2_P0Q2
	{ 0x3664, 0xA0CD,}, 	//  P_G2_P0Q3
	{ 0x3666, 0x97AF,}, 	//  P_G2_P0Q4
	{ 0x3680, 0xA0AB,}, 	//  P_G1_P1Q0
	{ 0x3682, 0x5707,}, 	//  P_G1_P1Q1
	{ 0x3684, 0x108B,}, 	//  P_G1_P1Q2
	{ 0x3686, 0x2F8D,}, 	//  P_G1_P1Q3
	{ 0x3688, 0x1BED,}, 	//  P_G1_P1Q4
	{ 0x368A, 0x956C,}, 	//  P_R_P1Q0
	{ 0x368C, 0x3C89,}, 	//  P_R_P1Q1
	{ 0x368E, 0x91CB,}, 	//  P_R_P1Q2
	{ 0x3690, 0x79AB,}, 	//  P_R_P1Q3
	{ 0x3692, 0x7F6C,}, 	//  P_R_P1Q4
	{ 0x3694, 0x6266,}, 	//  P_B_P1Q0
	{ 0x3696, 0xA32B,}, 	//  P_B_P1Q1
	{ 0x3698, 0x84CF,}, 	//  P_B_P1Q2
	{ 0x369A, 0x8E2C,}, 	//  P_B_P1Q3
	{ 0x369C, 0x6BAF,}, 	//  P_B_P1Q4
	{ 0x369E, 0xE109,}, 	//  P_G2_P1Q0
	{ 0x36A0, 0x47EC,}, 	//  P_G2_P1Q1
	{ 0x36A2, 0xA98F,}, 	//  P_G2_P1Q2
	{ 0x36A4, 0x944C,}, 	//  P_G2_P1Q3
	{ 0x36A6, 0x4A2F,}, 	//  P_G2_P1Q4
	{ 0x36C0, 0x11D1,}, 	//  P_G1_P2Q0
	{ 0x36C2, 0x17CD,}, 	//  P_G1_P2Q1
	{ 0x36C4, 0x8812,}, 	//  P_G1_P2Q2
	{ 0x36C6, 0xE56F,}, 	//  P_G1_P2Q3
	{ 0x36C8, 0x5073,}, 	//  P_G1_P2Q4
	{ 0x36CA, 0x36F1,}, 	//  P_R_P2Q0
	{ 0x36CC, 0x022E,}, 	//  P_R_P2Q1
	{ 0x36CE, 0x8A31,}, 	//  P_R_P2Q2
	{ 0x36D0, 0xC96F,}, 	//  P_R_P2Q3
	{ 0x36D2, 0x6272,}, 	//  P_R_P2Q4
	{ 0x36D4, 0x0151,}, 	//  P_B_P2Q0
	{ 0x36D6, 0x32AE,}, 	//  P_B_P2Q1
	{ 0x36D8, 0x8992,}, 	//  P_B_P2Q2
	{ 0x36DA, 0xD90F,}, 	//  P_B_P2Q3
	{ 0x36DC, 0x69F3,}, 	//  P_B_P2Q4
	{ 0x36DE, 0x1271,}, 	//  P_G2_P2Q0
	{ 0x36E0, 0x6BCC,}, 	//  P_G2_P2Q1
	{ 0x36E2, 0x8692,}, 	//  P_G2_P2Q2
	{ 0x36E4, 0xA0CF,}, 	//  P_G2_P2Q3
	{ 0x36E6, 0x4933,}, 	//  P_G2_P2Q4
	{ 0x3700, 0xFA2E,}, 	//  P_G1_P3Q0
	{ 0x3702, 0xCF8C,}, 	//  P_G1_P3Q1
	{ 0x3704, 0x1850,}, 	//  P_G1_P3Q2
	{ 0x3706, 0xCF4F,}, 	//  P_G1_P3Q3
	{ 0x3708, 0xE671,}, 	//  P_G1_P3Q4
	{ 0x370A, 0xF00E,}, 	//  P_R_P3Q0
	{ 0x370C, 0xFF6D,}, 	//  P_R_P3Q1
	{ 0x370E, 0x7BCE,}, 	//  P_R_P3Q2
	{ 0x3710, 0xA96D,}, 	//  P_R_P3Q3
	{ 0x3712, 0xCA70,}, 	//  P_R_P3Q4
	{ 0x3714, 0xD70C,}, 	//  P_B_P3Q0
	{ 0x3716, 0x1B0C,}, 	//  P_B_P3Q1
	{ 0x3718, 0x716F,}, 	//  P_B_P3Q2
	{ 0x371A, 0x8E2B,}, 	//  P_B_P3Q3
	{ 0x371C, 0xCC51,}, 	//  P_B_P3Q4
	{ 0x371E, 0xD6EE,}, 	//  P_G2_P3Q0
	{ 0x3720, 0x872E,}, 	//  P_G2_P3Q1
	{ 0x3722, 0x0D90,}, 	//  P_G2_P3Q2
	{ 0x3724, 0xFAEC,}, 	//  P_G2_P3Q3
	{ 0x3726, 0xE710,}, 	//  P_G2_P3Q4
	{ 0x3740, 0xB34F,}, 	//  P_G1_P4Q0
	{ 0x3742, 0x87AE,}, 	//  P_G1_P4Q1
	{ 0x3744, 0x08F4,}, 	//  P_G1_P4Q2
	{ 0x3746, 0x0892,}, 	//  P_G1_P4Q3
	{ 0x3748, 0x9CB5,}, 	//  P_G1_P4Q4
	{ 0x374A, 0x8E50,}, 	//  P_R_P4Q0
	{ 0x374C, 0xF58E,}, 	//  P_R_P4Q1
	{ 0x374E, 0x7132,},	//  P_R_P4Q2
	{ 0x3750, 0x5C91,}, 	//  P_R_P4Q3
	{ 0x3752, 0x9094,}, 	//  P_R_P4Q4
	{ 0x3754, 0xE090,}, 	//  P_B_P4Q0
	{ 0x3756, 0xAD30,}, 	//  P_B_P4Q1
	{ 0x3758, 0x0E34,}, 	//  P_B_P4Q2
	{ 0x375A, 0x0092,}, 	//  P_B_P4Q3
	{ 0x375C, 0xA315,}, 	//  P_B_P4Q4
	{ 0x375E, 0xAD2F,}, 	//  P_G2_P4Q0
	{ 0x3760, 0x09AB,}, 	//  P_G2_P4Q1
	{ 0x3762, 0x0154,}, 	//  P_G2_P4Q2
	{ 0x3764, 0x32F1,}, 	//  P_G2_P4Q3
	{ 0x3766, 0x8CD5,}, 	//  P_G2_P4Q4
	{ 0x3784, 0x0278,}, 	//  CENTER_COLUMN
	{ 0x3782, 0x01EC,}, 	//  CENTER_ROW
	{ 0x37C0, 0xCA47,}, 	//  P_GR_Q5
	{ 0x37C2, 0x49C5,}, 	//  P_RD_Q5
	{ 0x37C4, 0x9AEA,}, 	//  P_BL_Q5
	{ 0x37C6, 0xE348,}, 	//  P_GB_Q5
	{ 0x098E, 0x0000,}, 	//  LOGICAL addressing

	// -- 0518, Alias
	{ 0xC960, 0x0AF0,}, 	// CAM_PGA_L_CONFIG_COLOUR_TEMP
	{ 0xC962, 0x79CF,}, 	// CAM_PGA_L_CONFIG_GREEN_RED_Q14
	{ 0xC964, 0x6A40,}, 	// CAM_PGA_L_CONFIG_RED_Q14
	{ 0xC966, 0x790F,}, 	// CAM_PGA_L_CONFIG_GREEN_BLUE_Q14
	{ 0xC968, 0x78C6,}, 	// CAM_PGA_L_CONFIG_BLUE_Q14
	//
	{ 0xC96A, 0x0FA0,}, 	// CAM_PGA_M_CONFIG_COLOUR_TEMP
	{ 0xC96C, 0x7FCD,}, 	// CAM_PGA_M_CONFIG_GREEN_RED_Q14
	{ 0xC96E, 0x800D,}, 	// CAM_PGA_M_CONFIG_RED_Q14
	{ 0xC970, 0x7F8E,}, 	// CAM_PGA_M_CONFIG_GREEN_BLUE_Q14
	{ 0xC972, 0x7ECA,}, 	// CAM_PGA_M_CONFIG_BLUE_Q14
	{ 0xC974, 0x1964,},	// CAM_PGA_R_CONFIG_COLOUR_TEMP
	{ 0xC976, 0x7CFB,}, 	// CAM_PGA_R_CONFIG_GREEN_RED_Q14
	{ 0xC978, 0x70C6,}, 	// CAM_PGA_R_CONFIG_RED_Q14
	{ 0xC97A, 0x7EA8,}, 	// CAM_PGA_R_CONFIG_GREEN_BLUE_Q14
	{ 0xC97C, 0x784F,}, 	// CAM_PGA_R_CONFIG_BLUE_Q14
	{ 0xC95E, 0x0003,}, 	// CAM_PGA_PGA_CONTROL
	{ 0x3786, 0x0004,}, 	// PGA_Y_ADDR_START
	{ 0x3788, 0x03CB,}, 	// PGA_Y_ADDR_END
	{ 0x378A, 0x0004,}, 	// PGA_X_ADDR_START
	{ 0x378C, 0x050B,}, 	// PGA_X_ADDR_END

	/*[CCM]*/
	{0xC892, 0x0267,},/*CAM_AWB_CCM_L_0*/
	{0xC894, 0xFF1A,},/*CAM_AWB_CCM_L_1*/
	{0xC896, 0xFFB3,},/*CAM_AWB_CCM_L_2*/
	{0xC898, 0xFF80,},/*CAM_AWB_CCM_L_3*/
	{0xC89A, 0x0166,},/*CAM_AWB_CCM_L_4*/
	{0xC89C, 0x0003,},/*CAM_AWB_CCM_L_5*/
	{0xC89E, 0xFF9A,},/*CAM_AWB_CCM_L_6*/
	{0xC8A0, 0xFEB4,},/*CAM_AWB_CCM_L_7*/
	{0xC8A2, 0x024D,},/*CAM_AWB_CCM_L_8*/
	{0xC8A4, 0x01BF,},/*CAM_AWB_CCM_M_0*/
	{0xC8A6, 0xFF01,},/*CAM_AWB_CCM_M_1*/
	{0xC8A8, 0xFFF3,},/*CAM_AWB_CCM_M_2*/
	{0xC8AA, 0xFF75,},/*CAM_AWB_CCM_M_3*/
	{0xC8AC, 0x0198,},/*CAM_AWB_CCM_M_4*/
	{0xC8AE, 0xFFFD,},/*CAM_AWB_CCM_M_5*/
	{0xC8B0, 0xFF9A,},/*CAM_AWB_CCM_M_6*/
	{0xC8B2, 0xFEE7,},/*CAM_AWB_CCM_M_7*/
	{0xC8B4, 0x02A8,},/*CAM_AWB_CCM_M_8*/

	{0xC8B6, 0x0100,},/*CAM_AWB_CCM_R_0*/
	{0xC8B8, 0xFF84,},/*CAM_AWB_CCM_R_1*/
	{0xC8BA, 0xFFEB,},/*CAM_AWB_CCM_R_2*/
	{0xC8BC, 0xFFBD,},/*CAM_AWB_CCM_R_3*/
	{0xC8BE, 0x0105,},/*CAM_AWB_CCM_R_4*/
	{0xC8C0, 0xFFDB,},/*CAM_AWB_CCM_R_5*/
	{0xC8C2, 0xFFCC,},/*CAM_AWB_CCM_R_6*/
	{0xC8C4, 0xFE2A,},/*CAM_AWB_CCM_R_7*/
	{0xC8C6, 0x03AB,},/*CAM_AWB_CCM_R_8*/

	{0xC8DA, 0x004D,},// CAM_AWB_LL_CCM_0
	{0xC8DC, 0x0096,},// CAM_AWB_LL_CCM_1
	{0xC8DE, 0x001D,},// CAM_AWB_LL_CCM_2
	{0xC8E0, 0x004D,},// CAM_AWB_LL_CCM_3
	{0xC8E2, 0x0096,},// CAM_AWB_LL_CCM_4
	{0xC8E4, 0x001D,},// CAM_AWB_LL_CCM_5
	{0xC8E6, 0x004D,},// CAM_AWB_LL_CCM_6
	{0xC8E8, 0x0096,},// CAM_AWB_LL_CCM_7
	{0xC8EA, 0x001D,},// CAM_AWB_LL_CCM_8

	/*[AWB Gain Setting]*/
	{0xC8C8, 0x0075,},// CAM_AWB_CCM_L_RG_GAIN
	{0xC8CA, 0x011C,},// CAM_AWB_CCM_L_BG_GAIN
	{0xC8CC, 0x00A0,},// CAM_AWB_CCM_M_RG_GAIN
	{0xC8CE, 0x00F3,},// CAM_AWB_CCM_M_BG_GAIN
	{0xC8D0, 0x00B3,},// CAM_AWB_CCM_R_RG_GAIN
	{0xC8D2, 0x009D,},// CAM_AWB_CCM_R_BG_GAIN
	{0xC8D4, 0x0A8C,},// CAM_AWB_CCM_L_CTEMP
	{0xC8D6, 0x1004,},// CAM_AWB_CCM_M_CTEMP
	{0xC8D8, 0x1964,},// CAM_AWB_CCM_R_CTEMP

	/*[AWB]*/
	{0xC914, 0x0000,},/*CAM_STAT_AWB_CLIP_WINDOW_XSTART*/
	{0xC916, 0x0000,},/*CAM_STAT_AWB_CLIP_WINDOW_YSTART*/
	{0xC918, 0x04FF,},/*CAM_STAT_AWB_CLIP_WINDOW_XEND*/
	{0xC91A, 0x03BF,},/*CAM_STAT_AWB_CLIP_WINDOW_YEND*/
	{0xC8F2, 0x04, MSM_CAMERA_I2C_BYTE_DATA},/*CAM_AWB_AWB_XSCALE*/
	{0xC8F3, 0x02, MSM_CAMERA_I2C_BYTE_DATA},/*CAM_AWB_AWB_YSCALE*/
	{0xC904, 0x0037,},/*CAM_AWB_AWB_YSHIFT_PRE_ADJ*/
	{0xC906, 0x003C,},/*CAM_AWB_AWB_YSHIFT_PRE_ADJ*/
	{0xC8F4, 0x0000,},// CAM_AWB_AWB_WEIGHTS_0
	{0xC8F6, 0x0000,},// CAM_AWB_AWB_WEIGHTS_1
	{0xC8F8, 0x0000,},// CAM_AWB_AWB_WEIGHTS_2
	{0xC8FA, 0xE724,},// CAM_AWB_AWB_WEIGHTS_3
	{0xC8FC, 0x1583,},// CAM_AWB_AWB_WEIGHTS_4
	{0xC8FE, 0x2045,},// CAM_AWB_AWB_WEIGHTS_5
	{0xC900, 0x05DC,},// CAM_AWB_AWB_WEIGHTS_6
	{0xC902, 0x007C,},// CAM_AWB_AWB_WEIGHTS_7
	{0xC90A, 0x1004,},// CAM_AWB_TINTS_CTEMP_THRESHOLD
	{0xC90C, 0x78, MSM_CAMERA_I2C_BYTE_DATA},// CAM_AWB_K_R_L
	{0xC90D, 0x80, MSM_CAMERA_I2C_BYTE_DATA},// CAM_AWB_K_G_L
	{0xC90E, 0x88, MSM_CAMERA_I2C_BYTE_DATA},// CAM_AWB_K_B_L
	{0xC90F, 0x76, MSM_CAMERA_I2C_BYTE_DATA},// CAM_AWB_K_R_R
	{0xC910, 0x83, MSM_CAMERA_I2C_BYTE_DATA},// CAM_AWB_K_G_R
	{0xC911, 0x78, MSM_CAMERA_I2C_BYTE_DATA},// CAM_AWB_K_B_R
	{0x098E, 0xAC06,}, // LOGICAL_ADDRESS_ACCESS [AWB_R_RATIO_LOWER]
	{0xAC06, 0x63, MSM_CAMERA_I2C_BYTE_DATA},// AWB_R_RATIO_LOWER
	{0xAC07, 0x65, MSM_CAMERA_I2C_BYTE_DATA},// AWB_R_RATIO_UPPER
	{0xAC08, 0x63, MSM_CAMERA_I2C_BYTE_DATA},// AWB_B_RATIO_LOWER
	{0xAC09, 0x65, MSM_CAMERA_I2C_BYTE_DATA},// AWB_B_RATIO_UPPER

	/*[Step7-CPIPE_Preference]*/
	{0xC926, 0x0020,},// CAM_LL_START_BRIGHTNESS
	{0xC928, 0x009A,},// CAM_LL_STOP_BRIGHTNESS
	{0xC946, 0x0070,},// CAM_LL_START_GAIN_METRIC
	{0xC948, 0x00F3,},// CAM_LL_STOP_GAIN_METRIC
	{0xC952, 0x0090,},// CAM_LL_START_TARGET_LUMA_BM
	{0xC954, 0x02C0,},// CAM_LL_STOP_TARGET_LUMA_BM
	{0xC95A, 0x04, MSM_CAMERA_I2C_BYTE_DATA},// CAM_SEQ_UV_COLOR_BOOST
	{0xC92A, 0x50, MSM_CAMERA_I2C_BYTE_DATA},// CAM_LL_START_SATURATION
	{0xC92B, 0x3C, MSM_CAMERA_I2C_BYTE_DATA},// CAM_LL_END_SATURATION
	{0xC92C, 0x00, MSM_CAMERA_I2C_BYTE_DATA},// CAM_LL_START_DESATURATION
	{0xC92D, 0xFF, MSM_CAMERA_I2C_BYTE_DATA},// CAM_LL_END_DESATURATION
	{0xC92E, 0x3C, MSM_CAMERA_I2C_BYTE_DATA},   // CAM_LL_START_DEMOSAIC
	{0xC92F, 0x02, MSM_CAMERA_I2C_BYTE_DATA},   // CAM_LL_START_AP_GAIN
	{0xC930, 0x06, MSM_CAMERA_I2C_BYTE_DATA},   // CAM_LL_START_AP_THRESH
	{0xC931, 0x64, MSM_CAMERA_I2C_BYTE_DATA},   // CAM_LL_STOP_DEMOSAIC
	{0xC932, 0x01, MSM_CAMERA_I2C_BYTE_DATA},   // CAM_LL_STOP_AP_GAIN
	{0xC933, 0x0C, MSM_CAMERA_I2C_BYTE_DATA},   // CAM_LL_STOP_AP_THRESH
	{0xC934, 0x3C, MSM_CAMERA_I2C_BYTE_DATA},   // CAM_LL_START_NR_RED
	{0xC935, 0x3C, MSM_CAMERA_I2C_BYTE_DATA},   // CAM_LL_START_NR_GREEN
	{0xC936, 0x3C, MSM_CAMERA_I2C_BYTE_DATA},   // CAM_LL_START_NR_BLUE
	{0xC937, 0x0F, MSM_CAMERA_I2C_BYTE_DATA},   // CAM_LL_START_NR_THRESH
	{0xC938, 0x64, MSM_CAMERA_I2C_BYTE_DATA},   // CAM_LL_STOP_NR_RED
	{0xC939, 0x64, MSM_CAMERA_I2C_BYTE_DATA},   // CAM_LL_STOP_NR_GREEN
	{0xC93A, 0x64, MSM_CAMERA_I2C_BYTE_DATA},   // CAM_LL_STOP_NR_BLUE
	{0xC93B, 0x32, MSM_CAMERA_I2C_BYTE_DATA},   // CAM_LL_STOP_NR_THRESH
	{0xC93C, 0x0005 ,},/*CAM_LL_START_CONTRAST_BM*/
	{0xC93E, 0x0358 ,},/*CAM_LL_STOP_CONTRAST_BM*/
	{0xC940, 0x00DC ,},/*CAM_LL_GAMMA*/
	/*CAM_LL_START_CONTRAST_GRADIENT*/
	{0xC942, 0x3C, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC943, 0x30, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC944, 0x22, MSM_CAMERA_I2C_BYTE_DATA},/*CAM_LL_START_CONTRAST_LUMA*/
	{0xC945, 0x19, MSM_CAMERA_I2C_BYTE_DATA},/*CAM_LL_STOP_CONTRAST_LUMA*/
	{0x098E, 0xC942,},// LOGICAL_ADDRESS_ACCESS [CAM_LL_START_CONTRAST_GRADIENT]
	{0x098E, 0xC92A,},// LOGICAL_ADDRESS_ACCESS [CAM_LL_START_SATURATION]
	{0xC94A, 0x00F0,},// CAM_LL_START_FADE_TO_BLACK_LUMA
	{0xC94C, 0x0010,},// CAM_LL_STOP_FADE_TO_BLACK_LUMA
	{0xC94E, 0x01CD,},// CAM_LL_CLUSTER_DC_TH_BM
	{0xC950, 0x05, MSM_CAMERA_I2C_BYTE_DATA},// CAM_LL_CLUSTER_DC_GATE_PERCENTAGE
	{0xC951, 0x40, MSM_CAMERA_I2C_BYTE_DATA},// CAM_LL_SUMMING_SENSITIVITY_FACTOR
	{0xC87A, 0x37, MSM_CAMERA_I2C_BYTE_DATA},// CAM_AET_TARGET_AVERAGE_LUMA
	{0xC87B, 0x0E, MSM_CAMERA_I2C_BYTE_DATA},// CAM_AET_TARGET_AVERAGE_LUMA_DARK
	{0xC878, 0x0E, MSM_CAMERA_I2C_BYTE_DATA},// CAM_AET_AEMODE  -- for DR test
	{0xC890, 0x0040 ,},/*CAM_AET_TARGET_GAIN*/
	{0xC882, 0x0100 ,},/*CAM_AET_AE_MAX_VIRT_AGAIN*/
	{0xC886, 0x0100 ,},/*CAM_AET_BLACK_CLIPPING_TARGET*/
	{0xA404, 0x02, MSM_CAMERA_I2C_BYTE_DATA},/*CCM_DELTA_GAIN*/
	/*AE_TRACK_AE_TRACKING_DAMPENING*/
	{0xC87C, 0x0016,}, // CAM_AET_BLACK_CLIPPING_TARGET
	{0xB42A, 0x05, MSM_CAMERA_I2C_BYTE_DATA}, // CCM_DELTA_GAIN
	{0xA80A, 0x18, MSM_CAMERA_I2C_BYTE_DATA}, // AE_TRACK_AE_TRACKING_DAMPENING_SPEED
	{0x326C, 0x1706,},// APERTURE_PARAMETERS_2D

	/*Step8-Features*/
	{ 0x098E, 0x0000,}, 	// LOGICAL_ADDRESS_ACCESS
	{ 0xC984, 0x8040,}, 	// CAM_PORT_OUTPUT_CONTROL
	{ 0x001E, 0x0777,}, 	// PAD_SLEW

	// LOAD=MIPI setting for SOC1040
	{ 0xC984, 0x8001,}, 	// CAM_PORT_OUTPUT_CONTROL
	{ 0xC988, 0x0F00,}, 	// CAM_PORT_MIPI_TIMING_T_HS_ZERO
	{ 0xC98A, 0x0B07,}, 	// CAM_PORT_MIPI_TIMING_T_HS_EXIT_HS_TRAIL
	{ 0xC98C, 0x0D01,}, 	// CAM_PORT_MIPI_TIMING_T_CLK_POST_CLK_PRE
	{ 0xC98E, 0x071D,}, 	// CAM_PORT_MIPI_TIMING_T_CLK_TRAIL_CLK_ZERO
	{ 0xC990, 0x0006,}, 	// CAM_PORT_MIPI_TIMING_T_LPX
	{ 0xC992, 0x0A0C,}, 	// CAM_PORT_MIPI_TIMING_INIT_TIMING

	// Improve AE Dampening speed
	//[Speed up AE/AWB]
	{ 0x098E, 0x2802,}, 	// LOGICAL_ADDRESS_ACCESS
//	{ 0xA802, 0x0008,}, 	// AE_TRACK_MODE
	{ 0xC908, 0x01, MSM_CAMERA_I2C_BYTE_DATA}, 	// CAM_AWB_SKIP_FRAMES
	{ 0xC879, 0x01, MSM_CAMERA_I2C_BYTE_DATA}, 	// CAM_AET_SKIP_FRAMES
	{ 0xC909, 0x02, MSM_CAMERA_I2C_BYTE_DATA}, 	// CAM_AWB_AWBMODE
//	{ 0xA80A, 0x18, MSM_CAMERA_I2C_BYTE_DATA}, 	// AE_TRACK_AE_TRACKING_DAMPENING_SPEED
	{ 0xA80B, 0x18, MSM_CAMERA_I2C_BYTE_DATA}, 	// AE_TRACK_AE_DAMPENING_SPEED
	{ 0xAC16, 0x18, MSM_CAMERA_I2C_BYTE_DATA}, 	// AWB_PRE_AWB_RATIOS_TRACKING_SPEED

	{ 0x31E0, 0x01, MSM_CAMERA_I2C_BYTE_DATA}, 	// COLOR DOT
};

#if 0
/* Aptina SENSOR SCCB */
struct MI1040_sensor {
	uint16_t addr;
	uint16_t data;
};

struct sensor_reg {
	uint16_t addr;
	uint16_t val;
};

struct sensor_reg_ex {
	uint16_t cmd;
	uint16_t addr;
	uint16_t val;
};

/* Sensor ID */
#define MI1040_SENSOR_ID 0x2481

#define capture_framerate 3000     /* 30fps capture frame rate */
#define g_preview_frameRate 3000  /* 30fps preview frame rate */

#define SENSOR_WAIT_MS          0 /* special number to indicate this is wait time require */
#define SENSOR_TABLE_END        1 /* special number to indicate this is end of table */
#define SENSOR_BYTE_WRITE       2
#define SENSOR_WORD_WRITE       3
#define SENSOR_MASK_BYTE_WRITE  4
#define SENSOR_MASK_WORD_WRITE  5
#define SEQ_WRITE_START         6
#define SEQ_WRITE_END           7
#define SENSOR_BYTE_READ		8
#define SENSOR_WORD_READ		9

static struct sensor_reg_ex mode_1280x960[] =
{
	//+
	//[Initialization]
	// Hardware Reset, need to be done by hardware or host processor.
	// Reset
	//POLL_FIELD= COMMAND_REGISTER, HOST_COMMAND_1, !=0, DELAY=10, TIMEOUT=100
	// LOAD = Step1-Reset 			//Reset

	{SENSOR_WORD_WRITE, 0x001A, 0x0001},
	{SENSOR_WAIT_MS, 0, 10}, 	//delay=10
	{SENSOR_WORD_WRITE, 0x001A, 0x0000},
	{SENSOR_WAIT_MS, 0, 50}, 	//delay=50
	{SENSOR_WORD_WRITE,0x301A, 0x0234} , // RESET_REGISTER
	// [Step2-PLL_Timing]
	// LOAD=PLL_settings
	{SENSOR_WORD_WRITE,0x098E, 0x1000} , // LOGICAL_ADDRESS_ACCESS
	{SENSOR_BYTE_WRITE,0xC97E, 0x01} , 	// CAM_SYSCTL_PLL_ENABLE
	{SENSOR_WORD_WRITE,0xC980, 0x0120} , 	// CAM_SYSCTL_PLL_DIVIDER_M_N
	{SENSOR_WORD_WRITE,0xC982, 0x0700} , 	// CAM_SYSCTL_PLL_DIVIDER_P
	//;REG= 0xC984, 0x8040} , 	// CAM_PORT_OUTPUT_CONTROL

	// LOAD=Timing_settings
	{SENSOR_WORD_WRITE,0x098E, 0x0000} , 	// set XDMA to logical addressing
	{SENSOR_WORD_WRITE,0xC800, 0x0004} , 	// CAM_SENSOR_CFG_Y_ADDR_START
	{SENSOR_WORD_WRITE,0xC802, 0x0004} , 	// CAM_SENSOR_CFG_X_ADDR_START
	{SENSOR_WORD_WRITE,0xC804, 0x03CB} , 	// CAM_SENSOR_CFG_Y_ADDR_END
	{SENSOR_WORD_WRITE,0xC806, 0x050B} , 	// CAM_SENSOR_CFG_X_ADDR_END
	// data length is double words
	{SENSOR_WORD_WRITE,0xC808, 0x02DC} ,
	{SENSOR_WORD_WRITE,0xC80A, 0x6C00} , 	// CAM_SENSOR_CFG_PIXCLK
	//
	{SENSOR_WORD_WRITE,0xC80C, 0x0001} , 	// CAM_SENSOR_CFG_ROW_SPEED
	{SENSOR_WORD_WRITE,0xC80E, 0x00DB} , 	// CAM_SENSOR_CFG_FINE_INTEG_TIME_MIN
	{SENSOR_WORD_WRITE,0xC810, 0x05B3} , 	// CAM_SENSOR_CFG_FINE_INTEG_TIME_MAX
	{SENSOR_WORD_WRITE,0xC812, 0x03EE} , 	// CAM_SENSOR_CFG_FRAME_LENGTH_LINES
	{SENSOR_WORD_WRITE,0xC814, 0x0636} , 	// CAM_SENSOR_CFG_LINE_LENGTH_PCK
	{SENSOR_WORD_WRITE,0xC816, 0x0060} , 	// CAM_SENSOR_CFG_FINE_CORRECTION
	{SENSOR_WORD_WRITE,0xC818, 0x03C3} , 	// CAM_SENSOR_CFG_CPIPE_LAST_ROW
	{SENSOR_WORD_WRITE,0xC834, 0x0000} , 	// CAM_SENSOR_CONTROL_READ_MODE
	{SENSOR_WORD_WRITE,0xC854, 0x0000} , 	// CAM_CROP_WINDOW_XOFFSET
	{SENSOR_WORD_WRITE,0xC856, 0x0000} , 	// CAM_CROP_WINDOW_YOFFSET
	{SENSOR_WORD_WRITE,0xC858, 0x0500} , 	// CAM_CROP_WINDOW_WIDTH
	{SENSOR_WORD_WRITE,0xC85A, 0x03C0} , 	// CAM_CROP_WINDOW_HEIGHT
	{SENSOR_BYTE_WRITE,0xC85C, 0x03} , 	// CAM_CROP_CROPMODE
	{SENSOR_WORD_WRITE,0xC868, 0x0500} , 	// CAM_OUTPUT_WIDTH
	{SENSOR_WORD_WRITE,0xC86A, 0x03C0} , 	// CAM_OUTPUT_HEIGHT
	{SENSOR_WORD_WRITE,0xC88C, 0x1E02} , 	// CAM_AET_MAX_FRAME_RATE
	// -- Rev2, 02242011, Alias
	// Changed to varied frame rate from 15~30fps
	//{SENSOR_WORD_WRITE,0xC88E, 0x0F00} ,	//0x1E02} , 	// CAM_AET_MIN_FRAME_RATE
	// -- Rev3, 03082011, Alias
	// Changed to varied frame rate to 7.5~30fps
	{SENSOR_WORD_WRITE,0xC88E, 0x0780} ,	//0x1E02} , 	// CAM_AET_MIN_FRAME_RATE
	{SENSOR_WORD_WRITE,0xC914, 0x0000} , 	// CAM_STAT_AWB_CLIP_WINDOW_XSTART
	{SENSOR_WORD_WRITE,0xC916, 0x0000} , 	// CAM_STAT_AWB_CLIP_WINDOW_YSTART
	{SENSOR_WORD_WRITE,0xC918, 0x04FF} , 	// CAM_STAT_AWB_CLIP_WINDOW_XEND
	{SENSOR_WORD_WRITE,0xC91A, 0x03BF} , 	// CAM_STAT_AWB_CLIP_WINDOW_YEND
	{SENSOR_WORD_WRITE,0xC91C, 0x0000} , 	// CAM_STAT_AE_INITIAL_WINDOW_XSTART
	{SENSOR_WORD_WRITE,0xC91E, 0x0000} , 	// CAM_STAT_AE_INITIAL_WINDOW_YSTART
	{SENSOR_WORD_WRITE,0xC920, 0x00FF} , 	// CAM_STAT_AE_INITIAL_WINDOW_XEND
	{SENSOR_WORD_WRITE,0xC922, 0x00BF} , 	// CAM_STAT_AE_INITIAL_WINDOW_YEND
	{SENSOR_BYTE_WRITE,0xE801, 0x00} , 	// AUTO_BINNING_MODE


	// [Step3-Recommended]
	// [Sensor optimization]
	{SENSOR_WORD_WRITE,0x316A, 0x8270} , 	// DAC_TXLO_ROW
	{SENSOR_WORD_WRITE,0x316C, 0x8270} , 	// DAC_TXLO
	{SENSOR_WORD_WRITE,0x3ED0, 0x2305} , 	// DAC_LD_4_5
	{SENSOR_WORD_WRITE,0x3ED2, 0x77CF} , 	// DAC_LD_6_7
	{SENSOR_WORD_WRITE,0x316E, 0x8202} , 	// DAC_ECL
	{SENSOR_WORD_WRITE,0x3180, 0x87FF} , 	// DELTA_DK_CONTROL
	{SENSOR_WORD_WRITE,0x30D4, 0x6080} , 	// COLUMN_CORRECTION
	{SENSOR_WORD_WRITE,0xA802, 0x0008} , 	// AE_TRACK_MODE

	// LOAD=Errata item 1
	{SENSOR_WORD_WRITE,0x3E14, 0xFF39} , 	// SAMP_COL_PUP2

	// LOAD=Errata item 2
	{SENSOR_WORD_WRITE,0x301A, 0x8234} , 	// RESET_REGISTER

	// -- Rev2, 02242011, Alias
	// removed FW patch due to no binning application

	// PGA parameter and APGA
	//[Step4-APGA]
	//[TP101_SOC1040_APGA]
	{SENSOR_WORD_WRITE,0x098E, 0x495E} , 	// LOGICAL_ADDRESS_ACCESS [CAM_PGA_PGA_CONTROL]
	{SENSOR_WORD_WRITE,0xC95E, 0x0000} , 	// CAM_PGA_PGA_CONTROL
	{SENSOR_WORD_WRITE,0x3640, 0x02B0} , 	// P_G1_P0Q0
	{SENSOR_WORD_WRITE,0x3642, 0x8063} , 	// P_G1_P0Q1
	{SENSOR_WORD_WRITE,0x3644, 0x78D0} , 	// P_G1_P0Q2
	{SENSOR_WORD_WRITE,0x3646, 0x50CC} , 	// P_G1_P0Q3
	{SENSOR_WORD_WRITE,0x3648, 0x3511} , 	// P_G1_P0Q4
	{SENSOR_WORD_WRITE,0x364A, 0x0110} , 	// P_R_P0Q0
	{SENSOR_WORD_WRITE,0x364C, 0xBD8A} , 	// P_R_P0Q1
	{SENSOR_WORD_WRITE,0x364E, 0x0CD1} , 	// P_R_P0Q2
	{SENSOR_WORD_WRITE,0x3650, 0x24ED} , 	// P_R_P0Q3
	{SENSOR_WORD_WRITE,0x3652, 0x7C11} , 	// P_R_P0Q4
	{SENSOR_WORD_WRITE,0x3654, 0x0150} , 	// P_B_P0Q0
	{SENSOR_WORD_WRITE,0x3656, 0x124C} , 	// P_B_P0Q1
	{SENSOR_WORD_WRITE,0x3658, 0x3130} , 	// P_B_P0Q2
	{SENSOR_WORD_WRITE,0x365A, 0x508C} , 	// P_B_P0Q3
	{SENSOR_WORD_WRITE,0x365C, 0x21F1} , 	// P_B_P0Q4
	{SENSOR_WORD_WRITE,0x365E, 0x0090} , 	// P_G2_P0Q0
	{SENSOR_WORD_WRITE,0x3660, 0xBFCA} , 	// P_G2_P0Q1
	{SENSOR_WORD_WRITE,0x3662, 0x0A11} , 	// P_G2_P0Q2
	{SENSOR_WORD_WRITE,0x3664, 0x4F4B} , 	// P_G2_P0Q3
	{SENSOR_WORD_WRITE,0x3666, 0x28B1} , 	// P_G2_P0Q4
	{SENSOR_WORD_WRITE,0x3680, 0x50A9} , 	// P_G1_P1Q0
	{SENSOR_WORD_WRITE,0x3682, 0xA04B} , 	// P_G1_P1Q1
	{SENSOR_WORD_WRITE,0x3684, 0x0E2D} , 	// P_G1_P1Q2
	{SENSOR_WORD_WRITE,0x3686, 0x73EC} , 	// P_G1_P1Q3
	{SENSOR_WORD_WRITE,0x3688, 0x164F} , 	// P_G1_P1Q4
	{SENSOR_WORD_WRITE,0x368A, 0xF829} , 	// P_R_P1Q0
	{SENSOR_WORD_WRITE,0x368C, 0xC1A8} , 	// P_R_P1Q1
	{SENSOR_WORD_WRITE,0x368E, 0xB0EC} , 	// P_R_P1Q2
	{SENSOR_WORD_WRITE,0x3690, 0xE76A} , 	// P_R_P1Q3
	{SENSOR_WORD_WRITE,0x3692, 0x69AF} , 	// P_R_P1Q4
	{SENSOR_WORD_WRITE,0x3694, 0x378C} , 	// P_B_P1Q0
	{SENSOR_WORD_WRITE,0x3696, 0xA70D} , 	// P_B_P1Q1
	{SENSOR_WORD_WRITE,0x3698, 0x884F} , 	// P_B_P1Q2
	{SENSOR_WORD_WRITE,0x369A, 0xEE8B} , 	// P_B_P1Q3
	{SENSOR_WORD_WRITE,0x369C, 0x5DEF} , 	// P_B_P1Q4
	{SENSOR_WORD_WRITE,0x369E, 0x27CC} , 	// P_G2_P1Q0
	{SENSOR_WORD_WRITE,0x36A0, 0xCAAC} , 	// P_G2_P1Q1
	{SENSOR_WORD_WRITE,0x36A2, 0x840E} , 	// P_G2_P1Q2
	{SENSOR_WORD_WRITE,0x36A4, 0xDAA9} , 	// P_G2_P1Q3
	{SENSOR_WORD_WRITE,0x36A6, 0xF00C} , 	// P_G2_P1Q4
	{SENSOR_WORD_WRITE,0x36C0, 0x1371} , 	// P_G1_P2Q0
	{SENSOR_WORD_WRITE,0x36C2, 0x272F} , 	// P_G1_P2Q1
	{SENSOR_WORD_WRITE,0x36C4, 0x2293} , 	// P_G1_P2Q2
	{SENSOR_WORD_WRITE,0x36C6, 0xE6D0} , 	// P_G1_P2Q3
	{SENSOR_WORD_WRITE,0x36C8, 0xEC32} , 	// P_G1_P2Q4
	{SENSOR_WORD_WRITE,0x36CA, 0x11B1} , 	// P_R_P2Q0
	{SENSOR_WORD_WRITE,0x36CC, 0x7BAF} , 	// P_R_P2Q1
	{SENSOR_WORD_WRITE,0x36CE, 0x5813} , 	// P_R_P2Q2
	{SENSOR_WORD_WRITE,0x36D0, 0xB871} , 	// P_R_P2Q3
	{SENSOR_WORD_WRITE,0x36D2, 0x8913} , 	// P_R_P2Q4
	{SENSOR_WORD_WRITE,0x36D4, 0x4610} , 	// P_B_P2Q0
	{SENSOR_WORD_WRITE,0x36D6, 0x7EEE} , 	// P_B_P2Q1
	{SENSOR_WORD_WRITE,0x36D8, 0x0DF3} , 	// P_B_P2Q2
	{SENSOR_WORD_WRITE,0x36DA, 0xB84F} , 	// P_B_P2Q3
	{SENSOR_WORD_WRITE,0x36DC, 0xB532} , 	// P_B_P2Q4
	{SENSOR_WORD_WRITE,0x36DE, 0x1171} , 	// P_G2_P2Q0
	{SENSOR_WORD_WRITE,0x36E0, 0x13CF} , 	// P_G2_P2Q1
	{SENSOR_WORD_WRITE,0x36E2, 0x22F3} , 	// P_G2_P2Q2
	{SENSOR_WORD_WRITE,0x36E4, 0xE090} , 	// P_G2_P2Q3
	{SENSOR_WORD_WRITE,0x36E6, 0x8133} , 	// P_G2_P2Q4
	{SENSOR_WORD_WRITE,0x3700, 0x88AE} , 	// P_G1_P3Q0
	{SENSOR_WORD_WRITE,0x3702, 0x00EA} , 	// P_G1_P3Q1
	{SENSOR_WORD_WRITE,0x3704, 0x344F} , 	// P_G1_P3Q2
	{SENSOR_WORD_WRITE,0x3706, 0xEC88} , 	// P_G1_P3Q3
	{SENSOR_WORD_WRITE,0x3708, 0x3E91} , 	// P_G1_P3Q4
	{SENSOR_WORD_WRITE,0x370A, 0xF12D} , 	// P_R_P3Q0
	{SENSOR_WORD_WRITE,0x370C, 0xB0EF} , 	// P_R_P3Q1
	{SENSOR_WORD_WRITE,0x370E, 0x77CD} , 	// P_R_P3Q2
	{SENSOR_WORD_WRITE,0x3710, 0x7930} , 	// P_R_P3Q3
	{SENSOR_WORD_WRITE,0x3712, 0x5C12} , 	// P_R_P3Q4
	{SENSOR_WORD_WRITE,0x3714, 0x500C} , 	// P_B_P3Q0
	{SENSOR_WORD_WRITE,0x3716, 0x22CE} , 	// P_B_P3Q1
	{SENSOR_WORD_WRITE,0x3718, 0x2370} , 	// P_B_P3Q2
	{SENSOR_WORD_WRITE,0x371A, 0x258F} , 	// P_B_P3Q3
	{SENSOR_WORD_WRITE,0x371C, 0x3D30} , 	// P_B_P3Q4
	{SENSOR_WORD_WRITE,0x371E, 0x370C} , 	// P_G2_P3Q0
	{SENSOR_WORD_WRITE,0x3720, 0x03ED} , 	// P_G2_P3Q1
	{SENSOR_WORD_WRITE,0x3722, 0x9AD0} , 	// P_G2_P3Q2
	{SENSOR_WORD_WRITE,0x3724, 0x7ECF} , 	// P_G2_P3Q3
	{SENSOR_WORD_WRITE,0x3726, 0x1093} , 	// P_G2_P3Q4
	{SENSOR_WORD_WRITE,0x3740, 0x2391} , 	// P_G1_P4Q0
	{SENSOR_WORD_WRITE,0x3742, 0xAAD0} , 	// P_G1_P4Q1
	{SENSOR_WORD_WRITE,0x3744, 0x28F2} , 	// P_G1_P4Q2
	{SENSOR_WORD_WRITE,0x3746, 0xBA4F} , 	// P_G1_P4Q3
	{SENSOR_WORD_WRITE,0x3748, 0xC536} , 	// P_G1_P4Q4
	{SENSOR_WORD_WRITE,0x374A, 0x1472} , 	// P_R_P4Q0
	{SENSOR_WORD_WRITE,0x374C, 0xD110} , 	// P_R_P4Q1
	{SENSOR_WORD_WRITE,0x374E, 0x2933} , 	// P_R_P4Q2
	{SENSOR_WORD_WRITE,0x3750, 0xD0D1} , 	// P_R_P4Q3
	{SENSOR_WORD_WRITE,0x3752, 0x9F37} , 	// P_R_P4Q4
	{SENSOR_WORD_WRITE,0x3754, 0x34D1} , 	// P_B_P4Q0
	{SENSOR_WORD_WRITE,0x3756, 0x1C6C} , 	// P_B_P4Q1
	{SENSOR_WORD_WRITE,0x3758, 0x3FD2} , 	// P_B_P4Q2
	{SENSOR_WORD_WRITE,0x375A, 0xCB72} , 	// P_B_P4Q3
	{SENSOR_WORD_WRITE,0x375C, 0xBA96} , 	// P_B_P4Q4
	{SENSOR_WORD_WRITE,0x375E, 0x1551} , 	// P_G2_P4Q0
	{SENSOR_WORD_WRITE,0x3760, 0xB74F} , 	// P_G2_P4Q1
	{SENSOR_WORD_WRITE,0x3762, 0x1672} , 	// P_G2_P4Q2
	{SENSOR_WORD_WRITE,0x3764, 0x84F1} , 	// P_G2_P4Q3
	{SENSOR_WORD_WRITE,0x3766, 0xC2D6} , 	// P_G2_P4Q4
	{SENSOR_WORD_WRITE,0x3782, 0x01E0} , 	// CENTER_ROW
	{SENSOR_WORD_WRITE,0x3784, 0x0280} , 	// CENTER_COLUMN
	{SENSOR_WORD_WRITE,0x37C0, 0xA6EA} , 	// P_GR_Q5
	{SENSOR_WORD_WRITE,0x37C2, 0x874B} , 	// P_RD_Q5
	{SENSOR_WORD_WRITE,0x37C4, 0x85CB} , 	// P_BL_Q5
	{SENSOR_WORD_WRITE,0x37C6, 0x968A} , 	// P_GB_Q5
	{SENSOR_WORD_WRITE,0x098E, 0x0000} , 	//  LOGICAL addressing
	{SENSOR_WORD_WRITE,0xC960, 0x0AF0} , 	//  CAM_PGA_L_CONFIG_COLOUR_TEMP
	{SENSOR_WORD_WRITE,0xC962, 0x79E2} , 	//  CAM_PGA_L_CONFIG_GREEN_RED_Q14
	{SENSOR_WORD_WRITE,0xC964, 0x5EC8} ,	//0x56C8} , 	//  CAM_PGA_L_CONFIG_RED_Q14
	{SENSOR_WORD_WRITE,0xC966, 0x791F} , 	//  CAM_PGA_L_CONFIG_GREEN_BLUE_Q14
	{SENSOR_WORD_WRITE,0xC968, 0x76EE} , 	//  CAM_PGA_L_CONFIG_BLUE_Q14
	{SENSOR_WORD_WRITE,0xC96A, 0x0FA0} , 	//  CAM_PGA_M_CONFIG_COLOUR_TEMP
	{SENSOR_WORD_WRITE,0xC96C, 0x7DFA} , 	//  CAM_PGA_M_CONFIG_GREEN_RED_Q14
	{SENSOR_WORD_WRITE,0xC96E, 0x7DAF} , 	//  CAM_PGA_M_CONFIG_RED_Q14
	{SENSOR_WORD_WRITE,0xC970, 0x7E02} , 	//  CAM_PGA_M_CONFIG_GREEN_BLUE_Q14
	{SENSOR_WORD_WRITE,0xC972, 0x7E0A} , 	//  CAM_PGA_M_CONFIG_BLUE_Q14
	{SENSOR_WORD_WRITE,0xC974, 0x1964} , 	//  CAM_PGA_R_CONFIG_COLOUR_TEMP
	{SENSOR_WORD_WRITE,0xC976, 0x7CDC} , 	//  CAM_PGA_R_CONFIG_GREEN_RED_Q14
	{SENSOR_WORD_WRITE,0xC978, 0x7838} ,	//0x7038} , 	//  CAM_PGA_R_CONFIG_RED_Q14
	{SENSOR_WORD_WRITE,0xC97A, 0x7C2F} , 	//  CAM_PGA_R_CONFIG_GREEN_BLUE_Q14
	{SENSOR_WORD_WRITE,0xC97C, 0x7792} , 	//  CAM_PGA_R_CONFIG_BLUE_Q14
	{SENSOR_WORD_WRITE,0xC95E, 0x0003} , 	//  CAM_PGA_PGA_CONTROL
	//

	// [Step4-APGA]
	{SENSOR_WORD_WRITE,0x098E, 0x0000} , 	// LOGICAL_ADDRESS_ACCESS
	// [Step4-APGA]
	{SENSOR_WORD_WRITE,0xC95E, 0x0003} , 	// CAM_PGA_PGA_CONTROL


	// [Step5-AWB_CCM]1: LOAD=CCM
	{SENSOR_WORD_WRITE,0xC892, 0x0267} , 	// CAM_AWB_CCM_L_0
	{SENSOR_WORD_WRITE,0xC894, 0xFF1A} , 	// CAM_AWB_CCM_L_1
	{SENSOR_WORD_WRITE,0xC896, 0xFFB3} , 	// CAM_AWB_CCM_L_2
	{SENSOR_WORD_WRITE,0xC898, 0xFF80} , 	// CAM_AWB_CCM_L_3
	{SENSOR_WORD_WRITE,0xC89A, 0x0166} , 	// CAM_AWB_CCM_L_4
	{SENSOR_WORD_WRITE,0xC89C, 0x0003} , 	// CAM_AWB_CCM_L_5
	{SENSOR_WORD_WRITE,0xC89E, 0xFF9A} , 	// CAM_AWB_CCM_L_6
	{SENSOR_WORD_WRITE,0xC8A0, 0xFEB4} , 	// CAM_AWB_CCM_L_7
	{SENSOR_WORD_WRITE,0xC8A2, 0x024D} , 	// CAM_AWB_CCM_L_8
	{SENSOR_WORD_WRITE,0xC8A4, 0x01BF} , 	// CAM_AWB_CCM_M_0
	{SENSOR_WORD_WRITE,0xC8A6, 0xFF01} , 	// CAM_AWB_CCM_M_1
	{SENSOR_WORD_WRITE,0xC8A8, 0xFFF3} , 	// CAM_AWB_CCM_M_2
	{SENSOR_WORD_WRITE,0xC8AA, 0xFF75} , 	// CAM_AWB_CCM_M_3
	{SENSOR_WORD_WRITE,0xC8AC, 0x0198} , 	// CAM_AWB_CCM_M_4
	{SENSOR_WORD_WRITE,0xC8AE, 0xFFFD} , 	// CAM_AWB_CCM_M_5
	{SENSOR_WORD_WRITE,0xC8B0, 0xFF9A} , 	// CAM_AWB_CCM_M_6
	{SENSOR_WORD_WRITE,0xC8B2, 0xFEE7} , 	// CAM_AWB_CCM_M_7
	{SENSOR_WORD_WRITE,0xC8B4, 0x02A8} , 	// CAM_AWB_CCM_M_8
	{SENSOR_WORD_WRITE,0xC8B6, 0x01D9} , 	// CAM_AWB_CCM_R_0
	{SENSOR_WORD_WRITE,0xC8B8, 0xFF26} , 	// CAM_AWB_CCM_R_1
	{SENSOR_WORD_WRITE,0xC8BA, 0xFFF3} , 	// CAM_AWB_CCM_R_2
	{SENSOR_WORD_WRITE,0xC8BC, 0xFFB3} , 	// CAM_AWB_CCM_R_3
	{SENSOR_WORD_WRITE,0xC8BE, 0x0132} , 	// CAM_AWB_CCM_R_4
	{SENSOR_WORD_WRITE,0xC8C0, 0xFFE8} , 	// CAM_AWB_CCM_R_5
	{SENSOR_WORD_WRITE,0xC8C2, 0xFFDA} , 	// CAM_AWB_CCM_R_6
	{SENSOR_WORD_WRITE,0xC8C4, 0xFECD} , 	// CAM_AWB_CCM_R_7
	{SENSOR_WORD_WRITE,0xC8C6, 0x02C2} , 	// CAM_AWB_CCM_R_8
	{SENSOR_WORD_WRITE,0xC8C8, 0x0075} , 	// CAM_AWB_CCM_L_RG_GAIN
	{SENSOR_WORD_WRITE,0xC8CA, 0x011C} , 	// CAM_AWB_CCM_L_BG_GAIN
	{SENSOR_WORD_WRITE,0xC8CC, 0x009A} , 	// CAM_AWB_CCM_M_RG_GAIN
	{SENSOR_WORD_WRITE,0xC8CE, 0x0105} , 	// CAM_AWB_CCM_M_BG_GAIN
	{SENSOR_WORD_WRITE,0xC8D0, 0x00A4} , 	// CAM_AWB_CCM_R_RG_GAIN
	{SENSOR_WORD_WRITE,0xC8D2, 0x00AC} , 	// CAM_AWB_CCM_R_BG_GAIN
	{SENSOR_WORD_WRITE,0xC8D4, 0x0A8C} , 	// CAM_AWB_CCM_L_CTEMP
	{SENSOR_WORD_WRITE,0xC8D6, 0x0F0A} , 	// CAM_AWB_CCM_M_CTEMP
	{SENSOR_WORD_WRITE,0xC8D8, 0x1964} , 	// CAM_AWB_CCM_R_CTEMP

	// LOAD=AWB
	{SENSOR_WORD_WRITE,0xC914, 0x0000} , 	// CAM_STAT_AWB_CLIP_WINDOW_XSTART
	{SENSOR_WORD_WRITE,0xC916, 0x0000} , 	// CAM_STAT_AWB_CLIP_WINDOW_YSTART
	{SENSOR_WORD_WRITE,0xC918, 0x04FF} , 	// CAM_STAT_AWB_CLIP_WINDOW_XEND
	{SENSOR_WORD_WRITE,0xC91A, 0x03BF} , 	// CAM_STAT_AWB_CLIP_WINDOW_YEND
	{SENSOR_WORD_WRITE,0xC904, 0x003B} , 	// CAM_AWB_AWB_XSHIFT_PRE_ADJ
	{SENSOR_WORD_WRITE,0xC906, 0x0041} , 	// CAM_AWB_AWB_YSHIFT_PRE_ADJ
	{SENSOR_BYTE_WRITE,0xC8F2, 0x03} , 	// CAM_AWB_AWB_XSCALE
	{SENSOR_BYTE_WRITE,0xC8F3, 0x02} , 	// CAM_AWB_AWB_YSCALE
	{SENSOR_WORD_WRITE,0xC906, 0x003C} , 	// CAM_AWB_AWB_YSHIFT_PRE_ADJ
	{SENSOR_WORD_WRITE,0xC8F4, 0x0000} , 	// CAM_AWB_AWB_WEIGHTS_0
	{SENSOR_WORD_WRITE,0xC8F6, 0x0000} , 	// CAM_AWB_AWB_WEIGHTS_1
	{SENSOR_WORD_WRITE,0xC8F8, 0x0000} , 	// CAM_AWB_AWB_WEIGHTS_2
	{SENSOR_WORD_WRITE,0xC8FA, 0xE724} , 	// CAM_AWB_AWB_WEIGHTS_3
	{SENSOR_WORD_WRITE,0xC8FC, 0x1583} , 	// CAM_AWB_AWB_WEIGHTS_4
	{SENSOR_WORD_WRITE,0xC8FE, 0x2045} , 	// CAM_AWB_AWB_WEIGHTS_5
	{SENSOR_WORD_WRITE,0xC900, 0x05DC} , 	// CAM_AWB_AWB_WEIGHTS_6
	{SENSOR_WORD_WRITE,0xC902, 0x007C} , 	// CAM_AWB_AWB_WEIGHTS_7
	{SENSOR_BYTE_WRITE,0xC90C, 0x80} , 	// CAM_AWB_K_R_L
	{SENSOR_BYTE_WRITE,0xC90D, 0x80} , 	// CAM_AWB_K_G_L
	{SENSOR_BYTE_WRITE,0xC90E, 0x80} , 	// CAM_AWB_K_B_L
	{SENSOR_BYTE_WRITE,0xC90F, 0x88} , 	// CAM_AWB_K_R_R
	{SENSOR_BYTE_WRITE,0xC910, 0x80} , 	// CAM_AWB_K_G_R
	{SENSOR_BYTE_WRITE,0xC911, 0x80} , 	// CAM_AWB_K_B_R

	// LOAD=Step7-CPIPE_Preference
	//{SENSOR_WORD_WRITE,0xC926, 0x0020} , 	// CAM_LL_START_BRIGHTNESS
	// -- Rev3, 03082011, Alias
	// changed for fixing AE test at LV6
	{SENSOR_WORD_WRITE,0xC926, 0x0060} , 	// CAM_LL_START_BRIGHTNESS
	{SENSOR_WORD_WRITE,0xC928, 0x009A} , 	// CAM_LL_STOP_BRIGHTNESS
	{SENSOR_WORD_WRITE,0xC946, 0x0070} , 	// CAM_LL_START_GAIN_METRIC
	{SENSOR_WORD_WRITE,0xC948, 0x00F3} , 	// CAM_LL_STOP_GAIN_METRIC
	//{SENSOR_WORD_WRITE,0xC952, 0x0020} , 	// CAM_LL_START_TARGET_LUMA_BM
	// -- Rev3, 03082011, Alias
	// changed for fixing AE test at LV6
	{SENSOR_WORD_WRITE,0xC952, 0x0060} , 	// CAM_LL_START_TARGET_LUMA_BM
	{SENSOR_WORD_WRITE,0xC954, 0x009A} , 	// CAM_LL_STOP_TARGET_LUMA_BM
	{SENSOR_BYTE_WRITE,0xC92A, 0x80} , 	// CAM_LL_START_SATURATION
	{SENSOR_BYTE_WRITE,0xC92B, 0x4B} , 	// CAM_LL_END_SATURATION
	{SENSOR_BYTE_WRITE,0xC92C, 0x00} , 	// CAM_LL_START_DESATURATION
	{SENSOR_BYTE_WRITE,0xC92D, 0xFF} , 	// CAM_LL_END_DESATURATION
	{SENSOR_BYTE_WRITE,0xC92E, 0x3C} , 	// CAM_LL_START_DEMOSAIC
	{SENSOR_BYTE_WRITE,0xC92F, 0x02} , 	// CAM_LL_START_AP_GAIN
	{SENSOR_BYTE_WRITE,0xC930, 0x06} , 	// CAM_LL_START_AP_THRESH
	{SENSOR_BYTE_WRITE,0xC931, 0x64} , 	// CAM_LL_STOP_DEMOSAIC
	{SENSOR_BYTE_WRITE,0xC932, 0x01} , 	// CAM_LL_STOP_AP_GAIN
	{SENSOR_BYTE_WRITE,0xC933, 0x0C} , 	// CAM_LL_STOP_AP_THRESH
	{SENSOR_BYTE_WRITE,0xC934, 0x3C} , 	// CAM_LL_START_NR_RED
	{SENSOR_BYTE_WRITE,0xC935, 0x3C} , 	// CAM_LL_START_NR_GREEN
	{SENSOR_BYTE_WRITE,0xC936, 0x3C} , 	// CAM_LL_START_NR_BLUE
	{SENSOR_BYTE_WRITE,0xC937, 0x0F} , 	// CAM_LL_START_NR_THRESH
	{SENSOR_BYTE_WRITE,0xC938, 0x64} , 	// CAM_LL_STOP_NR_RED
	{SENSOR_BYTE_WRITE,0xC939, 0x64} , 	// CAM_LL_STOP_NR_GREEN
	{SENSOR_BYTE_WRITE,0xC93A, 0x64} , 	// CAM_LL_STOP_NR_BLUE
	{SENSOR_BYTE_WRITE,0xC93B, 0x32} , 	// CAM_LL_STOP_NR_THRESH
	// Rev3, 03082011, Alias
	// changed for fixing AE test at LV6
	//{SENSOR_WORD_WRITE,0xC93C, 0x0020} , 	// CAM_LL_START_CONTRAST_BM
	{SENSOR_WORD_WRITE,0xC93C, 0x0060} , 	// CAM_LL_START_CONTRAST_BM
	{SENSOR_WORD_WRITE,0xC93E, 0x009A} , 	// CAM_LL_STOP_CONTRAST_BM
	{SENSOR_WORD_WRITE,0xC940, 0x00DC} , 	// CAM_LL_GAMMA
	{SENSOR_BYTE_WRITE,0xC942, 0x38} , 	// CAM_LL_START_CONTRAST_GRADIENT
	{SENSOR_BYTE_WRITE,0xC943, 0x30} , 	// CAM_LL_STOP_CONTRAST_GRADIENT
	{SENSOR_BYTE_WRITE,0xC944, 0x50} , 	// CAM_LL_START_CONTRAST_LUMA_PERCENTAGE
	{SENSOR_BYTE_WRITE,0xC945, 0x19} , 	// CAM_LL_STOP_CONTRAST_LUMA_PERCENTAGE
	//
	// Rev3, 03082011, Alias
	// changed for fixing FTB
	//{SENSOR_WORD_WRITE,0xC94A, 0x0230} , 	// CAM_LL_START_FADE_TO_BLACK_LUMA
	{SENSOR_WORD_WRITE,0xC94A, 0x00F0} , 	// CAM_LL_START_FADE_TO_BLACK_LUMA
	{SENSOR_WORD_WRITE,0xC94C, 0x0010} , 	// CAM_LL_STOP_FADE_TO_BLACK_LUMA
	{SENSOR_WORD_WRITE,0xC94E, 0x01CD} , 	// CAM_LL_CLUSTER_DC_TH_BM
	{SENSOR_BYTE_WRITE,0xC950, 0x05} , 	// CAM_LL_CLUSTER_DC_GATE_PERCENTAGE
	{SENSOR_BYTE_WRITE,0xC951, 0x40} , 	// CAM_LL_SUMMING_SENSITIVITY_FACTOR
	// Rev3, 03082011, Alias
	// set AE target to default to 128
	//{SENSOR_BYTE_WRITE,0xC87B, 0x1B} , 	// CAM_AET_TARGET_AVERAGE_LUMA_DARK
	{SENSOR_BYTE_WRITE,0xC87A, 0x42} , 	// CAM_AET_TARGET_AVERAGE_LUMA
	{SENSOR_BYTE_WRITE,0xC87B, 0x21} , 	// CAM_AET_TARGET_AVERAGE_LUMA_DARK
	{SENSOR_BYTE_WRITE,0xC878, 0x0E} , 	// CAM_AET_AEMODE
	{SENSOR_WORD_WRITE,0xC890, 0x0080} , 	// CAM_AET_TARGET_GAIN
	{SENSOR_WORD_WRITE,0xC886, 0x0100} , 	// CAM_AET_AE_MAX_VIRT_AGAIN
	// Rev3, 03082011, Alias
	// modified for 2'nd BL
	//{SENSOR_WORD_WRITE,0xC87C, 0x005A} , 	// CAM_AET_BLACK_CLIPPING_TARGET
	{SENSOR_WORD_WRITE,0xC87C, 0x0010} , 	// CAM_AET_BLACK_CLIPPING_TARGET
	{SENSOR_BYTE_WRITE,0xB42A, 0x05} , 	// CCM_DELTA_GAIN
	{SENSOR_BYTE_WRITE,0xA80A, 0x20} , 	// AE_TRACK_AE_TRACKING_DAMPENING_SPEED
	// LOAD=Step8-Features
	{SENSOR_WORD_WRITE,0x098E, 0x0000} , 	// LOGICAL_ADDRESS_ACCESS
	//{SENSOR_WORD_WRITE,0xC984, 0x8040} , 	// CAM_PORT_OUTPUT_CONTROL
	{SENSOR_WORD_WRITE,0xC984, 0x8041} , 	// CAM_PORT_OUTPUT_CONTROL
	{SENSOR_WORD_WRITE,0x001E, 0x0777} , 	// PAD_SLEW
	//+
	{SENSOR_WORD_WRITE,0x098E, 0xDC00} , 	// LOGICAL_ADDRESS_ACCESS [SYSMGR_NEXT_STATE]
	{SENSOR_BYTE_WRITE,0xDC00, 0x28} , 	// SYSMGR_NEXT_STAGE
	{SENSOR_WAIT_MS, 0,50} , //WK+
	{SENSOR_WORD_WRITE,0x0080, 0x8002} , // COMMAND_REGISTER
	//-
	{SENSOR_TABLE_END, 0x0000}
};

static struct msm_camera_i2c_reg_conf mi1040_start_settings[] = {
	{0x3C40, 0xAC34},  /* streaming on */
};

static struct msm_camera_i2c_reg_conf mi1040_stop_settings[] = {
	{0x3C40, 0xAC36},  /* streaming off*/
};

//It's reference to mt9m114_v4l2
static struct msm_camera_i2c_reg_conf mi1040_960p_settings[] = {
	{0x098E, 0x4868,},// LOGICAL_ADDRESS_ACCESS [CAM_OUTPUT_WIDTH]
	{0xC868, 0x0500,},// CAM_OUTPUT_WIDTH
	{0xC86A, 0x03C0,},// CAM_OUTPUT_HEIGHT
	{0xC85C, 0x03 , MSM_CAMERA_I2C_BYTE_DATA},  // CAM_CROP_CROPMODE
	{0xC854, 0x0000,},// CAM_CROP_WINDOW_XOFFSET
	{0xC856, 0x0000,},// CAM_CROP_WINDOW_YOFFSET
	{0xC858, 0x0500,},// CAM_CROP_WINDOW_WIDTH
	{0xC85A, 0x03C0,},// CAM_CROP_WINDOW_HEIGHT
	{0xC88C, 0x1E02,},// CAM_AET_MAX_FRAME_RATE
	{0xC88E, 0x1E02,},// CAM_AET_MIN_FRAME_RATE
	{0xC914, 0x0000,},// CAM_STAT_AWB_CLIP_WINDOW_XSTART
	{0xC916, 0x0000,},// CAM_STAT_AWB_CLIP_WINDOW_YSTART
	{0xC918, 0x04FF,},// CAM_STAT_AWB_CLIP_WINDOW_XEND
	{0xC91A, 0x03BF,},// CAM_STAT_AWB_CLIP_WINDOW_YEND
	{0xC91C, 0x0000,},// CAM_STAT_AE_INITIAL_WINDOW_XSTART
	{0xC91E, 0x0000,},// CAM_STAT_AE_INITIAL_WINDOW_YSTART
	{0xC920, 0x00FF,},// CAM_STAT_AE_INITIAL_WINDOW_XEND
	{0xC922, 0x00BF,},// CAM_STAT_AE_INITIAL_WINDOW_YEND
	{0xDC00, 0x28, MSM_CAMERA_I2C_BYTE_DATA},  // SYSMGR_NEXT_STATE
	// delay=50
	{0x0080, 0x8002,},// COMMAND_REGISTER
};

static struct msm_camera_i2c_reg_conf mi1040_recommend_settings[] = {
	{0x301A, 0x0234,},
	{0x098E, 0x0000,},
	{0xC97E, 0x01  , MSM_CAMERA_I2C_BYTE_DATA,},
	{0xC980, 0x0120,},
	{0xC982, 0x0700,},
	{0x98E,  0     ,},
	{0xC800, 0x0004,},
	{0xC802, 0x0004,},
	{0xC804, 0x03CB,},
	{0xC806, 0x050B,},
	{0xC808, 0x02DC,},
	{0xC80A, 0x6C00,},
	{0xC80C, 0x0001,},
	{0xC80E, 0x00DB,},
	{0xC810, 0x05B3,},
	{0xC812, 0x03EE,},
	{0xC814, 0x0636,},
	{0xC816, 0x0060,},
	{0xC818, 0x03C3,},
	{0xC834, 0x0000,},
	{0xC854, 0x0000,},
	{0xC856, 0x0000,},
	{0xC858, 0x0500,},
	{0xC85A, 0x03C0,},
	{0xC85C, 0x03  , MSM_CAMERA_I2C_BYTE_DATA,},
	{0xC868, 0x0500,},
	{0xC86A, 0x03C0,},
	{0xC88C, 0x1E02,},
	{0xC88E, 0x0780,},
	{0xC914, 0x0000,},
	{0xC916, 0x0000,},
	{0xC918, 0x04FF, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC91A, 0x03BF,},
	{0xC91C, 0x0000,},
	{0xC91E, 0x0000, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC920, 0x00FF,},
	{0xC922, 0x00BF,},
	{0xE801, 0x00  , MSM_CAMERA_I2C_BYTE_DATA,},
	{0x316A, 0x8270,},
	{0x316C, 0x8270,},
	{0x3ED0, 0x2305,},
	{0x3ED2, 0x77CF,},
	{0x316E, 0x8202,},
	{0x3180, 0x87FF,},
	{0x30D4, 0x6080,},
	{0xA802, 0x0008,},
	{0x3E14, 0xFF39,},
	{0x098E, 0x495E,},
	{0xC95E, 0x0002,},
	{0x3640, 0x0230,},
	{0x3642, 0xEF67,},
	{0x3644, 0x6590,},
	{0x3646, 0x178D,},
	{0x3648, 0x2031,},
	{0x364A, 0x00B0,},
	{0x364C, 0xDF48,},
	{0x364E, 0x11B1,},
	{0x3650, 0x2E6E,},
	{0x3652, 0x37B1,},
	{0x3654, 0x00F0,},
	{0x3656, 0x67AB,},
	{0x3658, 0x37B0,},
	{0x365A, 0x778D,},
	{0x365C, 0x7FB0,},
	{0x365E, 0x00D0,},
	{0x3660, 0xA8C9,},
	{0x3662, 0x7270,},
	{0x3664, 0x04ED,},
	{0x3666, 0x14F1,},
	{0x3680, 0x54EB,},
	{0x3682, 0x244C,},
	{0x3684, 0x47AE,},
	{0x3686, 0x34CC,},
	{0x3688, 0xD4CC,},
	{0x368A, 0x1F0B,},
	{0x368C, 0x6CEC,},
	{0x368E, 0x67CE,},
	{0x3690, 0x282A,},
	{0x3692, 0xCDAD,},
	{0x3694, 0x200C,},
	{0x3696, 0x858A,},
	{0x3698, 0x124C,},
	{0x369A, 0xD46D,},
	{0x369C, 0xA60F,},
	{0x369E, 0x152C,},
	{0x36A0, 0x0DAB,},
	{0x36A2, 0xA96A,},
	{0x36A4, 0xB26C,},
	{0x36A6, 0xA40F,},
	{0x36C0, 0x77D0,},
	{0x36C2, 0x2E8B,},
	{0x36C4, 0x7E92,},
	{0x36C6, 0x4C4E,},
	{0x36C8, 0xEA71,},
	{0x36CA, 0x12F1,},
	{0x36CC, 0x1F8D,},
	{0x36CE, 0x2013,},
	{0x36D0, 0x1BB0,},
	{0x36D2, 0xE072,},
	{0x36D4, 0x4010,},
	{0x36D6, 0x060A,},
	{0x36D8, 0x6F92,},
	{0x36DA, 0x2CCF,},
	{0x36DC, 0x9F32,},
	{0x36DE, 0x6FD0,},
	{0x36E0, 0x0B6C,},
	{0x36E2, 0x7672,},
	{0x36E4, 0x10CE,},
	{0x36E6, 0x8092,},
	{0x3700, 0x662A,},
	{0x3702, 0xA44C,},
	{0x3704, 0xFA6F,},
	{0x3706, 0xA10F,},
	{0x3708, 0x3690,},
	{0x370A, 0x696C,},
	{0x370C, 0xA5CE,},
	{0x370E, 0xCC4E,},
	{0x3710, 0xD989,},
	{0x3712, 0x534F,},
	{0x3714, 0x68AD,},
	{0x3716, 0xE48C,},
	{0x3718, 0xBE90,},
	{0x371A, 0x0AEF,},
	{0x371C, 0x4012,},
	{0x371E, 0x20ED,},
	{0x3720, 0xF76D,},
	{0x3722, 0xBD91,},
	{0x3724, 0x186E,},
	{0x3726, 0x0C73,},
	{0x3740, 0x0E51,},
	{0x3742, 0x31EF,},
	{0x3744, 0x5E73,},
	{0x3746, 0x8D70,},
	{0x3748, 0xEC56,},
	{0x374A, 0x23B1,},
	{0x374C, 0x2290,},
	{0x374E, 0x79B3,},
	{0x3750, 0xE5F2,},
	{0x3752, 0x9057,},
	{0x3754, 0x6550,},
	{0x3756, 0x1470,},
	{0x3758, 0x0673,},
	{0x375A, 0x92F2,},
	{0x375C, 0xAF76,},
	{0x375E, 0x1031,},
	{0x3760, 0x062F,},
	{0x3762, 0x72B3,},
	{0x3764, 0x15CC,},
	{0x3766, 0xEBB6,},
	{0x3784, 0x026C,},
	{0x3782, 0x01CC,},
	{0x37C0, 0xC00A,},
	{0x37C2, 0xE7E9,},
	{0x37C4, 0x806A,},
	{0x37C6, 0xDAE9,},
	{0x098E, 0x0000,},
	{0xC960, 0x0A8C,},
	{0xC962, 0x7DFC,},
	{0xC964, 0x67A8,},
	{0xC966, 0x7E28,},
	{0xC968, 0x7E42,},
	{0xC96A, 0x0FA0,},
	{0xC96C, 0x7E80,},
	{0xC96E, 0x8798,},
	{0xC970, 0x7F25,},
	{0xC972, 0x7EFF,},
	{0xC974, 0x1964,},
	{0xC976, 0x7EF8,},
	{0xC978, 0x78E4,},
	{0xC97A, 0x80EC,},
	{0xC97C, 0x79AE,},
	{0xC95E, 0x0003,},
	{0xC892, 0x0267,},
	{0xC894, 0xFF1A,},
	{0xC896, 0xFFB3,},
	{0xC898, 0xFF80,},
	{0xC89A, 0x0166,},
	{0xC89C, 0x0003,},
	{0xC89E, 0xFF9A,},
	{0xC8A0, 0xFEB4,},
	{0xC8A2, 0x024D,},
	{0xC8A4, 0x01BF,},
	{0xC8A6, 0xFF01,},
	{0xC8A8, 0xFFF3,},
	{0xC8AA, 0xFF75,},
	{0xC8AC, 0x0198,},
	{0xC8AE, 0xFFFD,},
	{0xC8B0, 0xFF9A,},
	{0xC8B2, 0xFEE7,},
	{0xC8B4, 0x02A8,},
	{0xC8B6, 0x0100,},
	{0xC8B8, 0xFF72,},
	{0xC8BA, 0xFFF8,},
	{0xC8BC, 0xFFA8,},
	{0xC8BE, 0x011D,},
	{0xC8C0, 0xFFFB,},
	{0xC8C2, 0xFFE3,},
	{0xC8C4, 0xFE48,},
	{0xC8C6, 0x0356,},
	{0xC8DA, 0x004D,},
	{0xC8DC, 0x0096,},
	{0xC8DE, 0x001D,},
	{0xC8E0, 0x004D,},
	{0xC8E2, 0x0096,},
	{0xC8E4, 0x001D,},
	{0xC8E6, 0x004D,},
	{0xC8E8, 0x0096,},
	{0xC8EA, 0x001D,},
	{0xC8C8, 0x0075,},
	{0xC8CA, 0x011C,},
	{0xC8CC, 0x009A,},
	{0xC8CE, 0x0105,},
	{0xC8D0, 0x00A4,},
	{0xC8D2, 0x00AC,},
	{0xC8D4, 0x0A8C,},
	{0xC8D6, 0x0F0A,},
	{0xC8D8, 0x1964,},
	{0xC914, 0x0000,},
	{0xC916, 0x0000,},
	{0xC918, 0x04FF,},
	{0xC91A, 0x03BF,},
	{0xC8F2, 0x04  , MSM_CAMERA_I2C_BYTE_DATA,},
	{0xC8F3, 0x02  , MSM_CAMERA_I2C_BYTE_DATA,},
	{0xC904, 0x0036,},
	{0xC906, 0x0040,},
	{0xC8F4, 0x0000,},
	{0xC8F6, 0x0000,},
	{0xC8F8, 0x0000,},
	{0xC8FA, 0xE724,},
	{0xC8FC, 0x1583,},
	{0xC8FE, 0x2045,},
	{0xC900, 0x05DC,},
	{0xC902, 0x007C,},
	{0xC90A, 0x1388,},
	{0xC90C, 0x80  , MSM_CAMERA_I2C_BYTE_DATA,},
	{0xC90D, 0x90  , MSM_CAMERA_I2C_BYTE_DATA,},
	{0xC90E, 0x90  , MSM_CAMERA_I2C_BYTE_DATA,},
	{0xC90F, 0x80  , MSM_CAMERA_I2C_BYTE_DATA,},
	{0xC910, 0x88  , MSM_CAMERA_I2C_BYTE_DATA,},
	{0xC911, 0x80  , MSM_CAMERA_I2C_BYTE_DATA,},
	{0x098E, 0xAC06,},
	{0xAC06, 0x62  , MSM_CAMERA_I2C_BYTE_DATA,},
	{0xAC07, 0x66  , MSM_CAMERA_I2C_BYTE_DATA,},
	{0xAC08, 0x64  , MSM_CAMERA_I2C_BYTE_DATA,},
	{0xAC09, 0x66  , MSM_CAMERA_I2C_BYTE_DATA,},
	{0xC926, 0x0060,},
	{0xC928, 0x009A,},
	{0xC946, 0x0070,},
	{0xC948, 0x00F3,},
	{0xC952, 0x0060,},
	{0xC954, 0x009A,},
	{0xC95A, 0x04  , MSM_CAMERA_I2C_BYTE_DATA,},
	{0xC92A, 0x5A  , MSM_CAMERA_I2C_BYTE_DATA,},
	{0xC92B, 0x3C  , MSM_CAMERA_I2C_BYTE_DATA,},
	{0xC92C, 0x00  , MSM_CAMERA_I2C_BYTE_DATA,},
	{0xC92D, 0xFF  , MSM_CAMERA_I2C_BYTE_DATA,},
	{0xC92E, 0x32  , MSM_CAMERA_I2C_BYTE_DATA,},
	{0xC92F, 0x04  , MSM_CAMERA_I2C_BYTE_DATA,},
	{0xC930, 0x02  , MSM_CAMERA_I2C_BYTE_DATA,},
	{0xC931, 0x78  , MSM_CAMERA_I2C_BYTE_DATA,},
	{0xC932, 0x01  , MSM_CAMERA_I2C_BYTE_DATA,},
	{0xC933, 0x09  , MSM_CAMERA_I2C_BYTE_DATA,},
	{0xC934, 0x32  , MSM_CAMERA_I2C_BYTE_DATA,},
	{0xC935, 0x14  , MSM_CAMERA_I2C_BYTE_DATA,,},
	{0xC936, 0x32  , MSM_CAMERA_I2C_BYTE_DATA,},
	{0xC937, 0x14  , MSM_CAMERA_I2C_BYTE_DATA,},
	{0xC938, 0x50  , MSM_CAMERA_I2C_BYTE_DATA,},
	{0xC939, 0x50  , MSM_CAMERA_I2C_BYTE_DATA,},
	{0xC93A, 0x50  , MSM_CAMERA_I2C_BYTE_DATA,,},
	{0xC93B, 0x50  , MSM_CAMERA_I2C_BYTE_DATA,},
	{0xC93C, 0x0005,},
	{0xC93E, 0x0358,},
	{0xC940, 0x00DC,},
	{0xC942, 0x4B  , MSM_CAMERA_I2C_BYTE_DATA,},
	{0xC943, 0x3C  , MSM_CAMERA_I2C_BYTE_DATA,},
	{0xC944, 0x22  , MSM_CAMERA_I2C_BYTE_DATA,},
	{0xC945, 0x19  , MSM_CAMERA_I2C_BYTE_DATA,},
	{0x098E, 0xC942,},
	{0x098E, 0xC92A,},
	{0xC94A, 0x00F0,},
	{0xC94C, 0x0010,},
	{0xC94E, 0x01CD,},
	{0xC950, 0x05  , MSM_CAMERA_I2C_BYTE_DATA,},
	{0xC951, 0x40  , MSM_CAMERA_I2C_BYTE_DATA,},
	{0xC87A, 0x37  , MSM_CAMERA_I2C_BYTE_DATA,},
	{0xC87B, 0x32  , MSM_CAMERA_I2C_BYTE_DATA,},
	{0xC878, 0x0E  , MSM_CAMERA_I2C_BYTE_DATA,},
	{0xC890, 0x0080,},
	{0xC882, 0x0100,},
	{0xC886, 0x0100,},
	{0xA404, 0x02  , MSM_CAMERA_I2C_BYTE_DATA,},
	{0xC87C, 0x0016,},
	{0xB42A, 0x05  , MSM_CAMERA_I2C_BYTE_DATA,},
	{0xA80A, 0x10  , MSM_CAMERA_I2C_BYTE_DATA,},
	{0x326C, 0x0A08,},
	{0x098E, 0x0000,},
	{0xC984, 0x8040,},
	{0x001E, 0x0777,},
	{0xC984, 0x8041,},
	{0xC988, 0x0F00,},
	{0xC98A, 0x0B07,},
	{0xC98C, 0x0D01,},
	{0xC98E, 0x071D,},
	{0xC990, 0x0006,},
	{0xC992, 0x0A0C,},
	{0x3C5A, 0x0009,},
	{0xDC00, 0x40  , MSM_CAMERA_I2C_BYTE_DATA,},
	{0x0080, 0x8002,},
//SERIAL_REG=0xCA, 0x00, 0x8000, 8:16
//SERIAL_REG=0xCA, 0x00, 0x0006, 8:16
//LOAD=Leave Suspend
//REG= 0xDC00, 0x34 	// SYSMGR_NEXT_STATE
//REG= 0x0080, 0x8002 	// COMMAND_REGISTER
//POLL_REG= 0x0080, 0x0002, !=0, DELAY=10, TIMEOUT=100	// Polling R0x0080[1] till "0".
	{0x098E, 0x2802	,},
	{0xA802, 0x0008	,},
	{0xC908, 0x01 	, MSM_CAMERA_I2C_BYTE_DATA,},
	{0xC879, 0x01 	, MSM_CAMERA_I2C_BYTE_DATA,},
	{0xC909, 0x02 	, MSM_CAMERA_I2C_BYTE_DATA,},
	{0xA80A, 0x18 	, MSM_CAMERA_I2C_BYTE_DATA,},
	{0xA80B, 0x18 	, MSM_CAMERA_I2C_BYTE_DATA,},
	{0xAC16, 0x18 	, MSM_CAMERA_I2C_BYTE_DATA,},
	{0xC878, 0x0E   , MSM_CAMERA_I2C_BYTE_DATA,},
	{0xDC00, 0x28   , MSM_CAMERA_I2C_BYTE_DATA,},
	{0x0080, 0x8002 ,},
};
#endif
#endif
