package latexDraw.ui;

import java.awt.Color;
import java.awt.Component;
import java.awt.Cursor;
import java.awt.Point;
import java.awt.event.*;
import java.awt.geom.Arc2D;
import java.io.File;
import java.io.ObjectInputStream;

import javax.swing.*;

import latexDraw.figures.*;
import latexDraw.figures.properties.Arrowable;
import latexDraw.lang.LaTeXDrawLang;
import latexDraw.psTricks.PSTricksConstants;
import latexDraw.ui.dialog.ExceptionFrameDialog;
import latexDraw.util.LaTeXDrawCursor;
import latexDraw.util.LaTeXDrawNamespace;
import latexDraw.util.LaTeXDrawPoint2D;
import latexDraw.util.LaTeXDrawResources;
import latexDraw.util.UndoRedoManager;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

/** 
 * 
 * This class defines the panel which contains the drawing which is used to generate the psTricks code.<br>
 *<br>
 * This file is part of LaTeXDraw.<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 *<br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  any later version.<br>
 *<br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE. See the GNU General Public License for more details.<br>
 *<br>
 * 01/20/06<br>
 * @author Arnaud BLOUIN<br>
 * @version 2.0.0<br>
 */
public final class DrawPanel extends JScrollPane implements MouseListener, MouseMotionListener, ActionListener, MouseWheelListener
{
	private static final long serialVersionUID = 1L;

	/** Contains all figures of the current project */
	protected DrawContainer draw;
	
	/** Correspond to the codepanel of the program */
	protected CodePanel codePanel;
	
	/** Correspond to the current figure in drawing */
	protected Figure currentFigure;
	
	/** The former position of the cursor. Useful for 
	 * figure shifting */
	protected Point formerPoint;
	
	/** The frame parent of the drawPanel @see LaTeXDrawFrame */
	protected LaTeXDrawFrame frameParent;
	
	/** The pop-up menu which allows to do some actions about general
	 * Functions (draw a figure, select a figure,...) */
	protected JPopupMenu popupGeneral;
    
    /** This pop-up menu allows the user to do some actions about
     * the selected figure (delete, properties, ...) */
    protected JPopupMenu popupFigure;
    
    /** If not null, it represents a figure before being moved */
    protected Draw cloneForDragged;

    protected LaTeXDrawRectangle bordersSelection;
    
    private JRadioButtonMenuItem menuLine;
    private JRadioButtonMenuItem menuDot;
    private JRadioButtonMenuItem menuRect;
    private JRadioButtonMenuItem menuCircle;
    private JRadioButtonMenuItem menuEllipse;
    private JRadioButtonMenuItem menuSquare;
    private JRadioButtonMenuItem menuRhombus;
    private JRadioButtonMenuItem menuPoly;
    private JRadioButtonMenuItem menuArc;
    private JRadioButtonMenuItem menuWedge;
    private JRadioButtonMenuItem menuTriangle;
    private JRadioButtonMenuItem menuSelect;
    private JRadioButtonMenuItem menuText;
    private JRadioButtonMenuItem menuPointsAkin;
    private JRadioButtonMenuItem menuGrid;
    private JRadioButtonMenuItem menuAxes;
    private JRadioButtonMenuItem menuChord;
    private JRadioButtonMenuItem menuBezier;
    private JRadioButtonMenuItem menuClosedBezier;
    private JRadioButtonMenuItem menuJoinedLines;
    private JMenuItem menuDel;
    private JMenuItem menuProperties;
    private JMenuItem menuJoin;
    private JMenuItem menuSeparate;
    public JMenuItem menuUpdateToGrid;
    private JCheckBoxMenuItem menuRotate;

	
    /** Allows to know if the user must select a figure to put
     * the selected figure behind the clicked figure */
    private boolean onWaitingClicDispositionBehind;
    
    /** Allows to know if the user must select a figure to put
     * the selected figure in front of the clicked figure */
    private boolean onWaitingClicDispositionFront;
    
    /** This field allows to know if the last selection was empty or not (useful to the method onDragged). */
    private transient boolean formerSelectionEmpty = true;
    
	public static final String LABEL_UPDATE_TO_GRID = LaTeXDrawLang.getString1_9("DrawPanel.0"); //$NON-NLS-1$
    
    public static final String ACTION_CMD_UPDATETOGRID = "updToGrid";//$NON-NLS-1$
    
	
    
	/**
	 * The second constructor
	 * @param d The drawing of the project (compulsory in order to
	 * call the super-constructor of the @see JScrollPane class to use
	 * elevator
	 */
	public DrawPanel(DrawContainer d, CodePanel codePanel, LaTeXDrawFrame parent) 
	{
    	super(d);	    	
    	
    	d.getGrid().setMagnetic(false);
    	bordersSelection = null;
    	cloneForDragged = null;
    	frameParent = parent;
    	this.codePanel = codePanel;
    	currentFigure = null;
    	draw = d;
    	draw.setParentPanel(this);
		draw.addMouseListener(this);
		draw.addMouseMotionListener(this);
		addMouseWheelListener(this);
		draw.updatePreferredSize(); 
		draw.revalidate(); 
		onWaitingClicDispositionBehind = false;
		onWaitingClicDispositionFront = false;
		createPopupGeneral();
		createPopupFigure();
		setDoubleBuffered(true);
    }
	
	
	
	public boolean isOnRotation()
	{
		return draw.isOnRotation();
	}
	
	
	
	public void setOnRotation(boolean on)
	{
		draw.setOnRotation(on);
		menuRotate.setSelected(on);
	}
	
	
	
	/**
	 * Allows to put the selected figure in the background
	 */
	public void selectedInBackground()
	{
		draw.selectedInBackground();
		if(codePanel.isInAutoUpdate())
			updateCode();
	}
	
	
	
	/**
	 * Allows to put the selected figure in the foreground
	 */
	public void selectedInForeground()
	{
		draw.selectedInForeground();
		if(codePanel.isInAutoUpdate())	
			updateCode();
	}
	
	
	
	
	/**
	 * Allows to zoom in the drawing (up to 450%)
	 */
	public void zoomIn()
	{
		draw.zoomIn();
	}
	
	
	
	/**
	 * Allows to zoom out the drawing (up to 25%)
	 */
	public void zoomOut()
	{
		draw.zoomOut();
	}
	
	
	
	/**
	 * Allows to set the zoom to default
	 */
	public void zoomDefault()
	{
		draw.zoomDefault();
	}
	
	
	
	/**
	 * Allows to get the current zoom
	 * @return The zoom
	 */
	public double getZoom()
	{
		return draw.getZoom();
	}
		
	
	/**
	 * Allows to create the pop-up menu which concern a selected figure
	 */
	private void createPopupFigure()
	{
		popupFigure  = new JPopupMenu();
		
		menuDel =  new JMenuItem(LaTeXDrawResources.LABEL_DELETE, LaTeXDrawResources.delIcon);
		menuDel.setActionCommand(LaTeXDrawResources.LABEL_DELETE);
		menuDel.addActionListener(frameParent.menusListener);
		popupFigure.add(menuDel);
		
		popupFigure.addSeparator();
		
		menuProperties =  new JMenuItem(LaTeXDrawResources.LABEL_FIGURE_PROPERTIES, LaTeXDrawResources.propertiesIcon);
		menuProperties.setActionCommand(LaTeXDrawResources.LABEL_FIGURE_PROPERTIES);
		menuProperties.addActionListener(frameParent.menusListener);
		popupFigure.add(menuProperties);
		
		popupFigure.addSeparator();
		
		JMenuItem menu2 =  new JMenuItem(LaTeXDrawResources.LABEL_FOREGROUND, LaTeXDrawResources.foregroundIcon);
		menu2.setActionCommand(LaTeXDrawResources.LABEL_FOREGROUND);
		menu2.addActionListener(frameParent);
		popupFigure.add(menu2);
		
		menu2 =  new JMenuItem(LaTeXDrawResources.LABEL_BACKGROUND, LaTeXDrawResources.backgroundIcon);
		menu2.setActionCommand(LaTeXDrawResources.LABEL_BACKGROUND);
		menu2.addActionListener(frameParent);
		popupFigure.add(menu2);

		menu2 = new JMenuItem(LaTeXDrawResources.LABEL_FRONT_SEL, LaTeXDrawResources.inFrontOfSelIcon);
		menu2.setActionCommand(LaTeXDrawResources.LABEL_FRONT_SEL);
		menu2.addActionListener(frameParent);
		popupFigure.add(menu2);
		
		menu2 =  new JMenuItem(LaTeXDrawResources.LABEL_BEHIND_SEL, LaTeXDrawResources.behindSelIcon);
		menu2.setActionCommand(LaTeXDrawResources.LABEL_BEHIND_SEL);
		menu2.addActionListener(frameParent);
		popupFigure.add(menu2);
		
		popupFigure.addSeparator();
		
		menuJoin =  new JMenuItem(LaTeXDrawResources.LABEL_JOIN, LaTeXDrawResources.joinIcon);
		menuJoin.setActionCommand(LaTeXDrawResources.LABEL_JOIN);
		menuJoin.addActionListener(frameParent);
		popupFigure.add(menuJoin);
		
		menuSeparate =  new JMenuItem(LaTeXDrawResources.LABEL_SEPARATE, LaTeXDrawResources.separateIcon);
		menuSeparate.setActionCommand(LaTeXDrawResources.LABEL_SEPARATE);
		menuSeparate.addActionListener(frameParent);
		popupFigure.add(menuSeparate);	
		
		popupFigure.addSeparator();
		
		menuRotate =  new JCheckBoxMenuItem(LaTeXDrawResources.LABEL_ROTATE, LaTeXDrawResources.rotateIcon);
		menuRotate.setActionCommand(LaTeXDrawResources.LABEL_ROTATE);
		menuRotate.addActionListener(frameParent);
		popupFigure.add(menuRotate);	
		
		popupFigure.addSeparator();
		
		menuUpdateToGrid =  new JMenuItem(LABEL_UPDATE_TO_GRID, LaTeXDrawResources.emptyIcon);
		menuUpdateToGrid.setActionCommand(ACTION_CMD_UPDATETOGRID);
		menuUpdateToGrid.addActionListener(this);
		popupFigure.add(menuUpdateToGrid);	
		
		popupFigure.addSeparator();
		
		menu2 =  new JMenuItem(LaTeXDrawResources.LABEL_TEMPLATE_EXPORT, LaTeXDrawResources.emptyIcon);
		menu2.setActionCommand(LaTeXDrawResources.LABEL_TEMPLATE_EXPORT);
		menu2.addActionListener(frameParent.menusListener);
		popupFigure.add(menu2);	
	}
	
	
	
	
	
	/**
	 * Allows to create a pop-up menu concerning general actions
	 */
	private void createPopupGeneral()
	{
		ButtonGroup drawGroup = new ButtonGroup();		
		popupGeneral = new JPopupMenu();
		
		menuSelect = new JRadioButtonMenuItem(LaTeXDrawResources.LABEL_SELECT_FIGURE, LaTeXDrawResources.selectIcon);
		menuSelect.setActionCommand(LaTeXDrawResources.LABEL_SELECT_FIGURE);
		menuSelect.addActionListener(frameParent.menusListener);
		menuSelect.setSelected(true);
		popupGeneral.add(menuSelect);
		drawGroup.add(menuSelect);
		
		popupGeneral.addSeparator();
		
		menuLine = new JRadioButtonMenuItem(LaTeXDrawResources.LABEL_LINE, LaTeXDrawResources.lineIcon);
		menuLine.setActionCommand(LaTeXDrawResources.LABEL_LINE);
		menuLine.addActionListener(frameParent.menusListener);
		popupGeneral.add(menuLine);
		drawGroup.add(menuLine);
		
		menuJoinedLines = new JRadioButtonMenuItem(LaTeXDrawResources.LABEL_JOINED_LINES, LaTeXDrawResources.joinedLinesIcon);
		menuJoinedLines.setActionCommand(LaTeXDrawResources.LABEL_JOINED_LINES);
		menuJoinedLines.addActionListener(frameParent.menusListener);
		popupGeneral.add(menuJoinedLines);
		drawGroup.add(menuJoinedLines);
		
		menuPoly = new JRadioButtonMenuItem(LaTeXDrawResources.LABEL_POLY, LaTeXDrawResources.polygonIcon);
		menuPoly.setActionCommand(LaTeXDrawResources.LABEL_POLY);
		menuPoly.addActionListener(frameParent.menusListener);
		popupGeneral.add(menuPoly);
		drawGroup.add(menuPoly);
		
		menuDot = new JRadioButtonMenuItem(LaTeXDrawResources.LABEL_DOT, LaTeXDrawResources.dotIcon);
		menuDot.setActionCommand(LaTeXDrawResources.LABEL_DOT);
		menuDot.addActionListener(frameParent.menusListener);
		popupGeneral.add(menuDot);
		drawGroup.add(menuDot);
		
		menuRect = new JRadioButtonMenuItem(LaTeXDrawResources.LABEL_RECT, LaTeXDrawResources.rectIcon);
		menuRect.setActionCommand(LaTeXDrawResources.LABEL_RECT);
		menuRect.addActionListener(frameParent.menusListener);
		popupGeneral.add(menuRect);
		drawGroup.add(menuRect);
		
		menuSquare = new JRadioButtonMenuItem(LaTeXDrawResources.LABEL_SQUARE, LaTeXDrawResources.squareIcon);
		menuSquare.setActionCommand(LaTeXDrawResources.LABEL_SQUARE);
		menuSquare.addActionListener(frameParent.menusListener);
		popupGeneral.add(menuSquare);
		drawGroup.add(menuSquare);
		
		menuRhombus = new JRadioButtonMenuItem(LaTeXDrawResources.LABEL_RHOMBUS, LaTeXDrawResources.rhombusIcon);
		menuRhombus.setActionCommand(LaTeXDrawResources.LABEL_RHOMBUS);
		menuRhombus.addActionListener(frameParent.menusListener);
		popupGeneral.add(menuRhombus);
		drawGroup.add(menuRhombus);
		
		menuTriangle = new JRadioButtonMenuItem(LaTeXDrawResources.LABEL_TRIANGLE, LaTeXDrawResources.triangleIcon);
		menuTriangle.setActionCommand(LaTeXDrawResources.LABEL_TRIANGLE);
		menuTriangle.addActionListener(frameParent.menusListener);
		popupGeneral.add(menuTriangle);
		drawGroup.add(menuTriangle);
		
		menuEllipse = new JRadioButtonMenuItem(LaTeXDrawResources.LABEL_ELLIPSE, LaTeXDrawResources.ellipseIcon);
		menuEllipse.setActionCommand(LaTeXDrawResources.LABEL_ELLIPSE);
		menuEllipse.addActionListener(frameParent.menusListener);
		popupGeneral.add(menuEllipse);
		drawGroup.add(menuEllipse);
		
		menuCircle = new JRadioButtonMenuItem(LaTeXDrawResources.LABEL_CIRCLE, LaTeXDrawResources.circleIcon);
		menuCircle.setActionCommand(LaTeXDrawResources.LABEL_CIRCLE);
		menuCircle.addActionListener(frameParent.menusListener);
		popupGeneral.add(menuCircle);
		drawGroup.add(menuCircle);
		
		menuArc = new JRadioButtonMenuItem(LaTeXDrawResources.LABEL_ARC, LaTeXDrawResources.arcIcon);
		menuArc.setActionCommand(LaTeXDrawResources.LABEL_ARC);
		menuArc.addActionListener(frameParent.menusListener);
		popupGeneral.add(menuArc);
		drawGroup.add(menuArc);
		
		menuChord = new JRadioButtonMenuItem(LaTeXDrawResources.LABEL_CHORD, LaTeXDrawResources.chordIcon);
		menuChord.setActionCommand(LaTeXDrawResources.LABEL_CHORD);
		menuChord.addActionListener(frameParent.menusListener);
		popupGeneral.add(menuChord);
		drawGroup.add(menuChord);
		
		menuWedge = new JRadioButtonMenuItem(LaTeXDrawResources.LABEL_WEDGE, LaTeXDrawResources.wedgeIcon);
		menuWedge.setActionCommand(LaTeXDrawResources.LABEL_WEDGE);
		menuWedge.addActionListener(frameParent.menusListener);
		popupGeneral.add(menuWedge);
		drawGroup.add(menuWedge);
		
		menuGrid = new JRadioButtonMenuItem(LaTeXDrawResources.LABEL_GRID, LaTeXDrawResources.gridIcon);
		menuGrid.setActionCommand(LaTeXDrawResources.LABEL_GRID);
		menuGrid.addActionListener(frameParent.menusListener);
		popupGeneral.add(menuGrid);
		drawGroup.add(menuGrid);
		
		menuAxes = new JRadioButtonMenuItem(LaTeXDrawResources.LABEL_AXES, LaTeXDrawResources.axesIcon);
		menuAxes.setActionCommand(LaTeXDrawResources.LABEL_AXES);
		menuAxes.addActionListener(frameParent.menusListener);
		popupGeneral.add(menuAxes);
		drawGroup.add(menuAxes);
		
		menuBezier = new JRadioButtonMenuItem(LaTeXDrawResources.LABEL_BEZIER_CURVE, LaTeXDrawResources.bezierCurveIcon);
		menuBezier.setActionCommand(LaTeXDrawResources.LABEL_BEZIER_CURVE);
		menuBezier.addActionListener(frameParent.menusListener);
		popupGeneral.add(menuBezier);
		drawGroup.add(menuBezier);
		
		menuClosedBezier = new JRadioButtonMenuItem(LaTeXDrawResources.LABEL_CLOSED_BEZIER_CURVE, LaTeXDrawResources.closedBezierIcon);
		menuClosedBezier.setActionCommand(LaTeXDrawResources.LABEL_CLOSED_BEZIER_CURVE);
		menuClosedBezier.addActionListener(frameParent.menusListener);
		popupGeneral.add(menuClosedBezier);
		drawGroup.add(menuClosedBezier);
		
		menuPointsAkin = new JRadioButtonMenuItem(LaTeXDrawResources.LABEL_POINTS_AKIN, LaTeXDrawResources.pointsAkinIcon);
		menuPointsAkin.setActionCommand(LaTeXDrawResources.LABEL_POINTS_AKIN);
		menuPointsAkin.addActionListener(frameParent.menusListener);
		popupGeneral.add(menuPointsAkin);
		drawGroup.add(menuPointsAkin);
		
		menuText = new JRadioButtonMenuItem(LaTeXDrawResources.LABEL_TEXT, LaTeXDrawResources.textIcon);
		menuText.setActionCommand(LaTeXDrawResources.LABEL_TEXT);
		menuText.addActionListener(frameParent.menusListener);
		popupGeneral.add(menuText);
		drawGroup.add(menuText);	
	}
	
	
	
	
	
	/**
	 * Allows to select a menu in the general pop-up
	 * @param select The label of the new selected menu
	 */
	public void setMenuSelectedPopupGeneral(String select, boolean state) 
	{
		int i=0, size = popupGeneral.getComponentCount();
		boolean found = false;
		Component c=null;
		
		while(!found && i<size)
		{
			 c = popupGeneral.getComponent(i);
			
			if(c instanceof JMenuItem)
			{
				if(((JMenuItem)c).getActionCommand().equals(select)) 
					 found = true;
				else i++;
			}
			else i++;
		}
		
		if(found && c!=null) 
			((JMenuItem)c).setSelected(state);
	}
	
	
	
	
	
	/**
	 * Allows to get the frame parent of the panel
	 * @return The LaTeXDrawFrame frame parent
	 */
	public LaTeXDrawFrame getFrameParent()
	{
		return frameParent;
	}
	
	
	
	/**
	 * Allows to set the attribute drawing with an another
	 * @param d The new drawing
	 */
	public void setDraw(DrawContainer d)
	{
		draw.copy(d);
		updateDraw(true);
	}
	
	
	
	/**
	 * Allows to get the drawing
	 * @return The drawing
	 */
	public DrawContainer getDraw()
	{
		return draw;
	}
	
	
	
	/**
	 * Exports the drawing as a picture.
	 * @param file The new picture to create.
	 * @param format The extension of the picture to create.
	 * @param compressionRate The compression rate for jpg pictures.
	 */
	public void exportAsPicture(File file, String format, float compressionRate) 
	{
		if(format.equals(LaTeXDrawResources.LABEL_EXPORT_JPG))
			  draw.exportAsJPG(file, compressionRate);
		else if(format.equals(LaTeXDrawResources.LABEL_EXPORT_EPS))	
			draw.exportAsEPS(file);
		else if(format.equals(LaTeXDrawResources.LABEL_EXPORT_PNG))	
			draw.exportAsPNG(file);
		else if(format.equals(LaTeXDrawResources.LABEL_EXPORT_BMP))	
			draw.exportAsBMP(file);
		else if(format.equals(LaTeXDrawResources.LABEL_EXPORT_PPM))	
			draw.exportAsPPMFile(file);
		else if(format.equals(LaTeXDrawResources.LABEL_EXPORT_PDF_LATEX))
			draw.createPDFFile(frameParent.getPathDistribLatex(), file.getPath(), false);
		else if(format.equals(LaTeXDrawResources.LABEL_EXPORT_PDF_LATEX_CROP))
			draw.createPDFFile(frameParent.getPathDistribLatex(), file.getPath(), true);
		else if(format.equals(LaTeXDrawResources.LABEL_EXPORT_EPS_LATEX))	
			draw.createPSFile(frameParent.getPathDistribLatex(), file.getPath());
		else throw new IllegalArgumentException(format);
	}
	
		
		
	
		
	/**
	 * Allows to set the variable isModified
	 * @param state The new state of isModified
	 */
	public void setIsModified(boolean state)
	{
		draw.setIsModified(state);
	}
		

		
		
	
	/**
	 * Allows to get the number of pixels per centimetre
	 * @return The number of pixels per centimetre
	 */
	public int getPixelsPerCm()
	{
		return draw.getPixelsPerCm();
	}
	
	
	
	
	/**
	 * Allows to set the number of pixels per centimetre
	 * @param pPerCm The new number of pixels per centimetre
	 */
	public void setPixelsPerCm(int pPerCm)
	{
		draw.setPixelsPerCm(pPerCm);
		if(codePanel.isInAutoUpdate())
			updateCode();
	}

	
	
	
	/**
	 * Allows to set the value of the attribute
	 * onWaitingClicDispositionFront
	 * @param state Its new state
	 */
	public void setOnWaitingClicDispositionFront(boolean state)
	{
		onWaitingClicDispositionFront = state;
	}
	
	
	
	/**
	 * Allows to set the value of the attribute
	 * onWaitingClicDispositionBehind
	 * @param state Its new state
	 */
	public void setOnWaitingClicDispositionBehind(boolean state)
	{
		onWaitingClicDispositionBehind = state;
	}
	
	


	/**
	 * Allows to set the boolean autoAdjustement in the drawing
	 */
	public void setAutoAdjustement(boolean state)
	{
		draw.setAutoAdjustement(state);
	}



	/**
	 * This method allows the user to delete the selected figure
	 * in the drawing.
	 * @return The new number of figures
	 */
	public int deleteSelection()
	{		
		try
		{
			draw.deleteSelection(frameParent.getUndoManager());
			frameParent.updateUndoRedo();
			updateDraw(true);
			return draw.getNbFigures();
			
		}catch(Exception ex) 
		{
			ex.printStackTrace(); 
			ExceptionFrameDialog.showExceptionDialog(ex);
		}

		return 0;
	}



	/**
	 * Allow the display or to hide the borders of the drawing
	 * 
	 * @param display Allow to know if the user want to display the borders
	 * of the drawing.
	 */
	public void displayBorders(boolean display)
	{
		draw.displayBorders(display);
	}

	
	
	
	/**
	 * Allows to know if the program is in figure selection
	 * @param is Return true if the program is in figure selection
	 */
	public void isSelection(boolean is)
	{
		draw.isSelection(is);
	}

	
	
	/**
	 * Allows to know if the grid must be displayed
	 * @return True if the grid must be displayed
	 */
	public boolean isGridDisplayed()
	{
		return draw.isGridDisplayed();
	}
	
	
	
	/**
	 * This method allows to know if the project has been
	 * modified or not
	 * @return true if the project has been modified
	 */
	public boolean isModified()
	{		
		return draw.isModified();
	}
	
	
	
	

	/**
	 * Allows to get the current figure in drawing
	 * @return The current figure in drawing
	 */
	public Figure getCurrentFigure()
	{
		return currentFigure;
	}
 
	
	
	
	public LaTeXDrawRectangle getBordersSelection()
	{
		return bordersSelection; 
	}
	
	
	
	/**
	 * Allows to create a new project.
	 */
	public void newProject()
	{
		currentFigure=null;
		formerPoint=null;
		formerSelectionEmpty = true;
		draw.newDrawing();
		updateDraw(true);
	}
	
	

    /**
     * Actions to do on a mouse left button pressed
     * @param e The event
     */
    public void mousePressedLeftButton(MouseEvent e)
    {
    	// these actions are managed is mouse clicked
    	if(onWaitingClicDispositionBehind || onWaitingClicDispositionFront)
    		return;
    	
    	Point point = e.getPoint(); 
    	
    	// We must put the point in the user's view : rescale the point
    	point.x/=draw.getZoom();
    	point.y/=draw.getZoom();
    	
    	formerPoint = draw.getGrid().getTransformedPointToGrid(point, true);
    	
    	String currentChoice = frameParent.getCurrentChoice();

    	// If the user wants to draw a line.
    	if(currentChoice.equals(LaTeXDrawResources.LABEL_LINE))
	    { 	    		
        	point = draw.getGrid().getTransformedPointToGrid(point, true);
    		currentFigure = new Line(new LaTeXDrawPoint2D(point.x, point.y), new LaTeXDrawPoint2D(point.x, point.y), true);   
	    }
    	else
    	// If the user wants to draw a rectangle.
    	if(currentChoice.equals(LaTeXDrawResources.LABEL_RECT))
	    {     		   		
        	point = draw.getGrid().getTransformedPointToGrid(point, true);
    		currentFigure = new LaTeXDrawRectangle(new LaTeXDrawPoint2D(point.x, point.y), new LaTeXDrawPoint2D(point.x, point.y), 
											new LaTeXDrawPoint2D(point.x, point.y), new LaTeXDrawPoint2D(point.x, point.y), true);	
	    }
    	else
    	// If the user wants to draw a square.
    	if(currentChoice.equals(LaTeXDrawResources.LABEL_SQUARE))
	    {     		 
        	point = draw.getGrid().getTransformedPointToGrid(point, true);
    		currentFigure = new Square(new LaTeXDrawPoint2D(point.x, point.y), new LaTeXDrawPoint2D(point.x, point.y),
				    				new LaTeXDrawPoint2D(point.x, point.y), new LaTeXDrawPoint2D(point.x, point.y), true);	    		
	    }
    	else
    	// If the user wants to draw a rhombus.
    	if(currentChoice.equals(LaTeXDrawResources.LABEL_RHOMBUS))
	    {     		  	
        	point = draw.getGrid().getTransformedPointToGrid(point, true);
    		currentFigure = new Rhombus(new LaTeXDrawPoint2D(point.x, point.y), new LaTeXDrawPoint2D(point.x, point.y),
			    				new LaTeXDrawPoint2D(point.x, point.y), new LaTeXDrawPoint2D(point.x, point.y), true);	    		
	    }
    	else
    	// If the user wants to draw a triangle.
    	if(currentChoice.equals(LaTeXDrawResources.LABEL_TRIANGLE))
	    {     		  	
        	point = draw.getGrid().getTransformedPointToGrid(point, true);
    		currentFigure = new Triangle(new LaTeXDrawPoint2D(point.x, point.y), new LaTeXDrawPoint2D(point.x, point.y),
			    				new LaTeXDrawPoint2D(point.x, point.y), new LaTeXDrawPoint2D(point.x, point.y), true);	    		
	    }
    	else
    	// If the user wants to draw a circle.
    	if(currentChoice.equals(LaTeXDrawResources.LABEL_CIRCLE))
	    {     		  	
        	point = draw.getGrid().getTransformedPointToGrid(point, true);
    		currentFigure = new Circle(new LaTeXDrawPoint2D(point.x, point.y), new LaTeXDrawPoint2D(point.x, point.y),
			    				new LaTeXDrawPoint2D(point.x, point.y), new LaTeXDrawPoint2D(point.x, point.y), true);	    		
	    }	    	
    	else
    	// If the user wants to draw a circle.
    	if(currentChoice.equals(LaTeXDrawResources.LABEL_ELLIPSE))
	    {     		  
        	point = draw.getGrid().getTransformedPointToGrid(point, true);
    		currentFigure = new Ellipse(new LaTeXDrawPoint2D(point.x, point.y), new LaTeXDrawPoint2D(point.x, point.y),
				    				new LaTeXDrawPoint2D(point.x, point.y), new LaTeXDrawPoint2D(point.x, point.y), true);	    		
	    }
    	else
//    	 If the user wants to draw an arc.
    	if(currentChoice.equals(LaTeXDrawResources.LABEL_ARC))
	    {     		 
        	point = draw.getGrid().getTransformedPointToGrid(point, true);
    		currentFigure = new Arc(new LaTeXDrawPoint2D(point.x, point.y), new LaTeXDrawPoint2D(point.x, point.y), Arc2D.OPEN, true);	    		
	    }
    	else
//   	 If the user wants to draw a chord.
    	if(currentChoice.equals(LaTeXDrawResources.LABEL_CHORD))
	    {     		  	
        	point = draw.getGrid().getTransformedPointToGrid(point, true);
    		currentFigure = new Arc(new LaTeXDrawPoint2D(point.x, point.y), new LaTeXDrawPoint2D(point.x, point.y), Arc2D.CHORD, true);	    		
	    }
    	else
//   	 If the user wants to draw a wedge.
    	if(currentChoice.equals(LaTeXDrawResources.LABEL_WEDGE))
	    {     		  	
        	point = draw.getGrid().getTransformedPointToGrid(point, true);
    		currentFigure = new Arc(new LaTeXDrawPoint2D(point.x, point.y), new LaTeXDrawPoint2D(point.x, point.y), Arc2D.PIE, true);	    		
	    }
    	else
    	// If the user wants to draw a dot.
    	if(currentChoice.equals(LaTeXDrawResources.LABEL_DOT))
	    {     		
	    	point = draw.getGrid().getTransformedPointToGrid(point, true);
	    	currentFigure = new Dot(new LaTeXDrawPoint2D(point.x, point.y), frameParent.genDotS, frameParent.genDotSize, true);
	    	
	    	setCurrentFigureGeneralValues();
			draw.addFigure(currentFigure, frameParent.getUndoManager());
			currentFigure = null;
			updateDraw(true);
			frameParent.updateUndoRedo();
			frameParent.setExportAsMenu(true);
	    }
    	else
//    	 If the user wants to draw akin points.
    	if(currentChoice.equals(LaTeXDrawResources.LABEL_POINTS_AKIN))
	    {     		
        	point = draw.getGrid().getTransformedPointToGrid(point, true);
			currentFigure = new AkinPoints(new LaTeXDrawPoint2D(point.x, point.y), true);
			updateDraw(false);	
	    }
    	
    	if(currentFigure!=null)
    	{
    		setCurrentFigureGeneralValues();
    		return ;
    	}

    	// If the user wants to select a figure.
    	if(currentChoice.equals(LaTeXDrawResources.LABEL_SELECT_FIGURE))
    	{ 
    		try
    		{
	    		boolean in = draw.onClick(point, e);
	    		DrawBorders db = draw.getCompleteBorders();
	    		
	    		if(db!=null && db.isSelected() && db.isADelimitorSelected())
	    			return ;

	    		Draw selection = draw.getSelected();
	    		int mod = e.getModifiers();
	    		
	    		if((selection!=null && selection.isEmpty()) || 
	    			mod==InputEvent.CTRL_MASK+InputEvent.BUTTON1_MASK || mod==InputEvent.SHIFT_MASK+InputEvent.BUTTON1_MASK)
	    		{
		    		frameParent.setSelection(selection!=null && !selection.isEmpty());
		    		bordersSelection = new LaTeXDrawRectangle(new LaTeXDrawPoint2D(point.x, point.y),
			    											new LaTeXDrawPoint2D(point.x, point.y), false);
		    		bordersSelection.setLineStyle(PSTricksConstants.LINE_DASHED_STYLE);
		    		bordersSelection.setLinesColor(Color.GRAY);
		    		bordersSelection.setThickness(1);
	    		}
	    		else
	    			if(!in)
	    				draw.setSelected(null);
	    			else 
    				{
	    				frameParent.setSelection(true);
	    				
	    				if(selection!=null)
	    				{
	    					selection.setOnRotation(draw.isOnRotation());
	    					if(draw.isOnRotation())
	    						selection.beginRotation();
	    				}
    				}

	    		if(selection!=null && !selection.isEmpty())
	    		{
	    			Figure f;
	    			if(selection.size()==1)
	    				f = selection.getFigureAt(0);
	    			else
	    				f = selection;
	    			
	    			if(f.isDelimitorSelected())
	    			{
	    				int delimitorOrient = f.getSelectedDelimitorOrientation();
	    				
	    				switch(delimitorOrient)
	    				{
	    					case Figure.DELIMITOR_ORIENTATION_EAST:
	    						frameParent.setCursor(Cursor.getPredefinedCursor(Cursor.E_RESIZE_CURSOR));
	    					break;
	    					
	    					case Figure.DELIMITOR_ORIENTATION_WEST:
	    						frameParent.setCursor(Cursor.getPredefinedCursor(Cursor.W_RESIZE_CURSOR));
	    						break;
	    						
	    					case Figure.DELIMITOR_ORIENTATION_SOUTH:
	    						frameParent.setCursor(Cursor.getPredefinedCursor(Cursor.S_RESIZE_CURSOR));
	    						break;
	    						
	    					case Figure.DELIMITOR_ORIENTATION_NORTH:
	    						frameParent.setCursor(Cursor.getPredefinedCursor(Cursor.N_RESIZE_CURSOR));
	    						break;
	    						
	    					case Figure.DELIMITOR_ORIENTATION_NW:
	    						frameParent.setCursor(Cursor.getPredefinedCursor(Cursor.NW_RESIZE_CURSOR));
	    						break;
	    						
	    					case Figure.DELIMITOR_ORIENTATION_NE:
	    						frameParent.setCursor(Cursor.getPredefinedCursor(Cursor.NE_RESIZE_CURSOR));
	    						break;
	    						
	    					case Figure.DELIMITOR_ORIENTATION_SE:
	    						frameParent.setCursor(Cursor.getPredefinedCursor(Cursor.SE_RESIZE_CURSOR));
	    						break;
	    						
	    					case Figure.DELIMITOR_ORIENTATION_SW:
	    						frameParent.setCursor(Cursor.getPredefinedCursor(Cursor.SW_RESIZE_CURSOR));
	    						break;
	    						
	    					case Figure.DELIMITOR_ROTATION:
	    						frameParent.setCursor(LaTeXDrawCursor.ROTATE_CURSOR);
	    						break;
	    						
    						default:
    							frameParent.setCursor(Cursor.getPredefinedCursor(Cursor.MOVE_CURSOR));
	    				}
	    			}
	    			else
	    				frameParent.setCursor(Cursor.getPredefinedCursor(Cursor.MOVE_CURSOR));
	    		}
	    		
	    		repaint();
    		}catch(Exception ex) 
    		{
    			ex.printStackTrace(); 
    			ExceptionFrameDialog.showExceptionDialog(ex);
    		}
    		return;
    	}
    }


    

    /**
     * Set the general parameters to the current figure in creation.
     * @since 1.9.1
     */
    protected void setCurrentFigureGeneralValues()
	{
    	if(currentFigure!=null)
    	{
    		currentFigure.setLinesColor(frameParent.genLineCB.getColor());
    		currentFigure.setInteriorColor(frameParent.genInteriorCB.getColor());
    		currentFigure.setThickness(frameParent.genThickness);
    		
    		if(currentFigure.canBeFilled())
    			currentFigure.setIsFilled(frameParent.genIsFilled);
    		
    		if(currentFigure.canBeHatched())
    		{
    			currentFigure.setHatchingStyle(frameParent.genHatchS);
    			currentFigure.setHatchingColor(frameParent.genHatchCB.getColor());
    			currentFigure.setGradientStartColor(frameParent.genGradientStartCB.getColor());
    			currentFigure.setGradientEndColor(frameParent.genGradientEndCB.getColor());
    		}
    		
    		if(currentFigure.isBordersMovable())
    			currentFigure.setBordersPosition(frameParent.genBordPosS);
    		
    		if(currentFigure.isDoubleBoundaryable())
    		{
    			currentFigure.setHasDoubleBoundary(frameParent.genHasDbleBord);
    			currentFigure.setDoubleColor(frameParent.genDbleBordCB.getColor());
    		}
    		
    		if(currentFigure.isDashableOrDotable())
    			currentFigure.setLineStyle(frameParent.genLineStyleS);
    		
    		if(currentFigure.canHaveShadow())
    		{
    			currentFigure.setShadowColor(frameParent.genShadowCB.getColor());
    			currentFigure.setHasShadow(frameParent.genHasShadow);
    		}
    		
    		if(currentFigure.canHaveArrow())
    		{
    			((Arrowable)currentFigure).setArrow1Style(frameParent.genLeftArrS);
    			((Arrowable)currentFigure).setArrow2Style(((Arrowable)currentFigure).hasTwoLeftArrows() ?
    												ArrowHead.invertArrowStyle(frameParent.genRightArrS) : frameParent.genRightArrS);
    		}
    	}
	}



	/** 
     * Define the actions that it must be realised by a click of the mouse.
     * @param e The event
     */
	public void mouseClicked(MouseEvent e) 
	{
		try
		{
			frameParent.requestFocus();
			
	    	switch(e.getButton())
		    {
			    case MouseEvent.BUTTON1 :   	
			    			mouseClickedLeftButton(e);
			    			break;
				
			    case MouseEvent.BUTTON2 : 
			    			break;
		
			    case MouseEvent.BUTTON3 : 
							mouseClickedRightButton(e);    
							break;
					
			    default : break;
		    } // switch
		}catch(Exception ex)
		{
			ex.printStackTrace();
			ExceptionFrameDialog.showExceptionDialog(ex);
		}
    	e.consume();
	}


	
	
	
    /** 
     * Defines the actions that it must be realised by a left click
     * of the mouse.
     * @param e Corresponds to the click of the mouse
     */
	public void mouseClickedLeftButton(MouseEvent e)
    {
		Point point = e.getPoint();
		point.x/=draw.getZoom();
		point.y/=draw.getZoom();
		String currentChoice = frameParent.getCurrentChoice();
		
		if(onWaitingClicDispositionBehind)
		{
			onWaitingClicDispositionBehind = false;
			frameParent.setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
			
			Draw listSelected = draw.getSelected();
			boolean thereIsOne =  draw.onClick(point, e);
			
			if(thereIsOne && !listSelected.contains(draw.getSelected().getFigureAt(0)))
			{
				draw.getSelected().setSelected(false);
				listSelected.setSelected(true);
				draw.putBehind(listSelected, draw.getSelected().getFigureAt(0));		
				draw.setSelected(listSelected);
				frameParent.setIsModified(true);
				updateDraw(true);
			}
			else
			{
				draw.isSelection(false);
				frameParent.setSelection(false);
				draw.repaint();
			}
			return;
		}
		if(onWaitingClicDispositionFront)
		{
			onWaitingClicDispositionFront = false;
			frameParent.setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
		
			Draw listSelected = draw.getSelected();
			boolean thereIsOne = draw.onClick(point, e);
			if(thereIsOne && !listSelected.contains(draw.getSelected().getFigureAt(0)))
			{
				draw.getSelected().setSelected(false);
				listSelected.setSelected(true);				
				draw.putInFrontOf(listSelected, draw.getSelected().getFigureAt(0));				
				draw.setSelected(listSelected);
				frameParent.setIsModified(true);
				updateDraw(true);
			}
			else
			{
				draw.isSelection(false);
				frameParent.setSelection(false);
				draw.repaint();
			}
			return;
		}	
		
		// A double click = show the frame which allows to set the parameters of the figure or
		// stop the creation of the current figure.
		if(e.getClickCount()==2)
		{
			if(currentFigure!=null && (currentChoice.equals(LaTeXDrawResources.LABEL_POLY) ||
				currentChoice.equals(LaTeXDrawResources.LABEL_CLOSED_BEZIER_CURVE) ||
				currentChoice.equals(LaTeXDrawResources.LABEL_BEZIER_CURVE) ||
				currentChoice.equals(LaTeXDrawResources.LABEL_JOINED_LINES)))
			{
				if(((LaTeXDrawPolygon)currentFigure).getNbPoints()>3 || 
					((currentChoice.equals(LaTeXDrawResources.LABEL_CLOSED_BEZIER_CURVE) ||
					currentChoice.equals(LaTeXDrawResources.LABEL_BEZIER_CURVE)) &&
					((LaTeXDrawPolygon)currentFigure).getNbPoints()>2))
				{
					((LaTeXDrawPolygon)currentFigure).removePointAt(-1);
					draw.addFigure(currentFigure,frameParent.getUndoManager());

					updateDraw(true);
					currentFigure = null;
					
					frameParent.updateUndoRedo();
					frameParent.setExportAsMenu(true);
				}
				else 
				{
					currentFigure = null;
					draw.repaint();
				}
			}
			else
				if(draw.getSelected()!=null && draw.getSelected().size()==1)
					frameParent.showParamFrame(draw.getSelected().whereIsIt(point), false);
			return ;
		}
		
    	if(currentChoice.equals(LaTeXDrawResources.LABEL_POLY) ||
			currentChoice.equals(LaTeXDrawResources.LABEL_JOINED_LINES))
	    {     		//	 If the user wants to draw a polygon (menu polygon selected)
		   	point = draw.getGrid().getTransformedPointToGrid(point, true);
    		if(currentFigure==null)// We create of new polygon
    		{
    			if(currentChoice.equals(LaTeXDrawResources.LABEL_POLY))
	    			currentFigure = new LaTeXDrawPolygon(new LaTeXDrawPoint2D(point.x, point.y), 
														 new LaTeXDrawPoint2D(point.x, point.y), true);	
    			else currentFigure = new JoinedLines(new LaTeXDrawPoint2D(point.x, point.y), 
												     new LaTeXDrawPoint2D(point.x, point.y), true);	
    		}
    		else // We add a point to the polygon
	    		if(currentFigure.getClass().getSimpleName().equals("LaTeXDrawPolygon") ||//$NON-NLS-1$
	    				currentFigure instanceof JoinedLines) 
	    		{
	    			((LaTeXDrawPolygon)currentFigure).setLastPoint(point);
	    			((LaTeXDrawPolygon)currentFigure).addPoint(new LaTeXDrawPoint2D(point.x+1, point.y+1));
	    		}

    		draw.repaint();
	    }else
    	if(currentChoice.equals(LaTeXDrawResources.LABEL_TEXT))
    	{
		   	point = draw.getGrid().getTransformedPointToGrid(point, true);
    		Figure f = new Text("", new LaTeXDrawPoint2D(point.x,point.y), true); //$NON-NLS-1$
    		draw.addFigure(f, null);
    		int size = draw.getNbFigures();
    		frameParent.showParamFrame(f, true);
    		
    		if(size==draw.getNbFigures())
    		{
    			draw.deleteFigure(f, null);
    			draw.addFigure(f, frameParent.getUndoManager());
        		frameParent.updateUndoRedo();
        		updateDraw(true);
    		}
    	}else
    	if(currentChoice.equals(LaTeXDrawResources.LABEL_GRID))
    	{
		   	point = draw.getGrid().getTransformedPointToGrid(point, true);
    		Figure f = new Grid(new LaTeXDrawPoint2D(point.x,point.y), true);
    		draw.addFigure(f, null);
    		int size = draw.getNbFigures();
    		
    		frameParent.showParamFrame(f, true);
    		
    		if(size==draw.getNbFigures())
    		{
    			draw.deleteFigure(f, null);
    			draw.addFigure(f, frameParent.getUndoManager());
        		frameParent.updateUndoRedo();
        		updateDraw(true);
    		}
    	}
    	else
		if(currentChoice.equals(LaTeXDrawResources.LABEL_AXES))
    	{
		   	point = draw.getGrid().getTransformedPointToGrid(point, true);
			Figure f = new Axe(new LaTeXDrawPoint2D(point.x,point.y), true);
    		draw.addFigure(f, null);
    		int size = draw.getNbFigures();
    		
    		frameParent.showParamFrame(f, true);
    		
    		if(size==draw.getNbFigures())
    		{
    			draw.deleteFigure(f, null);
    			draw.addFigure(f, frameParent.getUndoManager());
        		frameParent.updateUndoRedo();
        		updateDraw(true);
    		}
    	}
		else
    	if(currentChoice.equals(LaTeXDrawResources.LABEL_BEZIER_CURVE) ||
			currentChoice.equals(LaTeXDrawResources.LABEL_CLOSED_BEZIER_CURVE))
	    { 	    	
		   	point = draw.getGrid().getTransformedPointToGrid(point, true);
    		if(currentFigure==null)
	    		currentFigure = new BezierCurve(new LaTeXDrawPoint2D(point.x, point.y), 
	    						new LaTeXDrawPoint2D(point.x, point.y), 
			    						currentChoice.equals(LaTeXDrawResources.LABEL_BEZIER_CURVE), true);  
    		else
    		{
    			((BezierCurve)currentFigure).setLastPoint(point);
    			((BezierCurve)currentFigure).addPoint(new LaTeXDrawPoint2D(point.x, point.y));
    		}
	    }
    	
    	if(currentFigure!=null)
    		setCurrentFigureGeneralValues();
    }
	
	
	
	
	
    /** 
     * Defines the actions that it must be realised by a right click
     * of the mouse.
     * @param e Corresponds to the click of the mouse
     */
	public void mouseClickedRightButton(MouseEvent e)
    {
		try
		{
			Point point = e.getPoint();
			
			if(currentFigure!=null)
			{// If the user draw a polygon and if he do a right click, we
				String currentChoice = frameParent.getCurrentChoice();
				
				// must stop the creation of the polygon
				if(currentChoice.equals(LaTeXDrawResources.LABEL_POLY) ||
					currentChoice.equals(LaTeXDrawResources.LABEL_CLOSED_BEZIER_CURVE) ||
					currentChoice.equals(LaTeXDrawResources.LABEL_BEZIER_CURVE) ||
					currentChoice.equals(LaTeXDrawResources.LABEL_JOINED_LINES)) 
					if(((LaTeXDrawPolygon)currentFigure).getNbPoints()>2 || 
						((currentChoice.equals(LaTeXDrawResources.LABEL_CLOSED_BEZIER_CURVE) ||
						currentChoice.equals(LaTeXDrawResources.LABEL_BEZIER_CURVE)) &&
						((LaTeXDrawPolygon)currentFigure).getNbPoints()>1))
					{
						draw.addFigure(currentFigure,frameParent.getUndoManager());
	
						updateDraw(true);
						currentFigure = null;
						
						frameParent.updateUndoRedo();
						frameParent.setExportAsMenu(true);
					}
					else 
					{
						currentFigure = null;
						draw.repaint();
					}
			}
			else
			{// We must show a popup menu
				int nbFiguresSelected = draw.getSelected().size();
				
				if(frameParent.getCurrentChoice().equals(LaTeXDrawResources.LABEL_SELECT_FIGURE) && nbFiguresSelected>0)	
				{
					menuProperties.setEnabled(nbFiguresSelected==1);
					menuJoin.setEnabled(nbFiguresSelected>1);
					menuSeparate.setEnabled(nbFiguresSelected==1 && draw.getSelected().getFigureAt(0) instanceof Draw);
				    popupFigure.show(e.getComponent(), point.x, point.y);
				}
				else
					popupGeneral.show(e.getComponent(), point.x, point.y);
			}
		}catch(Exception ex) 
		{
			ex.printStackTrace(); 
			ExceptionFrameDialog.showExceptionDialog(ex);
		}
    }
	
	


    /**
     * Actions to do on a mouse button pressed
     * @param e The event
     */
	public void mousePressed(MouseEvent e) 
	{
		try
		{
			frameParent.requestFocus();

	    	switch(e.getButton())
		    {
		    	case MouseEvent.BUTTON1 : mousePressedLeftButton(e);
										  break;
			
		    	case MouseEvent.BUTTON2 : break;
		    	case MouseEvent.BUTTON3 :
		    		Draw selection = draw.getSelected();
		    		
		    		if(selection!=null && !selection.isEmpty())
		    			frameParent.setCursor(Cursor.getPredefinedCursor(Cursor.MOVE_CURSOR));
		    		break;
		    }
		}catch(Exception ex)
		{
			ex.printStackTrace(); 
			ExceptionFrameDialog.showExceptionDialog(ex);
		}
    	e.consume();
	}


	
	
    /**
     * Actions to do when the mouse is released
     * @param e The event
     */
	public void mouseReleased(MouseEvent e) 
	{
		try
		{
			Draw selection = draw.getSelected();
	    	
			frameParent.setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
			
			if(bordersSelection != null)
			{
	    		int mod = e.getModifiers();
				Point pt = e.getPoint();
				
		    	// We must put the point in the user's view : rescale the point
		    	pt.x/=draw.getZoom();
		    	pt.y/=draw.getZoom();
	    		bordersSelection.setLastPoint(new LaTeXDrawPoint2D(pt.x, pt.y));
	    		draw.defineSelected(bordersSelection, mod==InputEvent.SHIFT_MASK+InputEvent.BUTTON1_MASK, 
	    				mod==InputEvent.CTRL_MASK+InputEvent.BUTTON1_MASK);
	    		
	    		if(!draw.getSelected().isEmpty() || formerSelectionEmpty != draw.getSelected().isEmpty())
	    			frameParent.setSelection(!draw.getSelected().isEmpty());
	    		
	    		formerSelectionEmpty = draw.getSelected().isEmpty();
	    		draw.repaint();
	    		
				bordersSelection=null;
				formerPoint = null;
				draw.repaint();
				return ;
			}
			
			if(selection!=null && selection.isOnRotation() && selection.isSelected())
				selection.endRotation();
			
			DrawBorders db = draw.getCompleteBorders();
			if(db!=null && db.isSelected() && db.isADelimitorSelected())
			{
				setIsModified(true);
				db.onDelimitorRelease();
				return ;
			}
			
			if(currentFigure!=null && 
				!currentFigure.getClass().getSimpleName().equals("LaTeXDrawPolygon") &&//$NON-NLS-1$
				!(currentFigure instanceof JoinedLines) && !(currentFigure instanceof BezierCurve)) 
			{
				if(!currentFigure.getTheNWPoint().equals(currentFigure.getTheSEPoint(), 0.001))
				{	
					draw.addFigure(currentFigure, frameParent.getUndoManager());
					frameParent.updateUndoRedo();
					frameParent.setExportAsMenu(true);
				}
				
				updateDraw(true);
				currentFigure = null;
				formerPoint = null;
				return;
			}

			Draw listSelected = draw.getSelected();
			if(!listSelected.isEmpty())
			{
				if(cloneForDragged!=null)
				{
					int[] id = new int[listSelected.size()];
					for(int i=0, size=listSelected.size(); i<size; i++)
					{
						id[i] = draw.getFigures().indexOf(listSelected.getFigureAt(i));
						if(id[i]<0)
							throw new IndexOutOfBoundsException();
					}

					frameParent.getUndoManager().add(UndoRedoManager.LABEL_CHANGE_MOVE, id, cloneForDragged, false);
					frameParent.updateUndoRedo();
					cloneForDragged = null;
				}
				updateDraw(true);
				listSelected.onDelimitorRelease(); 	
				formerPoint = null;
				return ;
			}
			
			if(!draw.isAutoAdjustement() && draw.getCompleteBorders()!=null &&
				draw.getCompleteBorders().isSelected())
			{// fixing #1554162
				updateDraw(true);
				setIsModified(true);
			}
			
		}catch(Exception ex) 
		{
			ex.printStackTrace(); 
			ExceptionFrameDialog.showExceptionDialog(ex);
		}
		formerPoint = null;
	}

	
	
	
	
    /**
     * Actions to do when the mouse is entered
     * @param e The event
     */
	public void mouseEntered(MouseEvent e) 
	{
		/*
		 * No action to do
		 */
	}

	
	

	
    /**
     * Actions to do when the mouse is exited
     * @param e The event
     */	
	public void mouseExited(MouseEvent e) 
	{
		/*
		 * No action to do
		 */
	}


	
	
	
    /**
     * Actions to do when the mouse is dragged
     * @param e The event
     */
	public void mouseDragged(MouseEvent e) 
	{
		try
		{	
			Point pt = e.getPoint();
	    	DrawBorders db = draw.getCompleteBorders();
	    	
			frameParent.requestFocus();
	    	
	    	// We must put the point in the user's view : rescale the point
	    	pt.x/=draw.getZoom();
	    	pt.y/=draw.getZoom();
						
			if(db!=null && db.isSelected() && db.isADelimitorSelected())
			{
				if(formerPoint!=null)
					db.onDragged(formerPoint, pt);
				
				formerPoint = pt;	
				draw.updateDraw();
				return ;
			}
	    	
	    	if(bordersSelection!=null)
	    	{
	    		int mod = e.getModifiers();
	    		
	    		bordersSelection.setLastPoint(new LaTeXDrawPoint2D(pt.x, pt.y));
	    		draw.defineSelected(bordersSelection, mod==InputEvent.SHIFT_MASK+InputEvent.BUTTON1_MASK, 
	    				mod==InputEvent.CTRL_MASK+InputEvent.BUTTON1_MASK);

	    		formerSelectionEmpty = draw.getSelected().isEmpty();
	    		draw.repaint();
	    		return ;
	    	}
	    	
	    	if(currentFigure!=null)
	    	{
	    		pt = draw.getGrid().getTransformedPointToGrid(pt, true);
				// We must not draw out the desktop
				if(pt.x>currentFigure.getThickness()/2 && pt.y>currentFigure.getThickness()/2)
				{
					if(currentFigure instanceof AkinPoints)
						((AkinPoints)currentFigure).addPoint(new LaTeXDrawPoint2D(pt.x, pt.y));
					else //We move the south-east point
						currentFigure.setLastPoint(new LaTeXDrawPoint2D(pt.x, pt.y));
					
		    		draw.repaint();
		    		setIsModified(true);
				return;
				}
	    	}
	    	
	    	Draw selected = draw.getSelected();
			if(!selected.isEmpty())
			{// We must not draw out the desktop
				pt = draw.getGrid().getTransformedPointToGrid(pt, true);
				
				if(pt.x>selected.getThickness()/2 && pt.y>selected.getThickness()/2)
				{
					if(formerPoint!=null)
					{
						if(cloneForDragged==null)
							cloneForDragged = (Draw) selected.clone();

						selected.onDragged(formerPoint, pt);
						
						if(selected.size()==1 && (selected.getFigureAt(0) instanceof Dot))
							frameParent.setDotFieldValue(((Dot)selected.getFigureAt(0)).getWidth());
						
						getDraw().repaint();
					}
					formerPoint = pt;	
					setIsModified(true);
				}
			}
		}catch(Exception ex)
		{
			ex.printStackTrace(); 
			ExceptionFrameDialog.showExceptionDialog(ex);
		}
	}


	
	
    /**
     * Actions to do when the mouse is moved
     * @param e The event
     */
	public void mouseMoved(MouseEvent e) 
	{
		Point pt = e.getPoint();
    	// We must put the point in the user's view : rescale the point
    	pt.x/=draw.getZoom();
    	pt.y/=draw.getZoom();
					
    	if(currentFigure!=null)
    	{
    		pt = draw.getGrid().getTransformedPointToGrid(pt, true);
    		
			// We must not draw out the desktop
			if(pt.x>currentFigure.getThickness()/2 && pt.y>currentFigure.getThickness()/2)
				if(currentFigure.getClass().getSimpleName().equals("LaTeXDrawPolygon") || //$NON-NLS-1$
					currentFigure instanceof JoinedLines || currentFigure instanceof BezierCurve)
				{
		    		currentFigure.setLastPoint(new LaTeXDrawPoint2D(pt.x, pt.y));	
					
		    		if(currentFigure instanceof BezierCurve)
		    			((BezierCurve)currentFigure).equilibrate();
		    		
		    		draw.repaint();
		    		return ;
				}
    	}
	}
	
	
	
	/**
	 * Allows to update the drawing.
	 * @param withCode If true, the code is updated (if in auto update).
	 */
	public void updateDraw(boolean withCode)
	{	
		frameParent.requestFocus();
		draw.updateDraw();
		
		if(codePanel.isInAutoUpdate() && withCode)
			updateCode();
	}
	
	
	
	
	/**
	 * Allows to update the code of the codepanel
	 */
	public void updateCode()
	{
		try
		{
			if(draw.getNbFigures()==0)
				codePanel.setEmpty();
			else
			{	
				codePanel.setStart(draw.getTheCompleteSWPoint(), draw.getTheCompleteNEPoint(), 
									draw.getPixelsPerCm(), draw.getRealBorders().getOriginPoint());
				codePanel.setBody(draw.getCode());
			}	
		}catch(Exception ex) 
		{ 
			ex.printStackTrace(); 
			ExceptionFrameDialog.showExceptionDialog(ex);
		}
	}
	
		

	/**
	 * A drawPanel is not serializable
	 */	
	@SuppressWarnings("unused")
	private void readObject(ObjectInputStream ois)
	{		
		/*
		 * We don't read anything
		 */
	}




	/**
	 * Allows to copy the selected figure
	 */
	public void copy() 
	{
		draw.copy();
	}



	/**
	 * Allows to paste a copied or a cut figure
	 */
	public void paste() 
	{
		draw.paste();
		updateDraw(true);
	}




	/**
	 * Allows to cut the selected figure
	 */
	public void cut() 
	{
		draw.cut();
	}




	/**
	 * Allows to join the selected figures
	 * @param undoManager The undo/redo manager
	 */
	public void joinSelection(UndoRedoManager undoManager)
	{
		draw.joinSelection(undoManager);
		updateDraw(false);
	}




	/**
	 * Allows to join the figures of the selected drawing
	 * @param undoManager The undo/redo manager
	 */
	public void separateSelection(UndoRedoManager undoManager)
	{
		draw.separateSelection(undoManager);
		updateDraw(true);
	}


	
	/**
	 * Allows to insert a picture into the drawing
	 * @param undoManager The undo/redo manager
	 * @param path The path of the picture
	 * @return True is the picture is inserted
	 */
	public boolean insertPicture(UndoRedoManager undoManager, String path)
	{
		boolean state = draw.insertPicture(undoManager, path);
		updateDraw(true);
		
		return state;
	}
	
	
	
	/**
	 * Allows to set if the grid must be displayed or not
	 * @param isDisplay True : the grid must be displayed
	 */
	public void setIsGridDisplay(boolean isDisplay)
	{
		draw.getGrid().setGridDisplayed(isDisplay);
		menuUpdateToGrid.setEnabled(isDisplay);
		frameParent.menuBar.getUpdateToGridMenu().setEnabled(isDisplay);
		updateDraw(false);
	}



	/**
	 * @return the gridIsMagnetic.
	 */
	public boolean isGridMagnetic()
	{
		return draw.getGrid().isMagnetic();
	}



	/**
	 * @param gridIsMagnetic the gridIsMagnetic to set.
	 */
	public void setGridIsMagnetic(boolean gridIsMagnetic)
	{
		if(gridIsMagnetic!=isGridMagnetic())
			setIsModified(true);
		
		draw.getGrid().setMagnetic(gridIsMagnetic);
		menuUpdateToGrid.setEnabled(true);
	}
	
	
	
	/**
	 * Move the horizontal scrollbar.
	 * @since 1.9
	 * @param add This value is added to the current position of the scollbar.
	 */
	public void moveHorizontalScrollbar(int add)
	{
		getHorizontalScrollBar().setValue(getHorizontalScrollBar().getValue()+add);
	}
	
	
	
	/**
	 * Move the vertical scrollbar.
	 * @since 1.9
	 * @param add This value is added to the current position of the scollbar.
	 */
	public void moveVerticalScrollbar(int add)
	{
		getVerticalScrollBar().setValue(getVerticalScrollBar().getValue()+add);
	}



	
	/**
	 * Move horizontally the selected figures.<br>
	 * The increment is 1 pixel or, if the magnetic is activated, the gap of the
	 * grid.
	 * @since 1.9
	 * @param direction The direction of the movement:<br>
	 * <ul>
	 * <li>1: the X-coordinates of the figures are increased;</li>
	 * <li>-1: the X-coordinates of the figures are decreased.</li>
	 * </ul>
	 */
	public void moveSelectedFiguresHorizontally(int direction)
	{
		try
		{
			if(direction!=-1 && direction!=1)
				return;
			
			Draw selection = draw.getSelected();
			
			if(selection!=null && !selection.isEmpty())
			{
				double gap = draw.getGrid().isMagnetic() ? draw.getGrid().getMagneticGridGap() : 1.;
				
				if(direction==-1)
					gap*=-1;
				
				selection.shift(gap, 0);
				updateDraw(true);
			}
		}
		catch(Exception e)
		{
			e.printStackTrace();
			ExceptionFrameDialog.showExceptionDialog(e);
		}
	}
	
	
	
	
	
	/**
	 * Move vertically the selected figures.<br>
	 * The increment is 1 pixel or, if the magnetic is activated, the gap of the
	 * grid.
	 * @since 1.9
	 * @param direction The direction of the movement:<br>
	 * <ul>
	 * <li>1: the Y-coordinates of the figures are increased;</li>
	 * <li>-1: the Y-coordinates of the figures are decreased.</li>
	 * </ul>
	 */
	public void moveSelectedFiguresVertically(int direction)
	{
		try
		{
			if(direction!=-1 && direction!=1)
				return;
			
			Draw selection = draw.getSelected();
			
			if(selection!=null && !selection.isEmpty())
			{
				double gap = draw.getGrid().isMagnetic() ? draw.getGrid().getMagneticGridGap() : 1.;
				
				if(direction==-1)
					gap*=-1;
				
				selection.shift(0, gap);
				updateDraw(true);
			}
		}
		catch(Exception e)
		{
			e.printStackTrace();
			ExceptionFrameDialog.showExceptionDialog(e);
		}
	}


	

	/**
	 * Update the selected figures of the drawing to the displayed grid.
	 * @since 1.9
	 */
	public void updateSelectionToGrid()
	{
		Draw selection = draw.getSelected();
		
		if(selection!=null && !selection.isEmpty() && isGridMagnetic())
		{
			try
			{
				Figure clone = (Draw) selection.clone();
				int[] id = new int[selection.size()];
				
				for(int i=0, size=selection.size(); i<size; i++)
				{
					id[i] = draw.getFigures().indexOf(selection.getFigureAt(i));
					if(id[i]<0)
						throw new IndexOutOfBoundsException();
				}
	
				frameParent.getUndoManager().add(UndoRedoManager.LABEL_TRANSFORM, id, clone, false);
				frameParent.updateUndoRedo();
				selection.updateToGrid(draw.getGrid());
				updateDraw(true);
				
			}catch(CloneNotSupportedException e)
			{
				e.printStackTrace();
			}
		}
	}



	/**
	 * @since 1.9
	 */
	public void actionPerformed(ActionEvent e)
	{
		try
		{
	    	Object src = e.getSource();
	
	    	if(src instanceof JMenuItem)
	    	{
	    		AbstractButton ab = ((AbstractButton)src);
	    		String label = ab.getActionCommand();
	    		
	    		if(label.equals(ACTION_CMD_UPDATETOGRID))
	    		{
	    			updateSelectionToGrid();
	    			return ;
	    		}
	    		
	    		return ;
	    	}
		}
		catch(Exception ex)
		{
			ex.printStackTrace();
			ExceptionFrameDialog.showExceptionDialog(ex);
		}
	}


	

	/**
	 * @since 1.9
	 */
	public void mouseWheelMoved(MouseWheelEvent e)
	{
		int mod = e.getModifiers();
	
		if(mod==InputEvent.CTRL_MASK)
		{// ctrl+wheel = zoom in/out
			int wheelRot = e.getWheelRotation();
			boolean zoomed = false;
			
			if(wheelRot>0)
			{
				zoomOut();
				zoomed = true;
			}
			else
				if(wheelRot<0)
				{
					zoomIn();
					zoomed = true;
				}
			
			if(zoomed)
			{
				frameParent.getXScale().repaint();
				frameParent.getYScale().repaint();
				frameParent.getToolbar().getZoomField().setValue(getZoom()*100);
			}
			
			return ;
		}
	}
	
	
	
	
	/**
	 * Sets the XML element "meta" with the parameters of the code panel.
	 * @param document The XML document.
	 * @param meta The Element to fill.
	 * @throws IllegalArgumentException If document or meta is null.
	 * @since 2.0.0
	 */
	public void getXMLMetadata(Document document, Element meta)
	{
		if(document==null || meta==null)
			throw new IllegalArgumentException();
		
		Element magnet = document.createElement(LaTeXDrawNamespace.LATEXDRAW_NAMESPACE+':'+LaTeXDrawNamespace.XML_MAGNETIC_GRID);
		magnet.appendChild(document.createTextNode(String.valueOf(isGridMagnetic())));
		meta.appendChild(magnet);
	}



	/**
	 * Sets the latexdraw parameters from the <code>metadata</code> SVG tag.
	 * @param nl The list of the parameters.
	 * @since 2.0.0
	 */
	public void setXMLMetadata(NodeList nl) {
		// Nothing to do.
	}


	/**
	 * @return the codePanel.
	 * @since 2.0.0
	 */
	public CodePanel getCodePanel()
	{
		return codePanel;
	}
}
