package latexDraw.figures;

import java.awt.*;
import java.awt.geom.Arc2D;
import java.awt.geom.Line2D;
import java.awt.geom.Rectangle2D;
import java.io.IOException;
import java.io.ObjectInputStream;

import javax.swing.JComboBox;

import latexDraw.figures.properties.Arrowable;
import latexDraw.lang.LaTeXDrawLang;
import latexDraw.psTricks.DviPsColors;
import latexDraw.psTricks.PSTricksConstants;
import latexDraw.ui.LaTeXDrawFrame;
import latexDraw.ui.components.Delimitor;
import latexDraw.ui.components.MagneticGrid;
import latexDraw.ui.dialog.AbstractParametersFrame;
import latexDraw.util.LaTeXDrawNumber;
import latexDraw.util.LaTeXDrawPoint2D;


/** 
 * This class defines an Arc.<br>
 *<br>
 * This file is part of LaTeXDraw<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 *<br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.<br>
 *<br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE. See the GNU General Public License for more details.<br>
 *<br>
 * 06/06/06<br>
 * @author Arnaud BLOUIN<br>
 * @version 2.0.0<br>
 */
public class Arc extends Circle implements Arrowable
{
	private static final long serialVersionUID = 1L;

	/** The delimiter of the start angle */
	protected Delimitor dStart;
	
	/** The delimiter of the end angle */
	protected Delimitor dEnd;
	
	/** The angle of the start (radian)*/
	protected double startAngle;
	
	/** The angle of the end (radian)*/
	protected double endAngle;
	
	/** the arrowhead of the first point */
	protected ArrowHead arrowHead1;
	
	/** The arrow head of the second point */
	protected ArrowHead arrowHead2;
	
	/** The type of arc : wedge, arc, chord */
	protected int type;
	
	/** Allows to display the points of the arc */
	protected boolean showPoints;

	/** The value by default of the attribute DEFAULT_SHOWPOINTS */
	public static final boolean DEFAULT_SHOWPOINTS = false;
	
	/** A kind of arc */
	public static final String TYPE_ARC = LaTeXDrawLang.getOthersString("Arc.arc"); //$NON-NLS-1$
	
	/** A kind of arc */
	public static final String TYPE_PIE = LaTeXDrawLang.getOthersString("Arc.wedge"); //$NON-NLS-1$
	
	/** A kind of arc */
	public static final String TYPE_CHORD = LaTeXDrawLang.getOthersString("Arc.chord"); //$NON-NLS-1$
	
	/** The label of the arc choice */
	public static final String LABEL_ARCCHOICE = LaTeXDrawLang.getOthersString("Arc.arcType"); //$NON-NLS-1$
	
	/** The start angle by default */
	public static final double DEFAULT_STARTANGLE = 0;
	
	/** The end angle by default */
	public static final double DEFAULT_ENDANGLE = Math.PI;
	
	/** The type of arc by default */
	public static final String DEFAULT_TYPE = TYPE_ARC;
	
	
	
	
	/**
	 * The constructor by default
	 */
	public Arc(boolean increaseMeter)
	{
		this(new LaTeXDrawPoint2D(), new LaTeXDrawPoint2D(), Arc2D.OPEN, increaseMeter);
	}
	
	
	
	/**
	 * The constructor taking the two opposed points
	 * @param p1 The first point of the parabola
	 * @param p2 The last point
	 */
	public Arc(LaTeXDrawPoint2D p1, LaTeXDrawPoint2D p2, int type, boolean increaseMeter)
	{
		super(p1, new LaTeXDrawPoint2D(p2.x,p1.y), new LaTeXDrawPoint2D(p1.x, p2.y), p2, increaseMeter);
		
		isBordersMovable = false;
		isDoubleBoundaryable = false;
		
		setType(type);
		dStart = new Delimitor(new LaTeXDrawPoint2D((p1.x+3*p2.x)/4,p2.y));
		dEnd   = new Delimitor(new LaTeXDrawPoint2D((3*p1.x+p2.x)/4,p2.y));
		
		dStart.setColorSet3();
		dEnd.setColorSet3();

		endAngle = DEFAULT_ENDANGLE;
		startAngle = DEFAULT_STARTANGLE;
		bordersPosition = PSTricksConstants.BORDERS_MIDDLE;
		
		arrowHead1 = new ArrowHead(p1, new Line(p1,(LaTeXDrawPoint2D)p1.clone(),false), this);
		arrowHead2 = new ArrowHead(p2, new Line(p2,(LaTeXDrawPoint2D)p2.clone(),false), this);
		shape = getInsideOutsideOrMiddleBorders();
	}
	
	
	
	/**
	 * Allows to set the style of the first arrow of the line
	 * @param style The new style of the first arrow of the line
	 */
	public synchronized void setArrow1Style(String style)
	{
		arrowHead1.setArrowStyle(style);
	}
	
	
	
	/**
	 * Allows to set the style of the second arrow of the line
	 * @param style The new style of the second arrow of the line
	 */
	public synchronized void setArrow2Style(String style)
	{
		arrowHead2.setArrowStyle(style);
	}
	

	
	/**
	 * Allows to get the kind of arc
	 * @return The kind of arc
	 */
	public synchronized int getType()
	{
		return type;
	}
	
	
	

	public String getTypeName()
	{
			if(type == Arc2D.OPEN)
				return TYPE_ARC;
			
			if(type == Arc2D.CHORD)
				return TYPE_CHORD;
			
			return TYPE_PIE;
	}
	
	
	
	
	@Override
	public void rescaleX(double formerX, double newX, double percent, LaTeXDrawRectangle bound) 
	{
		if(percent==1.) return ;
		super.rescaleX(formerX, newX, percent, bound);
		updateCenterStartEndDelimitors();
		shape = getInsideOutsideOrMiddleBorders();
	}




	@Override
	public void rescaleY(double formerY, double newY, double percent, LaTeXDrawRectangle bound) 
	{
		if(percent==1.) return ;
		super.rescaleY(formerY, newY, percent, bound);
		updateCenterStartEndDelimitors();
		shape = getInsideOutsideOrMiddleBorders();
	}

	
	
	
	
	
	/**
	 * Allows to set the kind of arc
	 * @param type The kind of arc
	 */
	public synchronized void setType(int type)
	{
		if(type!=Arc2D.OPEN && type!=Arc2D.PIE && type!=Arc2D.CHORD)
			throw new IllegalArgumentException();
		
		this.type = type;
		canHaveArrow = type==Arc2D.OPEN;
		shape = getInsideOutsideOrMiddleBorders();
	}
	
	

	
	/**
	 * Allows to set the kind of arc
	 * @param type The kind of arc
	 */
	public synchronized void setType(String type)
	{
		if(type.equals(TYPE_ARC))
			this.type = Arc2D.OPEN;
		else 
		if(type.equals(TYPE_CHORD))
			this.type = Arc2D.CHORD;
		else 
		if(type.equals(TYPE_PIE))
			this.type = Arc2D.PIE;
		else 
			throw new IllegalArgumentException();
	
		canHaveArrow = this.type==Arc2D.OPEN;
	}
	
	
	
	
	@Override
	public synchronized void setThickness(float val)
	{
		if(!Double.isInfinite(val) && !Double.isNaN(val) && val>0)
		{
			thickness = val;
			double dim = 1.33*val+3.33 +1.;
			if(dim<6.) 
				dim = 6;
			
			// We change the size of the delimiters when the size of 
			// the figure change
			dStart.setDim(dim);
			dEnd.setDim(dim);
			arrowHead1.getLine().setThickness(val);
			arrowHead2.getLine().setThickness(val);
			
			if(borders!=null)
				borders.setThickness(val);
			shape = getInsideOutsideOrMiddleBorders();
		}
	}
	
	
	
	
	
	
	/**
	 * Allows to set the start angle (radian)
	 * @param angle The new start angle
	 */
	public synchronized void setStartAngle(double angle)
	{
		startAngle = angle;
		shape = getInsideOutsideOrMiddleBorders();
	}
	
	
	
	/**
	 * Allows to set the end angle (radian)
	 * @param angle The new end angle
	 */
	public synchronized void setEndAngle(double angle)
	{
		endAngle = angle;
		shape = getInsideOutsideOrMiddleBorders();
	}
	

	

	/**
	 * Allows to get the start angle
	 * @return The start angle
	 */
	public synchronized double getStartAngle()
	{
		return startAngle%(2.*Math.PI);
	}
	
	
	

	/**
	 * Allows to get the end angle
	 * @return The end angle
	 */
	public synchronized double getEndAngle()
	{
		return endAngle%(2.*Math.PI);
	}
	
	
	
	@Override
	public void onDragged(Point formerPt, Point newPt)
	{
		if(formerPt.equals(newPt)) return;
		
		if(borders.dSelected!=null)
		{
			borders.onDragged(formerPt, newPt);
			shape = getInsideOutsideOrMiddleBorders();
		}
		else
		if(dSelected!=null)
		{
			LaTeXDrawPoint2D newPt2 = rotateInvertPoint(newPt);
			if(dSelected==dStart)
				moveStartAngle(newPt2);
			else
			if(dSelected==dEnd)
				moveEndAngle(newPt2);
		}
		else //	 If the user has clicked on the line
		{
			shift(formerPt, newPt);	
			shape = getInsideOutsideOrMiddleBorders();
		}

		updateGravityCenter();
		updateCenterStartEndDelimitors();
		rotationAngle = borders.getRotationAngle();//update the angle(when rotation)
	}
	
	
	

	@Override
	public void draw(Graphics2D g, Object antiAlias, Object rendering, Object alphaInter, Object colorRendering)
	{
		LaTeXDrawPoint2D NW = borders.getTheNWPoint();
		LaTeXDrawPoint2D SE = borders.getTheSEPoint();
		Color formerCol = g.getColor();
		double cx = (NW.x+SE.x)/2., cy = (NW.y+SE.y)/2.;
		double c2x = Math.cos(rotationAngle)*cx - Math.sin(rotationAngle)*cy;
		double c2y = Math.sin(rotationAngle)*cx + Math.cos(rotationAngle)*cy;
		double c3x = Math.cos(-rotationAngle)*(cx-c2x) - Math.sin(-rotationAngle)*(cy-c2y);
		double c3y = Math.sin(-rotationAngle)*(cx-c2x) + Math.cos(-rotationAngle)*(cy-c2y);
		LaTeXDrawPoint2D startPt = getNonRotatedStartPoint();
		LaTeXDrawPoint2D endRot  = getNonRotatedEndPoint();
		ArrowHead ar1=null, ar2=null;
		String oldStyleL = null, oldStyleR=null;
		startAngle%=(2*Math.PI);
		endAngle%=(2*Math.PI);
		double dx=0, dy=0;
		boolean changeFillStyle = false;
		
		if(rotationAngle%(Math.PI*2)!=0)
		{		
			g.rotate(rotationAngle);
			g.translate(c3x,c3y);
		}	
		
		if(hasShadow)
		{
			LaTeXDrawPoint2D cg = getGravityCenter();
			LaTeXDrawPoint2D shadowCg = (LaTeXDrawPoint2D)cg.clone();
			shadowCg.setLocation(cg.x+shadowSize, cg.y);
			shadowCg = Figure.rotatePoint(shadowCg, cg, shadowAngle);
			dx = shadowCg.x-cg.x;
			dy = cg.y-shadowCg.y;
		}
		
		double start = startAngle, end = endAngle;
		
		if(type==Arc2D.OPEN)
		{
			double radius = SE.x-NW.x;
			
			if(!arrowHead1.getArrowStyle().equals(PSTricksConstants.NONEARROW_STYLE))
			{
				double lgth = arrowHead1.getArrowHeadLength();
				
				if(lgth!=0)
				{
					startAngle+=(Math.atan(lgth/radius));
					startAngle%=(2*Math.PI);
				}

				Line line = new Line(getTangenteAt(startAngle, ((float)startAngle)<((float)Math.PI)), false);
				line.getPt2().move(line.getPt1(), startPt);
				line.setPointAt(startPt, 0);
				arrowHead1.setLine(line);
				arrowHead1.setPosition(line.getPt1());
				
				if(arrowHead1.getArrowStyle().equals(PSTricksConstants.DLARROW_STYLE) || 
					arrowHead1.getArrowStyle().equals(PSTricksConstants.DRARROW_STYLE))
				{
					oldStyleL = arrowHead1.getArrowStyle();
					double angle = startAngle;
					
					if(lgth!=0)
					{
						angle+=Math.atan((lgth*3)/radius);
						angle%=(2*Math.PI);
					}	
					
					line = new Line(getTangenteAt(angle, angle<=Math.PI), false);
					ar1 = new ArrowHead(line.getPt1(), line, this);
					ar1.setArrowInset(arrowHead1.getArrowInset());
					ar1.setArrowLength(arrowHead1.getArrowLength());
					ar1.setArrowSizeDim(arrowHead1.getArrowSizeDim());
					ar1.setArrowSizeNum(arrowHead1.getArrowSizeNum());
					
					if(arrowHead1.getArrowStyle().equals(PSTricksConstants.DLARROW_STYLE))
					{
						ar1.setArrowStyle(PSTricksConstants.LARROW_STYLE);
						arrowHead1.setArrowStyle(PSTricksConstants.LARROW_STYLE);
					}
					else
					{
						ar1.setArrowStyle(PSTricksConstants.RARROW_STYLE);
						arrowHead1.setArrowStyle(PSTricksConstants.RARROW_STYLE);
					}
				}
				
				if(lgth!=0)
				{
					startAngle=start+Math.atan(2*lgth/radius);
					startAngle%=(2*Math.PI);
				}
			}
			
			
			if(!arrowHead2.getArrowStyle().equals(PSTricksConstants.NONEARROW_STYLE))
			{
				double lgth = arrowHead2.getArrowHeadLength();	
				
				if(lgth!=0)
					 endAngle-=(Math.atan(lgth/radius))%(2*Math.PI);
	
				Line line = new Line(getTangenteAt(endAngle, ((float)endAngle)>=((float)Math.PI)), false);
				line.getPt2().move(line.getPt1(), endRot);
				line.setPointAt(endRot, 0);
				arrowHead2.setLine(line);
				arrowHead2.setPosition(line.getPt1());
				
				if(arrowHead2.getArrowStyle().equals(PSTricksConstants.DLARROW_STYLE) || 
					arrowHead2.getArrowStyle().equals(PSTricksConstants.DRARROW_STYLE))
				{
					oldStyleR = arrowHead2.getArrowStyle();
					double angle = endAngle;
					
					if(lgth!=0)
					{
						angle-=Math.atan((lgth*3)/radius);
						angle%=(2*Math.PI);
					}
					
					line = new Line(getTangenteAt(angle, angle>=Math.PI), false);
					ar2 = new ArrowHead(line.getPt1(), line, this);
					ar2.setArrowInset(arrowHead2.getArrowInset());
					ar2.setArrowLength(arrowHead2.getArrowLength());
					ar2.setArrowSizeDim(arrowHead2.getArrowSizeDim());
					ar2.setArrowSizeNum(arrowHead2.getArrowSizeNum());
					
					if(arrowHead2.getArrowStyle().equals(PSTricksConstants.DLARROW_STYLE))
					{
						ar2.setArrowStyle(PSTricksConstants.LARROW_STYLE);
						arrowHead2.setArrowStyle(PSTricksConstants.LARROW_STYLE);
					}
					else
					{
						ar2.setArrowStyle(PSTricksConstants.RARROW_STYLE);
						arrowHead2.setArrowStyle(PSTricksConstants.RARROW_STYLE);
					}
				}
				
				if(lgth!=0)
				{
					endAngle=end-Math.atan(2*lgth/radius);
					endAngle%=(2*Math.PI);
				}
			}
		}
		
		if(showPoints)
		{
			g.setStroke(new BasicStroke(thickness/2, BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER,
					1.f, new float[]{blackDashLength, whiteDashLength}, 0));
			g.setColor(linesColor);
		
			g.draw(new Line2D.Double(gravityCenter.x, gravityCenter.y, startPt.x, startPt.y));
			g.draw(new Line2D.Double(gravityCenter.x, gravityCenter.y, endRot.x, endRot.y));
		}
		
		Shape s0 = getInsideOutsideOrMiddleBorders();
		
		if(hasShadow)
		{
			g.setStroke(new BasicStroke(thickness, BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER));
			g.translate(dx, dy);
			g.setColor(shadowColor);
			g.fill(s0);
			g.draw(s0);
			g.translate(-dx, -dy);
			if(!isFilled)
			{
				changeFillStyle = true;
				isFilled = true;
			}
			g.setColor(interiorColor);
			g.draw(s0);
		}
		
		if(lineStyle.equals(PSTricksConstants.LINE_NONE_STYLE))
			g.setStroke(new BasicStroke(thickness, BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER));
		else 
		if(lineStyle.equals(PSTricksConstants.LINE_DOTTED_STYLE))
		{
			g.setStroke(new BasicStroke(thickness, BasicStroke.CAP_ROUND, BasicStroke.JOIN_MITER,
					1.f, new float[]{0,thickness+dotSep}, 0));
		}
		else
		if(lineStyle.equals(PSTricksConstants.LINE_DASHED_STYLE))
		{
			g.setStroke(new BasicStroke(thickness, BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER,
					1.f, new float[]{blackDashLength, whiteDashLength}, 0));
		}	
		
		fillFigure(g, antiAlias, rendering, alphaInter, colorRendering, s0);					
		g.setColor(linesColor);
		g.draw(s0);
		
		if(type==Arc2D.OPEN)
		{
			Stroke formerS = g.getStroke();
			g.setStroke(new BasicStroke(thickness, BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER));
			
			if(hasShadow)
			{
				g.translate(dx, dy);
				arrowHead1.draw(g, interiorColor, true);
				arrowHead2.draw(g, interiorColor, true);
				g.translate(-dx, -dy);
			}
			
			arrowHead1.draw(g, interiorColor, false);
			arrowHead2.draw(g, interiorColor, false);
			g.setStroke(formerS);
			
			if(ar1!=null)	
			{
				ar1.draw(g, interiorColor, false);
				arrowHead1.setArrowStyle(oldStyleL);
			}
			
			if(ar2!=null)	
			{
				ar2.draw(g, interiorColor, false);
				arrowHead2.setArrowStyle(oldStyleR);
			}
			
			startAngle = start;
			endAngle   = end;
		}
		
		if(changeFillStyle) isFilled = false;
		g.setColor(formerCol);
		
		if(isSelected)
		{			
			if(dStart!=null && dEnd!=null)
			{
				dStart.draw(g);
				dEnd.draw(g);
			}
			g.translate(-c3x, -c3y);
			g.rotate(-rotationAngle);

			borders.draw(g, false, antiAlias, rendering, alphaInter, colorRendering);
		}
		else
			if(rotationAngle%(Math.PI*2)!=0)
			{
				g.translate(-c3x, -c3y);
				g.rotate(-rotationAngle);
			}
	}

	
	
	
	/**
	 * Allows to get the position of the non-rotated end point
	 * @return The end point
	 */
	public LaTeXDrawPoint2D getNonRotatedEndPoint()
	{
		LaTeXDrawPoint2D NW = borders.getTheNWPoint();
		LaTeXDrawPoint2D SE = borders.getTheSEPoint();

		return rotatePoint(new LaTeXDrawPoint2D(SE.x, (SE.y+NW.y)/2.), -endAngle);
	}
	
	
	
	
	/**
	 * Allows to get the position of the non-rotated start point
	 * @return The start point
	 */
	public LaTeXDrawPoint2D getNonRotatedStartPoint()
	{
		LaTeXDrawPoint2D NW = borders.getTheNWPoint();
		LaTeXDrawPoint2D SE = borders.getTheSEPoint();

		return rotatePoint(new LaTeXDrawPoint2D(SE.x, (SE.y+NW.y)/2.), -startAngle);
	}
	
	
	
	
	
		
	@Override
	public synchronized void setLastPoint(LaTeXDrawPoint2D pt)
	{
		super.setLastPoint(pt);
		updateCenterStartEndDelimitors();
		shape = getInsideOutsideOrMiddleBorders();
	}

	
	
	@Override
	public synchronized void setFirstPoint(LaTeXDrawPoint2D pt)
	{
		super.setFirstPoint(pt);
		updateCenterStartEndDelimitors();
		shape = getInsideOutsideOrMiddleBorders();
	}
	
	
	
	@Override
	public boolean isIn(LaTeXDrawPoint2D p) 
	{
		LaTeXDrawPoint2D pt = rotateInvertPoint(p);
		// Is the point is in a selector
		if(isSelected && (borders.dNE.isIn(pt) || borders.dNW.isIn(pt) 
			|| borders.dSE.isIn(pt) || borders.dSW.isIn(pt) || dEnd.isIn(pt)|| dStart.isIn(pt)))
			return true;
		
		LaTeXDrawPoint2D NW = getTheNWPoint();
		LaTeXDrawPoint2D SE = getTheSEPoint();
		double width = SE.x - NW.x;
		double start2 = Math.toDegrees(startAngle);
		double end2   = startAngle>endAngle ? Math.toDegrees(2*Math.PI-startAngle+endAngle) : 
			Math.toDegrees(endAngle-startAngle);
		
		Shape s =  new Arc2D.Double(NW.x-thickness/2., NW.y-thickness/2., 
						width+thickness, width+thickness, start2, end2, type);
	
		if(!s.contains(pt))
			return false;
		
		if(isFilled() || hasShadow() || hasGradient())
			return true;
		
		double gap = thickness/2.;
		s =  new Arc2D.Double(NW.x+gap, NW.y+gap, width-thickness, width-thickness, start2, end2, type);
		
		if(type==Arc2D.PIE)
		{
			double radius = width/2.;
			LaTeXDrawPoint2D start = new LaTeXDrawPoint2D(
				gravityCenter.x + Math.cos(startAngle)*radius,
				gravityCenter.y - Math.sin(startAngle)*radius);
			LaTeXDrawPoint2D end = new LaTeXDrawPoint2D(
				gravityCenter.x + Math.cos(endAngle)*radius,
				gravityCenter.y - Math.sin(endAngle)*radius);
			
			Line l1 = new Line(gravityCenter, rotatePoint(start), false);
			l1.setThickness(thickness);
			if(l1.isIn(p))
				return true;
			
			Line l2 = new Line(gravityCenter, rotatePoint(end), false);
			l2.setThickness(thickness);
			if(l2.isIn(p))
				return true;
		}
		else if(type==Arc2D.CHORD)
		{
			double radius = width/2.;
			LaTeXDrawPoint2D start = new LaTeXDrawPoint2D(
				gravityCenter.x + Math.cos(startAngle)*radius,
				gravityCenter.y - Math.sin(startAngle)*radius);
			LaTeXDrawPoint2D end = new LaTeXDrawPoint2D(
				gravityCenter.x + Math.cos(endAngle)*radius,
				gravityCenter.y - Math.sin(endAngle)*radius);
			
			Line l1 = new Line(rotatePoint(start), rotatePoint(end), false);
			l1.setThickness(thickness);
			if(l1.isIn(p))
				return true;
		}
			
		
		if(s.contains(pt))
			return false;
		
		return true;
	}

	
	
	
	
	@Override
	public synchronized String getCodePSTricks(DrawBorders drawBorders, float ppc)
	{
		LaTeXDrawPoint2D d = drawBorders.getOriginPoint();
		LaTeXDrawPoint2D NW = borders.getTheNWPoint(), SE = borders.getTheSEPoint();
		String add1 = "", add2="", startCode="", addEnd="", fillType="";  //$NON-NLS-1$  //$NON-NLS-2$  //$NON-NLS-3$  //$NON-NLS-4$  //$NON-NLS-5$
		String arrowsCode="", endCode = ""; //$NON-NLS-1$//$NON-NLS-2$
		boolean isFilledWasChanged = false;
		double threshold = 0.001;
		
		if(hasShadow)
		{
			fillType+=",shadow=true";//$NON-NLS-1$
			if(Math.toDegrees(shadowAngle)!=PSTricksConstants.DEFAULT_SHADOW_ANGLE)
				fillType+=",shadowangle="+(float)Math.toDegrees(shadowAngle);//$NON-NLS-1$
			
			if(((float)shadowSize) != ((float)DEFAULT_SHADOW_SIZE))
				fillType+=",shadowsize="+LaTeXDrawNumber.getCutNumber((float)(shadowSize/PPC), threshold) ;//$NON-NLS-1$
			
			if(!shadowColor.equals(PSTricksConstants.DEFAULT_SHADOW_COLOR))
			{
				String name = DviPsColors.getColourName(shadowColor);
				if(name==null)
				{
					name = "color"+number+'e';//$NON-NLS-1$
					DviPsColors.addUserColour(shadowColor, name); 
				}
				fillType += ",shadowcolor=" + name; //$NON-NLS-1$
			}
			if(!isFilled)
			{
				isFilled = true;
				isFilledWasChanged = true;
			}
		}
		
		String str = getPSTricksCodeFilling(ppc);
		if(str.length()>0) fillType=fillType+','+str;
		
		str = getPSTricksCodeLine(ppc);
		if(str.length()>0) add1=add1+','+str;
		
		double x = (NW.x+SE.x)/2. - d.x, y = d.y - (SE.y+NW.y)/2., radius = (SE.x-NW.x)/2.;		
		
		if(type==Arc2D.OPEN)
		{
			startCode = "\\psarc"; //$NON-NLS-1$
			
			String currentArrowLStyle = arrowHead1.getArrowStyle();
			String currentArrowRStyle = arrowHead2.getArrowStyle();	
			
			if(currentArrowRStyle.equals(PSTricksConstants.DLARROW_STYLE))
				currentArrowRStyle = PSTricksConstants.DRARROW_STYLE;
			else if(currentArrowRStyle.equals(PSTricksConstants.DRARROW_STYLE))
				currentArrowRStyle = PSTricksConstants.DLARROW_STYLE;
			else if(currentArrowRStyle.equals(PSTricksConstants.RARROW_STYLE))
				currentArrowRStyle = PSTricksConstants.LARROW_STYLE;
			else if(currentArrowRStyle.equals(PSTricksConstants.LARROW_STYLE))
				currentArrowRStyle = PSTricksConstants.RARROW_STYLE;
			else if(currentArrowRStyle.equals(PSTricksConstants.DLARROW_STYLE))
				currentArrowRStyle = PSTricksConstants.DRARROW_STYLE;
			else if(currentArrowRStyle.equals(PSTricksConstants.LRBRACKET_STYLE))
				currentArrowRStyle = PSTricksConstants.RRBRACKET_STYLE;
			else if(currentArrowRStyle.equals(PSTricksConstants.RRBRACKET_STYLE))
				currentArrowRStyle = PSTricksConstants.LRBRACKET_STYLE;
			else if(currentArrowRStyle.equals(PSTricksConstants.RSBRACKET_STYLE))
				currentArrowRStyle = PSTricksConstants.LSBRACKET_STYLE;
			else if(currentArrowRStyle.equals(PSTricksConstants.LSBRACKET_STYLE))
				currentArrowRStyle = PSTricksConstants.RSBRACKET_STYLE;
			
			String paramsR = ","+ arrowHead2.getParametersCode(); //$NON-NLS-1$
			String paramsL = ","+ arrowHead1.getParametersCode(); //$NON-NLS-1$
			if(paramsR.equals(",")) paramsR = ""; //$NON-NLS-1$ //$NON-NLS-2$
			if(paramsL.equals(",")) paramsL = ""; //$NON-NLS-1$ //$NON-NLS-2$
			
			if(!currentArrowLStyle.equals(PSTricksConstants.NONEARROW_STYLE))
			{
				arrowsCode="{"+currentArrowLStyle+'-'; //$NON-NLS-1$
				
				if(!currentArrowRStyle.equals(PSTricksConstants.NONEARROW_STYLE))
					arrowsCode+=currentArrowRStyle;
				arrowsCode+='}';
				
				if(!arrowHead1.isOfTheSameTypeAs(arrowHead2))
				{
					if((currentArrowRStyle.equals(PSTricksConstants.LRBRACKET_STYLE) ||
						currentArrowRStyle.equals(PSTricksConstants.RRBRACKET_STYLE)  ||
						currentArrowRStyle.equals(PSTricksConstants.RSBRACKET_STYLE)  ||
						currentArrowRStyle.equals(PSTricksConstants.RSBRACKET_STYLE))	&&
						(currentArrowLStyle.equals(PSTricksConstants.BAREND_STYLE) ||
						currentArrowLStyle.equals(PSTricksConstants.BARIN_STYLE) ||
						currentArrowLStyle.equals(PSTricksConstants.LRBRACKET_STYLE) ||
						currentArrowLStyle.equals(PSTricksConstants.RRBRACKET_STYLE)  ||
						currentArrowLStyle.equals(PSTricksConstants.RSBRACKET_STYLE)  ||
						currentArrowLStyle.equals(PSTricksConstants.RSBRACKET_STYLE)))
						add1 = paramsR;
					else
						add1+= paramsL + paramsR;
				}else add1+=paramsR;
			}
			else 
				if(!currentArrowRStyle.equals(PSTricksConstants.NONEARROW_STYLE))
				{
					add1+=paramsR;
					arrowsCode="{-"+currentArrowRStyle+'}'; //$NON-NLS-1$
				}

		}
		else
		if(type==Arc2D.PIE)
			startCode = "\\pswedge"; //$NON-NLS-1$
		else
		{
			endCode = "\n\\psline[linewidth=" + LaTeXDrawNumber.getCutNumber(thickness/ppc, threshold) +add1+ //$NON-NLS-1$
			"](" +LaTeXDrawNumber.getCutNumber((float)((x+Math.cos(startAngle)*radius)/ppc), threshold)+ ',' + //$NON-NLS-1$
			LaTeXDrawNumber.getCutNumber((float)((y+Math.sin(startAngle)*radius)/ppc), threshold)+
			")(" +LaTeXDrawNumber.getCutNumber((float)((x+Math.cos(endAngle)*radius)/ppc), threshold)+ ','; //$NON-NLS-1$
			
			if(endAngle%Math.PI==0)
				endCode +=LaTeXDrawNumber.getCutNumber((float)(y/ppc), threshold)+ ")"; //$NON-NLS-1$
			else endCode +=LaTeXDrawNumber.getCutNumber((float)((y+Math.sin(endAngle)*radius)/ppc), threshold)+ ")"; //$NON-NLS-1$
			startCode = "\\psarc";//$NON-NLS-1$
		}
		
		if(rotationAngle%(Math.PI*2)!=0.)
		{
			double angle = -Math.toDegrees(rotationAngle);
			double cx = (gravityCenter.x-d.x)/ppc;
			double cy = (d.y-gravityCenter.y)/ppc;
			double x2 = -Math.cos(-rotationAngle)*cx+
						Math.sin(-rotationAngle)*cy+cx;
			double y2 =  -Math.sin(-rotationAngle)*cx-
						Math.cos(-rotationAngle)*cy+cy;
			
			if(Math.abs(x2) < 0.001) x2 = 0;
			if(Math.abs(y2) < 0.001) y2 = 0;
			
			startCode ="\\rput{"+(float)angle+ "}("+ //$NON-NLS-1$ //$NON-NLS-2$
			LaTeXDrawNumber.getCutNumber((float)x2, threshold)+','+LaTeXDrawNumber.getCutNumber((float)y2, threshold)+"){"+startCode; //$NON-NLS-1$
			addEnd = "}"; //$NON-NLS-1$
		}		
		
		if(showPoints)
			add1+=",showpoints=true"; //$NON-NLS-1$
		
		if(hasDoubleBoundary)
		{
			add1+=",doubleline=true,doublesep="+LaTeXDrawNumber.getCutNumber((float)(doubleSep/ppc), threshold); //$NON-NLS-1$
			add1+=",dimen="+bordersPosition;  //$NON-NLS-1$
			
			if(doubleColor!=PSTricksConstants.DEFAULT_DOUBLE_COLOR)
			{
				String name = DviPsColors.getColourName(doubleColor);
				if(name==null)
				{
					name = "color"+number+'d';//$NON-NLS-1$
					DviPsColors.addUserColour(linesColor, name); 
				}
				add1+= ",doublecolor="+name; //$NON-NLS-1$
			}
		}
		
		if(!isFilled && !interiorColor.equals(PSTricksConstants.DEFAULT_INTERIOR_COLOR))
		{
			String name = DviPsColors.getColourName(interiorColor);
			if(name==null)
			{
				name = "color"+number+'b';//$NON-NLS-1$
				DviPsColors.addUserColour(interiorColor, name); 
			}
			add1 += ",fillcolor=" + name; //$NON-NLS-1$
		}
		
		if(isFilledWasChanged) isFilled = false;
		
		return startCode+ "[linewidth=" + LaTeXDrawNumber.getCutNumber(thickness/ppc, threshold) + //$NON-NLS-1$
				add1 + add2 + fillType + "]"+ arrowsCode + "(" + LaTeXDrawNumber.getCutNumber((float)(x/ppc), threshold)+','+ //$NON-NLS-1$ //$NON-NLS-2$
				LaTeXDrawNumber.getCutNumber((float)(y/ppc), threshold) + "){"+ //$NON-NLS-1$
				LaTeXDrawNumber.getCutNumber((float)(radius/ppc), threshold) +"}{"+ //$NON-NLS-1$
				LaTeXDrawNumber.getCutNumber((float)Math.toDegrees(startAngle), threshold)
				 +"}{"+LaTeXDrawNumber.getCutNumber((float)Math.toDegrees(endAngle), threshold)+'}'+endCode+ addEnd; //$NON-NLS-1$
	}

	
	
	
	/**
	 * Allows to change the position of the start angle
	 * @param newPt The new position
	 */
	public void moveStartAngle(LaTeXDrawPoint2D newPt)
	{
		double radius = newPt.distance(gravityCenter);
		double angle = Math.acos((newPt.x-gravityCenter.x)/radius);
		
		if(newPt.y>gravityCenter.y)
			 startAngle = 2*Math.PI - angle;
		else startAngle = angle;
		
		updateCenterStartEndDelimitors();
		shape = getInsideOutsideOrMiddleBorders();
	}
	
	
	
	
	/**
	 * Allows to change the position of the end angle
	 * @param newPt The new position
	 */
	public void moveEndAngle(LaTeXDrawPoint2D newPt)
	{
		double radius = newPt.distance(gravityCenter);
		double angle = Math.acos((newPt.x-gravityCenter.x)/radius);
		
		if(newPt.y>gravityCenter.y)
			 endAngle = 2*Math.PI - angle;
		else endAngle = angle;
		
		updateCenterStartEndDelimitors();
		shape = getInsideOutsideOrMiddleBorders();
	}
	
	
	
	
	
	@Override
	public void onClick(Point p)
	{
		LaTeXDrawPoint2D pt = rotateInvertPoint(p);
		
		if(dStart.isIn(pt)) dSelected = dStart;
		else if(dEnd.isIn(pt)) dSelected = dEnd;
		
		isSelected = true;
		borders.onClick(p);
	}

	
	
	
	
	@Override
	public void shift(double shiftX, double shiftY)
	{
		if(shiftX==0 && shiftY==0) return ;
		
		super.shift(shiftX, shiftY);
		updateCenterStartEndDelimitors();
		shape = getInsideOutsideOrMiddleBorders();
	}

	
	
	
	/**
	 * Allows the update the centre, the start delimiter
	 * and the end delimiter 
	 * */
	public void updateCenterStartEndDelimitors()
	{
		LaTeXDrawPoint2D second = borders.getPoint(1);
		LaTeXDrawPoint2D first = borders.getPoint(0), fourth = borders.getPoint(3);
		double xStart, xEnd, yStart, yEnd;
		double radius = Math.abs(first.x-second.x)/2.;	
	
		gravityCenter.x = (first.x+second.x)/2;
		gravityCenter.y = (first.y+fourth.y)/2;
		
		xStart = gravityCenter.x + Math.cos(startAngle)*radius/2.;
		yStart = gravityCenter.y - Math.sin(startAngle)*radius/2.;
		dStart.getCenter().setLocation(xStart, yStart);
		
		xEnd = gravityCenter.x + Math.cos(endAngle)*radius/2.;
		yEnd = gravityCenter.y - Math.sin(endAngle)*radius/2.;
		dEnd.getCenter().setLocation(xEnd, yEnd);
	}


	
	/**
	 * Allows to create a choice containing all kinds of arc
	 * @return The choice
	 */
	public static JComboBox createArcChoice(AbstractParametersFrame f)
	{		
		JComboBox arcChoice = new JComboBox();
		arcChoice.setName(LABEL_ARCCHOICE);
		
     	arcChoice.addItem(Arc.TYPE_ARC);
     	arcChoice.addItem(Arc.TYPE_CHORD);
     	arcChoice.addItem(Arc.TYPE_PIE);
     	arcChoice.addItemListener(f);
     	
     	return arcChoice;
	}
	
	
	
	
	
	
	/**
	 * Allows to get the style of the left arrow of the line
	 * @return The style of the left arrow of the line
	 */
	public synchronized String getArrow1Style()
	{
		return arrowHead1.getArrowStyle();
	}
	

	
	
	
	/**
	 * Allows to get the style of the right arrow of the line
	 * @return The style of the left arrow of the line
	 */
	public synchronized String getArrow2Style()
	{
		return arrowHead2.getArrowStyle();
	}
	
	
	
	
	

	@Override
	public Object clone() throws CloneNotSupportedException
	{
		Arc a = (Arc) super.clone();

		a.dEnd   = (Delimitor) dEnd.clone();
		a.dStart = (Delimitor) dStart.clone();
		a.type = type;
		a.endAngle = endAngle;
		a.startAngle = startAngle;
		a.updateCenterStartEndDelimitors();
		a.arrowHead1 = (ArrowHead)arrowHead1.clone();
		a.arrowHead2 = (ArrowHead)arrowHead2.clone();
		return a;
	}

	
	
	private void readObject(ObjectInputStream ois) throws IOException, 
					ClassNotFoundException
	{
		canHaveShadow = true;
		interiorColor = (Color) ois.readObject();
		lineStyle = (String) ois.readObject();
		rotationAngle = ois.readDouble();
		thickness = ois.readFloat();
		isFilled = ois.readBoolean();
		isSelected = ois.readBoolean();
		isOnRotation = ois.readBoolean();
		linesColor = (Color) ois.readObject();
		blackDashLength = ois.readFloat();
		dotSep = ois.readFloat();
		whiteDashLength = ois.readFloat();
		borders = (LaTeXDrawRectangle) ois.readObject();
		endAngle = ois.readDouble();
		startAngle = ois.readDouble();
		dEnd   = (Delimitor) ois.readObject();
		dStart = (Delimitor) ois.readObject();

		if(LaTeXDrawFrame.getVersionOfFile().compareTo("1.5")>=0) //$NON-NLS-1$
		{
			hasDoubleBoundary = ois.readBoolean();
			doubleColor = (Color)ois.readObject();
			doubleSep = ois.readDouble();
			if(!(LaTeXDrawFrame.getVersionOfFile().compareTo("1.6")>=0)) //$NON-NLS-1$
				ois.readBoolean();
			hatchingAngle = ois.readDouble();
			hatchingColor = (Color)ois.readObject();
			hatchingStyle = (String)ois.readObject();
			hatchingWidth = ois.readFloat();
			arrowHead1 = (ArrowHead)ois.readObject();
			arrowHead2 = (ArrowHead)ois.readObject();
			
			if(LaTeXDrawFrame.getVersionOfFile().compareTo("1.6")>=0) //$NON-NLS-1$
				showPoints = ois.readBoolean();
			
			arrowHead2.setArrowInset(arrowHead1.getArrowInset());
			arrowHead2.setArrowLength(arrowHead1.getArrowLength());
			arrowHead2.setArrowSizeDim(arrowHead1.getArrowSizeDim());
			arrowHead2.setArrowSizeNum(arrowHead1.getArrowSizeNum());
			arrowHead2.setBracketNum(arrowHead1.getBracketNum());
			arrowHead2.setDotSizeDim(arrowHead1.getDotSizeDim());
			arrowHead2.setDotSizeNum(arrowHead1.getDotSizeNum());
			arrowHead2.setRBracketNum(arrowHead1.getRBracketNum());
			arrowHead2.setTBarSizeDim(arrowHead1.getTBarSizeDim());
			arrowHead2.setTBarSizeNum(arrowHead1.getTBarSizeNum());
			
			if(LaTeXDrawFrame.getVersionOfFile().compareTo("1.6") < 0)//$NON-NLS-1$
			{
				if(hatchingStyle.equals(DECREPETED_FILL_CROSS))
					hatchingStyle = PSTricksConstants.TOKEN_FILL_CROSSHATCH;
				else if(hatchingStyle.equals(DECREPETED_FILL_HORIZ))
					hatchingStyle = PSTricksConstants.TOKEN_FILL_HLINES;
				else if(hatchingStyle.equals(DECREPETED_FILL_VERT))
					hatchingStyle = PSTricksConstants.TOKEN_FILL_VLINES;
				else if(hatchingStyle.equals(DECREPETED_FILL_NO))
					hatchingStyle = PSTricksConstants.TOKEN_FILL_NONE;
			}
		}
		else
		{
			hasDoubleBoundary  = DEFAULT_HAS_DOUBLE_BOUNDARY;
			doubleColor = DEFAULT_DOUBLE_COLOR;
			doubleSep   = DEFAULT_DOUBLESEP;
			hatchingAngle = DEFAULT_HATCH_ANGLE;
			hatchingColor = DEFAULT_HATCH_COL;
			hatchingStyle = DEFAULT_HATCH_STYLE;
			hatchingWidth = DEFAULT_HATCH_WIDTH;
			arrowHead1 = new ArrowHead(this);
			arrowHead2 = new ArrowHead(this);
		}
		
		if(LaTeXDrawFrame.getVersionOfFile().compareTo("1.7")>=0) //$NON-NLS-1$
		{
			hasShadow 	= ois.readBoolean();
			shadowAngle = ois.readDouble();
			shadowSize	= ois.readDouble();
			shadowColor	= (Color)ois.readObject();
			gradientEndColor = (Color)ois.readObject();
			gradientStartColor = (Color)ois.readObject();
			gradientAngle = ois.readDouble();
			gradientMidPoint = ois.readDouble();
		}
		else
		{
			if(startAngle>endAngle)
			{
				double tmp = startAngle;
				ArrowHead tmpAH = arrowHead1;
				startAngle = endAngle;
				endAngle = tmp;
				arrowHead1 = arrowHead2;
				arrowHead2 = tmpAH;
			}
			hasShadow 	= DEFAULT_SHADOW_HAS;
			shadowAngle	= DEFAULT_SHADOW_ANGLE;
			shadowSize	= DEFAULT_SHADOW_SIZE;
			shadowColor	= DEFAULT_SHADOW_COLOR;
			gradientEndColor = PSTricksConstants.DEFAULT_GRADIENT_END_COLOR;
			gradientStartColor = PSTricksConstants.DEFAULT_GRADIENT_START_COLOR;
			gradientAngle = DEFAULT_GRADIENT_ANGLE;
			gradientMidPoint = DEFAULT_GRADIENT_MID_POINT;
		}
		
		if(LaTeXDrawFrame.getVersionOfFile().compareTo("1.8")>=0) //$NON-NLS-1$
		{
			hatchingSep = ois.readDouble();
			type = ois.readInt();// fixing #1555339
		}
		else
		{
			hatchingSep = DEFAULT_HATCH_SEP;
			type = Arc2D.OPEN;// fixing #1555339
		}
		
		updateGravityCenter();
		updateStyleOfDelimitors();
		updateCenterStartEndDelimitors();
		shape = getInsideOutsideOrMiddleBorders();
	}



	/**
	 * @return Returns the showPoints.
	 */
	public boolean isShowPoints()
	{
		return showPoints;
	}



	/**
	 * @param showPoints The showPoints to set.
	 */
	public synchronized void setShowPoints(boolean showPoints)
	{
		this.showPoints = showPoints;
	}



	/**
	 * @return Returns the arrowHead1.
	 */
	public ArrowHead getArrowHead1()
	{
		return arrowHead1;
	}



	/**
	 * @return Returns the arrowHead2.
	 */
	public ArrowHead getArrowHead2()
	{
		return arrowHead2;
	}
	
	
	
	
	@Override
	public Shape[] getDbleBoundariesOutside(Shape classicBord)
	{
		return getDbleBoundariesMiddle(classicBord);
	}
	
	
	
	
	
	@Override
	public Shape[] getDbleBoundariesInside(Shape classicBord)
	{
		return getDbleBoundariesMiddle(classicBord);
	}
	
	
	
	
	@Override
	public Shape[] getDbleBoundariesMiddle(Shape classicBord)
	{
		/*
		 * 
		 * Dx = (doubleSep/2.+A'x2+A'y2-Ay2-Ax2+2bAy-2bAy)
		 *      /(2*(A'x-Ax-aAy+aA'y)
		 * 
		 * 
		 */
		
		
		return null;
	}
	
	
	
	
	@Override
	public Shape[] getDbleBoundariesOutInOrMiddle(Shape classicBord)
	{
		return getDbleBoundariesMiddle(classicBord);
	}
	
	
	
	@Override
	public Shape getOutsideBorders()
	{
		return getMiddleBorders();
	}
	
	
	
	
	@Override
	public Shape getInsideBorders()
	{
		return getMiddleBorders();
	}
	
	

	@Override
	public Shape getMiddleBorders()
	{
		LaTeXDrawPoint2D NW = getTheNWPoint(), SE = getTheSEPoint();
		double width = Math.max(Math.abs(NW.x - SE.x), 1);
		double start =  Math.toDegrees(startAngle); 
		double end   = startAngle>endAngle ? Math.toDegrees(2*Math.PI-startAngle+endAngle) : Math.toDegrees(endAngle-startAngle);
		
		return new Arc2D.Double(NW.x, NW.y, width, width, start, end, type);
	}
	
	
	
	@Override
	public Shape getInsideOutsideOrMiddleBorders()
	{
		return getMiddleBorders();
	}



	@Override
	public boolean canHaveArrow()
	{
		return type==Arc2D.OPEN;
	}



	@Override
	public void mirrorHorizontal(LaTeXDrawPoint2D origin)
	{
		double tmp = startAngle;
		ArrowHead tmpAH = arrowHead1;
		
		startAngle = Math.PI*2 - endAngle;
		endAngle   = Math.PI*2 - tmp;
		arrowHead1 = arrowHead2;
		arrowHead2 = tmpAH;
		
		
		super.mirrorHorizontal(origin);
		updateCenterStartEndDelimitors();
	}



	@Override
	public void mirrorVertical(LaTeXDrawPoint2D origin)
	{
		double tmp = startAngle;
		ArrowHead tmpAH = arrowHead1;
		
		startAngle = Math.PI*3 - endAngle;
		endAngle   = Math.PI*3 - tmp;

		arrowHead1 = arrowHead2;
		arrowHead2 = tmpAH;
		
		super.mirrorVertical(origin);
		updateCenterStartEndDelimitors();
	}



	@Override
	public void updateToGrid(MagneticGrid grid)
	{
		super.updateToGrid(grid);
		updateCenterStartEndDelimitors();
	}
	
	
	
	/**
	 * @return The coordinate of the start point of the arc.
	 * @since 1.9
	 */
	public LaTeXDrawPoint2D getStartPoint(boolean rotation)
	{
		double radius = getRadius();
		LaTeXDrawPoint2D pt = new LaTeXDrawPoint2D(gravityCenter.x+Math.cos(startAngle)*radius,
												  gravityCenter.y-Math.sin(startAngle)*radius);
		
		if(rotation)
			return rotateInvertPoint(pt);
		
		return pt;
	}
	
	
	
	/**
	 * @return The coordinate of the end point of the arc.
	 * @since 1.9
	 */
	public LaTeXDrawPoint2D getEndPoint(boolean rotation)
	{
		double radius = getRadius();
		LaTeXDrawPoint2D pt = new LaTeXDrawPoint2D(gravityCenter.x+Math.cos(endAngle)*radius,
												   gravityCenter.y-Math.sin(endAngle)*radius);

		if(rotation)
			return rotateInvertPoint(pt);
		
		return pt;
	}

	
	

	@Override
	public boolean intersected(Rectangle2D.Double r)
	{
		if(r==null)
			return false;
		
		BasicStroke wideline = new BasicStroke(thickness);
        Shape outline = wideline.createStrokedShape(shape);
        boolean ok = outline.intersects(r);
        
        if(!ok && type==Arc2D.OPEN && (isFilled() || isHatched() || hasShadow() || hasGradient()))
        {
        	Line2D.Double l = new Line2D.Double(getStartPoint(true), getEndPoint(true));
        	
        	if(l.intersects(r))
        		ok = true;
        }
        
        return ok;
	}



	@Override
	public int getSelectedDelimitorOrientation()
	{
		if(dSelected!=null && (dSelected==dStart || dSelected==dEnd))
			return DELIMITOR_ORIENTATION_NONE;
		
		return super.getSelectedDelimitorOrientation();
	}
	
	
	@Override
	public int hashCode()
	{
		return super.hashCode()^3^type;
	}



	public boolean hasTwoLeftArrows()
	{
		return true;
	}
}
