/*
        testmsgr.cpp

        Test messenger
        Sends (randomized) XML messages to server for testing

--------------------------------------------------------------------------------
gSOAP XML Web services tools
Copyright (C) 2000-2017, Robert van Engelen, Genivia, Inc. All Rights Reserved.
This software is released under one of the following two licenses:
GPL or Genivia's license for commercial use.
--------------------------------------------------------------------------------
GPL license.

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free Software
Foundation; either version 2 of the License, or (at your option) any later
version.

This program is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
PARTICULAR PURPOSE. See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License along with
this program; if not, write to the Free Software Foundation, Inc., 59 Temple
Place, Suite 330, Boston, MA 02111-1307 USA

Author contact information:
engelen@genivia.com / engelen@acm.org
--------------------------------------------------------------------------------
A commercial use license is available from Genivia, Inc., contact@genivia.com
--------------------------------------------------------------------------------
*/

#include "soapH.h"              // generated by soapcpp2 -CSL import/dom.h
#include <fstream>

static void sigpipe_handler(int) { } // SIGPIPE handler

struct Namespace namespaces[] = {{NULL,NULL,NULL,NULL}}; // dummy namespace table

static size_t cid = 0; // cid:id# counter

////////////////////////////////////////////////////////////////////////////////
//
//      Indicator tags
//
////////////////////////////////////////////////////////////////////////////////

#define REP "__REPEAT"
#define SEL "__SELECT"

////////////////////////////////////////////////////////////////////////////////
//
//      Command line options
//
////////////////////////////////////////////////////////////////////////////////

static const char   *act   = NULL;      // -aact
static bool          blank = false;     // -b
static bool          cont  = false;     // -c
static bool          iind  = false;     // -i
static bool          jind  = false;     // -j
static ULONG64       lmax  = 3;         // -lmax
static ULONG64       mmax  = 255;       // -mmax
static LONG64        nlen  = -1;        // -nlen
static const char   *ofile = NULL;      // -ofile
static unsigned long pperc = 100;       // -pperc
static unsigned long qperc = 100;       // -qperc
static long          seed  = 0;         // -rseed
static int           tsec  = 1;         // -tsec
static bool          verb  = false;     // -v
static bool          mime  = false;     // -A
static bool          chnk  = false;     // -C
static bool          http  = false;     // -H
static bool          mtom  = false;     // -M
static const char   *rep   = REP;       // -Rrep
static const char   *sel   = SEL;       // -Ssel
static const char   *ifile = NULL;      // [infile|-]
static const char   *URL   = NULL;      // [URL]
static unsigned long port  = 0;         // [port]

////////////////////////////////////////////////////////////////////////////////
//
//      Inline
//
////////////////////////////////////////////////////////////////////////////////

inline void gen_send(struct soap *ctx, const char *s, size_t k = std::string::npos)
{
  if (k == std::string::npos)
    k = strlen(s);
  if (soap_send_raw(ctx, s, k))
  {
    if (URL)
      fprintf(stderr, "testmsgr: failed to send to %s\n", URL);
    else
      fprintf(stderr, "testmsgr: failed to write to output\n");
    soap_print_fault(ctx, stderr);
    exit(EXIT_FAILURE);
  }
}

inline void gen_blank(struct soap *ctx)
{
  if (blank)
    gen_send(ctx, " ", 1);
}

inline void gen_send_verb(struct soap *ctx, const char *t, const char *s, size_t k = std::string::npos)
{
  if (verb)
    printf("%s %s\n", t, s);
  gen_blank(ctx);
  gen_send(ctx, s, k);
  gen_blank(ctx);
}

inline void gen_indent(struct soap *ctx, size_t tab)
{
  if (iind)
  {
    gen_send(ctx, "\n", 1);
    for (size_t i = 0; i < tab; ++i)
      gen_send(ctx, " ", 1);
  }
  else if (jind)
  {
    gen_send(ctx, "\n", 1);
    for (size_t i = 0; i < tab; ++i)
      gen_send(ctx, "\t", 1);
  }
}

inline long gen_random()
{
#ifdef HAVE_RANDOM
  return random();
#else
  return srand();
#endif
}

////////////////////////////////////////////////////////////////////////////////
//
//      Proto
//
////////////////////////////////////////////////////////////////////////////////

static void test_server(struct soap *ctx, xsd__anyType& dom);
static void test_client(struct soap *ctx, xsd__anyType& dom);
static void gen_test(struct soap *ctx, xsd__anyType& dom);
static void gen_random_test(struct soap *ctx, xsd__anyType& dom, size_t tab = 0);
static void gen_random_attachments(struct soap *ctx);
static void get_minmax(struct soap *ctx, xsd__anyType& dom, unsigned long& n, unsigned long& m);
static void gen_value(struct soap *ctx, const char *s);
static bool get_range(const char *s, unsigned long& n, unsigned long& m);
static void gen_name(struct soap *ctx, ULONG64 n);
static void gen_ncname(struct soap *ctx, ULONG64 n);
static void gen_qname(struct soap *ctx, ULONG64 n);
static void gen_lang(struct soap *ctx, ULONG64 x, ULONG64 n);
static void gen_text(struct soap *ctx, ULONG64 n);
static void gen_hex(struct soap *ctx, ULONG64 n);
static void gen_base64(struct soap *ctx, ULONG64 n);
static int receive_request(struct soap *ctx);
static void receive_response(struct soap *ctx);

////////////////////////////////////////////////////////////////////////////////
//
//      testmsgr main
//
////////////////////////////////////////////////////////////////////////////////

int main(int argc, char **argv)
{
  if (argc >= 2)
  {
    for (int i = 1; i < argc; i++)
    {
      const char *a = argv[i];
      if ((a[0] == '-' && a[1])
#ifdef WIN32
        || a[0] == '/'
#endif
         )
      {
        bool f = true;
        while (f && *++a)
        {
          switch (*a)
          {
            case 'a':
              a++;
              f = false;
              if (*a)
                act = a;
              else if (i < argc)
                act = argv[++i];
              break;
            case 'b':
              blank = true;
              break;
            case 'c':
              cont = true;
              break;
            case '?':
            case 'h':
              fprintf(stderr,
                  "Usage: testmsgr [-aact] [-b] [-c] [-h] [-i] [-j] [-lmax] [-mmax] [-nlen] [-pperc] [-qperc] [-rseed] [-tsec] [-v] [-A] [-C] [-H] [-M] [-Rrep] [-Ssel] [infile|-] [URL|port]\n\n"
                  "-aact   use HTTP SOAP Action header value \"act\"\n"
                  "-b      add random spacing to randomized values when using -r\n"
                  "-c      continue testing until the service endpoint fails\n"
                  "-h      display help message\n"
                  "-i      indent XML with spaces when using -r\n"
                  "-j      indent XML with tabs when using -r\n"
                  "-lmax   max number of randomized element repeats when using -r (default=3)\n"
                  "-mmax   max length of randomized text values generated (default=255)\n"
                  "-nlen   display the server response up to len bytes\n"
                  "-ofile  save to file\n"
                  "-pperc  percentage XML kept when using -r (default=100)\n"
                  "-qperc  percentage XML kept of optional indicators when using -r (default=100)\n"
                  "-rseed  randomize XML message templates (use soapcpp2 -g)\n"
                  "-tsec   socket idle timeout seconds (default=1)\n"
                  "-v      verbose mode\n"
                  "-A      use SOAP with MIME attachments (SwA)\n"
                  "-C      use HTTP chunked transfers\n"
                  "-H      add HTTP headers when no URL is specified\n"
                  "-M      use MTOM (application/xop+xml)\n"
                  "-Rrep   set XML element repetition indicator tag (default=" REP ")\n"
                  "-Ssel   set XML element selection indicator tag (default=" SEL ")\n"
                  "infile  XML message template\n"
                  "-       read XML from standard input\n"
                  "URL     endpoint URL of service to test\n"
                  "port    stand-alone server port for clients to test\n\n");
              exit(EXIT_SUCCESS);
            case 'i':
              iind = true;
              break;
            case 'j':
              jind = true;
              break;
            case 'l':
              a++;
              f = false;
              if (*a)
                lmax = soap_strtoull(a, NULL, 10);
              else if (i < argc)
                lmax = strtoull(argv[++i], NULL, 10);
              break;
            case 'm':
              a++;
              f = false;
              if (*a)
                mmax = soap_strtoull(a, NULL, 10);
              else if (i < argc)
                mmax = strtoull(argv[++i], NULL, 10);
              break;
            case 'n':
              a++;
              f = false;
              if (*a)
                nlen = soap_strtoll(a, NULL, 10);
              else if (i < argc)
                nlen = strtoll(argv[++i], NULL, 10);
              break;
            case 'o':
              a++;
              f = false;
              if (*a)
                ofile = a;
              else if (i < argc)
                ofile = argv[++i];
              break;
            case 'p':
              a++;
              f = false;
              if (*a)
                pperc = soap_strtoul(a, NULL, 10);
              else if (i < argc)
                pperc = strtoul(argv[++i], NULL, 10);
              if (pperc > 100)
              {
                fprintf(stderr, "testmsgr: -p option percentage value exceeds 100\n\n");
                exit(EXIT_FAILURE);
              }
              break;
            case 'q':
              a++;
              f = false;
              if (*a)
                qperc = soap_strtoul(a, NULL, 10);
              else if (i < argc)
                qperc = strtoul(argv[++i], NULL, 10);
              if (qperc > 100)
              {
                fprintf(stderr, "testmsgr: -q option percentage value exceeds 100\n\n");
                exit(EXIT_FAILURE);
              }
              break;
            case 'r':
              a++;
              f = false;
              if (*a)
                seed = soap_strtol(a, NULL, 10);
              else if (i < argc)
                seed = strtol(argv[++i], NULL, 10);
              break;
            case 's':
              a++;
              f = false;
              if (*a)
                tsec = soap_strtol(a, NULL, 10);
              else if (i < argc)
                tsec = strtol(argv[++i], NULL, 10);
              break;
            case 'v':
              verb = true;
              break;
            case 'A':
              mime = true;
              http = true;
              break;
            case 'C':
              http = true;
              chnk = true;
              break;
            case 'H':
              http = true;
              break;
            case 'M':
              mtom = true;
              http = true;
              break;
            case 'R':
              a++;
              f = false;
              if (*a)
                rep = a;
              else if (i < argc)
                rep = argv[++i];
              break;
            case 'S':
              a++;
              f = false;
              if (*a)
                sel = a;
              else if (i < argc)
                sel = argv[++i];
              break;
            default:
              fprintf(stderr, "testmsgr: Unknown option %s\n\n", a);
              exit(EXIT_FAILURE);
          }
        }
      }
      else
      {
        if (!ifile)
        {
          ifile = argv[i];
        }
        else if (URL)
        {
          fprintf(stderr, "testmsgr: server URL already specified as %s, ignoring %s\n\n", URL, a);
        }
        else if (port)
        {
          fprintf(stderr, "testmsgr: port already specified as %lu, ignoring %s\n\n", port, a);
        }
        else
        {
          char *r = NULL;
          unsigned long n = strtoul(argv[i], &r, 10);
          if (r && !*r && n)
            port = n;
          else
            URL = argv[i];
        }
      }
    }
  }
  else
  {
    ifile = "-";
  }

  if (!URL)
    cont = false;

#ifndef WIN2
  signal(SIGPIPE, sigpipe_handler);
#endif

  soap *ctx = soap_new1(SOAP_IO_KEEPALIVE | SOAP_C_UTFSTRING | SOAP_DOM_ASIS);

  ctx->send_timeout = tsec;
  ctx->recv_timeout = tsec;

  std::ifstream ifs;

  if (ifile)
  {
    if (!strcmp(ifile, "-"))
    {
      ctx->is = &std::cin;
    }
    else
    {
      ifs.open(ifile, std::ifstream::in);
      if (!ifs.is_open())
      {
        fprintf(stderr, "testmsgr: failed to open %s for reading\n", ifile);
        exit(EXIT_FAILURE);
      }
      ctx->is = &ifs;
    }
  }

  std::ofstream ofs;

  if (ofile)
  {
    ofs.open(ofile, std::ofstream::out);
    if (!ofs.is_open())
    {
      fprintf(stderr, "testmsgr: failed to open %s for writing\n", ofile);
      exit(EXIT_FAILURE);
    }
    ctx->os = &ofs;
  }

  xsd__anyType dom(ctx);

  if (ifile)
  {
    if (soap_read_xsd__anyType(ctx, &dom))
    {
      soap_print_fault(ctx, stderr);
      soap_print_fault_location(ctx, stderr);
      exit(EXIT_FAILURE);
    }
    if (!dom.tag())
    {
      fprintf(stderr, "testmsgr: No root element in %s\n", ifile);
      exit(EXIT_FAILURE);
    }
  }

  if (ifile && strcmp(ifile, "-"))
  {
    ctx->is = NULL;
    ifs.close();
  }

  if (chnk)
    soap_set_omode(ctx, SOAP_IO_CHUNK);
  else
    soap_set_omode(ctx, SOAP_IO_STORE);

  if (port)
    test_server(ctx, dom);
  else
    test_client(ctx, dom);

  if (ofile)
  {
    ctx->os = NULL;
    ofs.close();
  }

  soap_destroy(ctx);
  soap_end(ctx);
  soap_free(ctx);

  return 0;
}

static void test_server(struct soap *ctx, xsd__anyType& dom)
{
  struct soap *soap = soap_copy(ctx);

  SOAP_SOCKET m = soap_bind(soap, NULL, (int)port, 100);

  if (!soap_valid_socket(m))
  {
    soap_print_fault(soap, stderr);
  }
  else
  {
    for (;;)
    {
      SOAP_SOCKET s = soap_accept(soap);

      if (!soap_valid_socket(s))
      {
        soap_print_fault(soap, stderr);
        break;
      }

      if (receive_request(soap) == SOAP_OK)
      {
        if (mime)
        {
          soap_set_mime(soap, NULL, NULL);
        }
        else if (mtom)
        {
          soap_set_omode(soap, SOAP_ENC_MTOM);
          soap_set_mime(soap, NULL, NULL);
        }

        if (soap_response(soap, SOAP_OK))
        {
          fprintf(stderr, "testmsgr: failed to send response\n");
          soap_print_fault(soap, stderr);
        }
        else
        {
          gen_test(soap, dom);

          if (soap_end_send(soap))
          {
            fprintf(stderr, "testmsgr: failed to send response\n");
            soap_print_fault(soap, stderr);
          }
        }
      }

      soap_closesock(soap);
      soap_destroy(soap);
      soap_end(soap);
    }
  }

  soap_destroy(soap);
  soap_end(soap);
  soap_free(soap);
}

static void test_client(struct soap *ctx, xsd__anyType& dom)
{
  struct soap *soap = soap_copy(ctx);

  if (mime)
  {
    soap_set_mime(soap, NULL, NULL);
  }
  else if (mtom)
  {
    soap_set_omode(soap, SOAP_ENC_MTOM);
    soap_set_mime(soap, NULL, NULL);
  }

#ifdef WITH_OPENSSL
  if (URL && !strncmp(URL, "https://", 8))
  {
    if (soap_ssl_client_context(
          soap,
          SOAP_SSL_DEFAULT,
          NULL,
          NULL,
          "cacert.pem",
          NULL,
          NULL
          ))
    {
      fprintf(stderr, "testmsgr: failed to initialize OpenSSL\n");
      soap_print_fault(soap, stderr);
      exit(EXIT_FAILURE);
    }
  }
#endif

  ULONG64 i = 1;
  do
  {
    if (cont)
      printf("Iteration " SOAP_ULONG_FORMAT "\n", i);

    if (URL)
    {
      if (soap_connect(soap, URL, act))
      {
        fprintf(stderr, "testmsgr: failed to connect to %s\n", URL);
        soap_print_fault(soap, stderr);
        exit(EXIT_FAILURE);
      }
    }
    else if (http)
    {
      if (soap_connect(soap, "http://", act))
      {
        fprintf(stderr, "testmsgr: failed to write to output\n");
        soap_print_fault(soap, stderr);
        exit(EXIT_FAILURE);
      }
    }
    else
    {
      if (soap_begin_send(soap))
      {
        fprintf(stderr, "testmsgr: failed to write to output\n");
        soap_print_fault(soap, stderr);
        exit(EXIT_FAILURE);
      }
    }

    gen_test(soap, dom);

    if (soap_end_send(soap))
    {
      if (URL)
        fprintf(stderr, "testmsgr: failed to send to %s\n", URL);
      else
        fprintf(stderr, "testmsgr: failed to write to output\n");
      soap_print_fault(soap, stderr);
      exit(EXIT_FAILURE);
    }

    if (URL)
      receive_response(soap);

    ++i;

    soap_destroy(soap);
    soap_end(soap);

  } while (cont);

  soap_destroy(soap);
  soap_end(soap);
  soap_free(soap);
}

static void gen_test(struct soap *ctx, xsd__anyType& dom)
{
  soap_set_version(ctx, 0);

  // generate MIME protocol headers (no SOAP envelope)
  if (soap_envelope_begin_out(ctx))
  {
    if (URL)
      fprintf(stderr, "testmsgr: failed to send to %s\n", URL);
    else
      fprintf(stderr, "testmsgr: failed to write output\n");
    soap_print_fault(ctx, stderr);
    exit(EXIT_FAILURE);
  }

  if (seed)
  {
    fprintf(stderr, "Random seed = %lu\n", seed);
#ifdef HAVE_RANDOM
    srandom(seed);
    seed = random();
#else
    srand((int)seed);
    seed = rand();
#endif
    cid = 0;
    gen_random_test(ctx, dom);
    gen_random_attachments(ctx);
  }
  else if (soap_out_xsd__anyType(ctx, NULL, 0, &dom, NULL))
  {
    if (URL)
      fprintf(stderr, "testmsgr: failed to send to %s\n", URL);
    else
      fprintf(stderr, "testmsgr: failed to write output\n");
    soap_print_fault(ctx, stderr);
    exit(EXIT_FAILURE);
  }
}

static void gen_random_test(struct soap *ctx, xsd__anyType& dom, size_t tab)
{
  if (pperc == 100 || gen_random() % 100 < pperc)
  {
    xsd__anyType::iterator i = dom.elt_begin();
    xsd__anyType::iterator end = dom.elt_end();

    if (dom.tag() && !strcmp(dom.tag(), rep))
    {
      unsigned long n, m;
      get_minmax(ctx, dom, n, m);
      if (m > lmax)
      {
        if (n > lmax)
          m = n;
        else
          m = lmax;
      }
      for (unsigned long j = 0; j < m; ++j)
      {
        xsd__anyType::iterator k = i;
        if (j < n || qperc == 100 || gen_random() % 100 < qperc)
        {
          while (k != end)
            gen_random_test(ctx, *k++, tab);
          if (dom.get_text())
            gen_value(ctx, dom.get_text());
        }
      }
    }
    else if (dom.tag() && !strcmp(dom.tag(), sel))
    {
      for (unsigned long n = gen_random() % (unsigned long)dom.elt_size(); n > 0; --n)
        ++i;
      gen_random_test(ctx, *i, tab);
    }
    else
    {
      const char *t = NULL;

      if (i != end)
        t = i->get_text();

      if (!t || strcmp(t, "???") || qperc == 100 || gen_random() % 100 < qperc)
      {
        if (dom.tag())
        {
          gen_send(ctx, "<");
          gen_send(ctx, dom.tag());

          for (xsd__anyAttribute::iterator j = dom.att_begin(); j != dom.att_end(); ++j)
          {
            if (!strncmp(j->tag(), "xml", 3) || pperc == 100 || gen_random() % 100 < pperc)
            {
              const char *t = j->get_text();

              if (!t || strncmp(t, "???", 3) || qperc == 100 || gen_random() % 100 < qperc)
              {
                gen_send(ctx, " ");
                gen_send(ctx, j->tag());

                if (t)
                {
                  gen_send(ctx, "=\"");
                  gen_value(ctx, t);
                  gen_send(ctx, "\"");
                }
              }
            }
          }
          t = dom.get_text();
          if ((mime || mtom) && t && !strcmp(t, "%[[BASE64]]%"))
          {
            if (mtom)
            {
              gen_send(ctx, ">");
              gen_indent(ctx, tab + 1);
              gen_send(ctx, "<xop:Include xmlns:xop=\"http://www.w3.org/2004/08/xop/include\"");
            }
            (SOAP_SNPRINTF(ctx->tmpbuf, sizeof(ctx->tmpbuf), 24), "%zu", cid++);
            gen_send(ctx, " href=\"cid:id");
            gen_send(ctx, ctx->tmpbuf);
            gen_send(ctx, "\"/>");
            if (mtom)
            {
              gen_indent(ctx, tab);
              gen_send(ctx, "</");
              gen_send(ctx, dom.tag());
              gen_send(ctx, ">");
            }
          }
          else
          {
            gen_send(ctx, ">");

            while (i != end)
            {
              gen_indent(ctx, tab + 1);
              gen_random_test(ctx, *i++, tab + 1);
            }

            if (t)
              gen_value(ctx, t);
            else
              gen_indent(ctx, tab);

            gen_send(ctx, "</");
            gen_send(ctx, dom.tag());
            gen_send(ctx, ">");
          }
        }
        else
        {
          t = dom.get_text();
          if (t && strcmp(t, "???"))
            gen_value(ctx, t);
        }
      }
    }
  }
}

static void gen_random_attachments(struct soap *ctx)
{
  for (size_t i = 0; i < cid; ++i)
  {
    struct soap_multipart content;
    content.type = "text/plain";
    content.encoding = SOAP_MIME_BINARY;
    (SOAP_SNPRINTF(ctx->tmpbuf, sizeof(ctx->tmpbuf), 24), "<id%zu>", i);
    content.id = ctx->tmpbuf;
    content.location = NULL;
    content.description = NULL;
    if (soap_putmimehdr(ctx, &content))
    {
      if (URL)
        fprintf(stderr, "testmsgr: failed to send to %s\n", URL);
      else
        fprintf(stderr, "testmsgr: failed to write output\n");
      soap_print_fault(ctx, stderr);
      exit(EXIT_FAILURE);
    }
    gen_value(ctx, "%[[TEXT]]%");
  }
}

static void get_minmax(struct soap *ctx, xsd__anyType& dom, unsigned long& n, unsigned long& m)
{
  (void)ctx;
  n = m = 1;
  xsd__anyAttribute::iterator i;
  i = dom.att_find("min");
  if (i != dom.att_end())
    n = soap_strtoul(i->get_text(), NULL, 10);
  i = dom.att_find("max");
  if (i != dom.att_end())
  {
    const char *t = i->get_text();
    if (!strcmp(t, "unbounded"))
      m = 4294967295;
    else
      m = soap_strtoul(i->get_text(), NULL, 10);
  }
  if (n > m)
    m = n;
}

static void gen_value(struct soap *ctx, const char *s)
{
  if (!strncmp(s, "???", 3))
  {
    if (qperc < 100 && gen_random() % 100 >= qperc)
      return;
    s += 3;
  }

  if (s[0] == '%' && s[1] == '[')
  {
    long x = gen_random();

    if (!strcmp(s, "%[[BOOL]]%"))
    {
      gen_send_verb(ctx, s, x % 2 ? "true" : "false");
      return;
    }

    if (!strcmp(s, "%[[INT8]]%") || !strcmp(s, "%[[BYTE]]%"))
    {
      char y;
      switch (x % 4)
      {
        case 0:
          gen_send_verb(ctx, s, "0");
          return;
        case 1:
          gen_send_verb(ctx, s, "-128");
          return;
        case 2:
          gen_send_verb(ctx, s, "127");
          return;
        default:
          y = (char)x;
          (SOAP_SNPRINTF(ctx->tmpbuf, sizeof(ctx->tmpbuf), 24), "%d", (int)y);
          gen_send_verb(ctx, s, ctx->tmpbuf);
          return;
      }
    }

    if (!strcmp(s, "%[[UINT8]]%") || !strcmp(s, "%[[UBYTE]]%"))
    {
      unsigned char y;
      switch (x % 4)
      {
        case 0:
          gen_send_verb(ctx, s, "0");
          return;
        case 1:
          gen_send_verb(ctx, s, "255");
          return;
        default:
          y = (unsigned char)x;
          (SOAP_SNPRINTF(ctx->tmpbuf, sizeof(ctx->tmpbuf), 24), "%u", (unsigned int)y);
          gen_send_verb(ctx, s, ctx->tmpbuf);
          return;
      }
    }

    if (!strcmp(s, "%[[INT16]]%") || !strcmp(s, "%[[SHORT]]%"))
    {
      short y;
      switch (x % 4)
      {
        case 0:
          gen_send_verb(ctx, s, "0");
          return;
        case 1:
          gen_send_verb(ctx, s, "-32768");
          return;
        case 2:
          gen_send_verb(ctx, s, "32767");
          return;
        default:
          y = (short)x;
          (SOAP_SNPRINTF(ctx->tmpbuf, sizeof(ctx->tmpbuf), 24), "%d", (int)y);
          gen_send_verb(ctx, s, ctx->tmpbuf);
          return;
      }
    }

    if (!strcmp(s, "%[[UINT16]]%") || !strcmp(s, "%[[USHORT]]%"))
    {
      unsigned short y;
      switch (x % 4)
      {
        case 0:
          gen_send_verb(ctx, s, "0");
          return;
        case 1:
          gen_send_verb(ctx, s, "65535");
          return;
        default:
          y = (unsigned short)x;
          (SOAP_SNPRINTF(ctx->tmpbuf, sizeof(ctx->tmpbuf), 24), "%d", (unsigned int)y);
          gen_send_verb(ctx, s, ctx->tmpbuf);
          return;
      }
    }

    if (!strcmp(s, "%[[INT32]]%") || !strcmp(s, "%[[INT]]%"))
    {
      int y;
      switch (x % 4)
      {
        case 0:
          gen_send_verb(ctx, s, "0");
          return;
        case 1:
          gen_send_verb(ctx, s, "-2147483648");
          return;
        case 2:
          gen_send_verb(ctx, s, "2147483647");
          return;
        default:
          y = (int)x;
          (SOAP_SNPRINTF(ctx->tmpbuf, sizeof(ctx->tmpbuf), 24), "%d", y);
          gen_send_verb(ctx, s, ctx->tmpbuf);
          return;
      }
    }

    if (!strcmp(s, "%[[UINT32]]%") || !strcmp(s, "%[[UINT]]%"))
    {
      unsigned int y;
      switch (x % 4)
      {
        case 0:
          gen_send_verb(ctx, s, "0");
          return;
        case 1:
          gen_send_verb(ctx, s, "4294967295");
          return;
        default:
          y = (unsigned int)x;
          (SOAP_SNPRINTF(ctx->tmpbuf, sizeof(ctx->tmpbuf), 24), "%u", y);
          gen_send_verb(ctx, s, ctx->tmpbuf);
          return;
      }
    }

    if (!strcmp(s, "%[[INT64]]%") || !strcmp(s, "%[[LONG]]%"))
    {
      LONG64 y;
      switch (x % 4)
      {
        case 0:
          gen_send_verb(ctx, s, "0");
          return;
        case 1:
          gen_send_verb(ctx, s, "-9223372036854775808");
          return;
        case 2:
          gen_send_verb(ctx, s, "9223372036854775807");
          return;
        default:
          y = (LONG64)x;
          (SOAP_SNPRINTF(ctx->tmpbuf, sizeof(ctx->tmpbuf), 24), SOAP_LONG_FORMAT, y);
          gen_send_verb(ctx, s, ctx->tmpbuf);
          return;
      }
    }

    if (!strcmp(s, "%[[UINT64]]%") || !strcmp(s, "%[[ULONG]]%"))
    {
      ULONG64 y;
      switch (x % 4)
      {
        case 0:
          gen_send_verb(ctx, s, "0");
          return;
        case 1:
          gen_send_verb(ctx, s, "18446744073709551615");
          return;
        default:
          y = (ULONG64)x;
          (SOAP_SNPRINTF(ctx->tmpbuf, sizeof(ctx->tmpbuf), 24), SOAP_ULONG_FORMAT, y);
          gen_send_verb(ctx, s, ctx->tmpbuf);
          return;
      }
    }

    if (!strcmp(s, "%[[FLOAT]]%"))
    {
      float y;
      switch (x % 10)
      {
        case 0:
          gen_send_verb(ctx, s, "NaN");
          return;
        case 1:
          gen_send_verb(ctx, s, "INF");
          return;
        case 2:
          gen_send_verb(ctx, s, "-INF");
          return;
        case 3:
          gen_send_verb(ctx, s, "0.0");
          return;
        case 4:
          gen_send_verb(ctx, s, "-0.0");
          return;
        case 5:
          y = -FLT_MAX;
          (SOAP_SNPRINTF(ctx->tmpbuf, sizeof(ctx->tmpbuf), 24), ctx->float_format, y);
          gen_send_verb(ctx, s, ctx->tmpbuf);
          return;
        case 6:
          y = FLT_MAX;
          (SOAP_SNPRINTF(ctx->tmpbuf, sizeof(ctx->tmpbuf), 24), ctx->float_format, y);
          gen_send_verb(ctx, s, ctx->tmpbuf);
          return;
        default:
          y = (float)x/(float)0xFFFF;
          (SOAP_SNPRINTF(ctx->tmpbuf, sizeof(ctx->tmpbuf), 24), ctx->float_format, y);
          gen_send_verb(ctx, s, ctx->tmpbuf);
          return;
      }
    }

    if (!strcmp(s, "%[[DOUBLE]]%"))
    {
      double y;
      switch (x % 10)
      {
        case 0:
          gen_send_verb(ctx, s, "NaN");
          return;
        case 1:
          gen_send_verb(ctx, s, "INF");
          return;
        case 2:
          gen_send_verb(ctx, s, "-INF");
          return;
        case 3:
          gen_send_verb(ctx, s, "0.0");
          return;
        case 4:
          gen_send_verb(ctx, s, "-0.0");
          return;
        case 5:
          y = -DBL_MAX;
          (SOAP_SNPRINTF(ctx->tmpbuf, sizeof(ctx->tmpbuf), 24), ctx->double_format, y);
          gen_send_verb(ctx, s, ctx->tmpbuf);
          return;
        case 6:
          y = DBL_MAX;
          (SOAP_SNPRINTF(ctx->tmpbuf, sizeof(ctx->tmpbuf), 24), ctx->double_format, y);
          gen_send_verb(ctx, s, ctx->tmpbuf);
          return;
        default:
          y = (double)x/(double)0xFFFF;
          (SOAP_SNPRINTF(ctx->tmpbuf, sizeof(ctx->tmpbuf), 24), ctx->double_format, y);
          gen_send_verb(ctx, s, ctx->tmpbuf);
          return;
      }
    }

    if (!strcmp(s, "%[[DATETIME]]%"))
    {
      time_t t = (time_t)x;
      switch (x % 4)
      {
        case 0:
          gen_send_verb(ctx, s, "0000-01-00T00:00:00Z");
          return;
        case 1:
          gen_send_verb(ctx, s, "1969-12-31T23:59:59Z");
          return;
        default:
          t = (time_t)x;
          gen_send_verb(ctx, s, soap_dateTime2s(ctx, t));
          return;
      }
    }

    if (!strcmp(s, "%[[DATE]]%"))
    {
      time_t t = (time_t)x;
      char *r;
      switch (x % 4)
      {
        case 0:
          gen_send_verb(ctx, s, "0000-01-00Z");
          return;
        case 1:
          gen_send_verb(ctx, s, "1969-12-31Z");
          return;
        default:
          t = (time_t)x;
          soap_dateTime2s(ctx, t);
          r = strchr(ctx->tmpbuf, 'T');
          r[0] = 'Z';
          r[1] = '\0';
          gen_send_verb(ctx, s, r);
          return;
      }
    }

    if (!strcmp(s, "%[[TIME]]%"))
    {
      time_t t = (time_t)x;
      switch (x % 4)
      {
        case 0:
          gen_send_verb(ctx, s, "00-00-00Z");
          return;
        default:
          t = (time_t)x;
          gen_send_verb(ctx, s, soap_dateTime2s(ctx, t) + 11);
          return;
      }
    }

    if (!strcmp(s, "%[[DURATION]]%"))
    {
      int y;
      switch (x % 2)
      {
        case 0:
          gen_send_verb(ctx, s, "PT0S");
          return;
        default:
          gen_send_verb(ctx, s, "PT");
          y = (int)x;
          (SOAP_SNPRINTF(ctx->tmpbuf, sizeof(ctx->tmpbuf), 24), "%d", y);
          gen_send_verb(ctx, s, ctx->tmpbuf);
          gen_send(ctx, "S");
          return;
      }
    }

    if (!strncmp(s, "%[[HEX", 6))
    {
      if (s[6] == '[')
      {
        unsigned long n, m;
        if (get_range(s + 7, n, m))
        {
          gen_hex(ctx, n + x % (m - n + 1));
          return;
        }
      }
      else if (s[6] == ']' && s[7] == ']' && s[8] == '%' && !s[9])
      {
        switch (x % 6)
        {
          case 0:
            gen_hex(ctx, 0);
            return;
          case 1:
            x %= 16;
            if ((ULONG64)x > mmax)
              x = (long)mmax;
            gen_hex(ctx, x);
            return;
          case 2:
            x %= 256;
            if ((ULONG64)x > mmax)
              x = (long)mmax;
            gen_hex(ctx, x);
            return;
          case 3:
            x %= 65536;
            if ((ULONG64)x > mmax)
              x = (long)mmax;
            gen_hex(ctx, x);
            return;
          case 4:
            if ((ULONG64)x > mmax)
              x = (long)mmax;
            gen_hex(ctx, x);
            return;
          default:
            gen_hex(ctx, mmax);
            return;
        }
      }
    }

    if (!strncmp(s, "%[[BASE64", 9))
    {
      if (s[9] == '[')
      {
        unsigned long n, m;
        if (get_range(s + 10, n, m))
        {
          gen_base64(ctx, n + x % (m - n + 1));
          return;
        }
      }
      else if (s[9] == ']' && s[10] == ']' && s[11] == '%' && !s[12])
      {
        switch (x % 6)
        {
          case 0:
            gen_base64(ctx, 0);
            return;
          case 1:
            x %= 16;
            if ((ULONG64)x > mmax)
              x = (long)mmax;
            gen_base64(ctx, x);
            return;
          case 2:
            x %= 256;
            if ((ULONG64)x > mmax)
              x = (long)mmax;
            gen_base64(ctx, x);
            return;
          case 3:
            x %= 65536;
            if ((ULONG64)x > mmax)
              x = (long)mmax;
            gen_base64(ctx, x);
            return;
          case 4:
            if ((ULONG64)x > mmax)
              x = (long)mmax;
            gen_base64(ctx, x);
            return;
          default:
            gen_base64(ctx, mmax);
            return;
        }
      }
    }

    if (!strncmp(s, "%[[ENTITY", 9) || !strncmp(s, "%[[ID", 5) || !strncmp(s, "%[[IDREF", 8) || !strncmp(s, "%[[NAME", 7) || !strncmp(s, "%[[NCNAME", 9) || !strncmp(s, "%[[NMTOKEN", 10))
    {
      size_t i = s[4] == 'C' ? 9 : s[4] == 'M' ? 10 : s[3] == 'N' ? 7 : s[5] == 'R' ? 8 : s[3] == 'I' ? 5 : 9;
      if (s[i] == '[')
      {
        unsigned long n, m;
        if (get_range(s + i + 1, n, m))
        {
          gen_name(ctx, n + x % (m - n + 1));
          return;
        }
      }
      else if (s[i] == ']' && s[i + 1] == ']' && s[i + 2] == '%' && !s[i + 3])
      {
        switch (x % 6)
        {
          case 0:
            gen_name(ctx, 0);
            return;
          case 1:
            x %= 16;
            if ((ULONG64)x > mmax)
              x = (long)mmax;
            gen_name(ctx, x);
            return;
          case 2:
            x %= 256;
            if ((ULONG64)x > mmax)
              x = (long)mmax;
            gen_name(ctx, x);
            return;
          case 3:
            x %= 65536;
            if ((ULONG64)x > mmax)
              x = (long)mmax;
            gen_name(ctx, x);
            return;
          case 4:
            if ((ULONG64)x > mmax)
              x = (long)mmax;
            gen_name(ctx, x);
            return;
          default:
            gen_name(ctx, mmax);
            return;
        }
      }
    }

    if (!strncmp(s, "%[[LANG", 7))
    {
      if (s[7] == '[')
      {
        unsigned long n, m;
        if (get_range(s + 8, n, m))
        {
          gen_lang(ctx, x, n + x % (m - n + 1));
          return;
        }
      }
      else if (s[7] == ']' && s[8] == ']' && s[9] == '%' && !s[10])
      {
        switch (x % 2)
        {
          case 0:
            gen_send_verb(ctx, s, "en");
            return;
          default:
            gen_lang(ctx, x, 2);
            return;
        }
      }
    }

    if (!strcmp(s, "%[[QNAME]]%"))
    {
      switch (x % 2)
      {
        case 0:
          gen_send_verb(ctx, s, "xsd:string");
          return;
        default:
          gen_qname(ctx, x % 32);
          return;
      }
    }

    if (!strncmp(s, "%[[TEXT", 7))
    {
      if (s[7] == '[')
      {
        unsigned long n, m;
        if (get_range(s + 8, n, m))
        {
          gen_text(ctx, n + x % (m - n + 1));
          return;
        }
      }
      else if (s[7] == ']' && s[8] == ']' && s[9] == '%' && !s[10])
      {
        switch (x % 6)
        {
          case 0:
            gen_text(ctx, 0);
            return;
          case 1:
            x %= 16;
            if ((ULONG64)x > mmax)
              x = (long)mmax;
            gen_text(ctx, x);
            return;
          case 2:
            x %= 256;
            if ((ULONG64)x > mmax)
              x = (long)mmax;
            gen_text(ctx, x);
            return;
          case 3:
            x %= 65536;
            if ((ULONG64)x > mmax)
              x = (long)mmax;
            gen_text(ctx, x);
            return;
          case 4:
            if ((ULONG64)x > mmax)
              x = (long)mmax;
            gen_text(ctx, x);
            return;
          default:
            gen_text(ctx, mmax);
            return;
        }
      }
    }

    if ((s[2] == '[' || s[2] == '(') && strchr(s, ':'))
    {
      // inclusive or exclusive numeric range %[[N:M]]% %[(N:M)]% %[(N:M]]% %[[N:M)]%
      char *r;
      double n = strtod(s + 3, &r);
      double m = n;
      if (r && *r == ':')
        m = strtod(r + 1, &r);
      bool isint = !strchr(s, '.');
      if (r && (r[0] == ']' || r[0] == ')') && r[1] == ']' && r[2] == '%' && !r[3] &&
          ( (isint && s[2] == '(' && r[0] == ')' && n + 1 < m) ||
            (isint && s[2] == '(' && r[0] == ']' && n < m) ||
            (isint && s[2] == '[' && r[0] == ')' && n < m) ||
            (isint && s[2] == '[' && r[0] == ']' && n <= m) ||
            (!isint && n <= m)
          ))
      {
        double y;
        for (;;)
        {
          y = exp((double)x/(double)RAND_MAX * log(m - n)) + n;
          if (isint)
            y = round(y);
          if ((s[2] != '(' || y > n) && (r[0] != ')' || y < m))
            break;
          x = gen_random();
        }
        (SOAP_SNPRINTF(ctx->tmpbuf, sizeof(ctx->tmpbuf), 24), ctx->double_format, y);
        gen_send_verb(ctx, s, ctx->tmpbuf);
        return;
      }
    }

    // enumeration %[[A][B][C]]%
    if (s[2] == '[' && strstr(s, "]["))
    {
      size_t k = 1;
      const char *r = s + 1;
      while ((r = strstr(r + 2, "][")) != NULL)
        ++k;
      long y = x % k;
      r = s + 1;
      while (y-- > 0)
        r = strstr(r + 2, "][");
      r += 2;
      const char *t = strstr(r, "][");
      if (t)
      {
        k = t - r;
      }
      else
      {
        k = strlen(r);
        if (k >= 3)
          k -= 3;
      }
      gen_send_verb(ctx, s, r, k);
      return;
    }
  }

  gen_send(ctx, s);
}

static bool get_range(const char *s, unsigned long& n, unsigned long& m)
{
  char *r;
  m = n = soap_strtoul(s, &r, 10);
  if (r && *r == ':')
    m = soap_strtoul(r + 1, &r, 10);
  return r && r[0] == ']' && r[1] == ']' && r[2] == ']' && r[3] == '%' && !r[4];
}

static void gen_name(struct soap *ctx, ULONG64 n)
{
  if (verb)
    printf("%%[[NAME]]%%");
  gen_ncname(ctx, n);
}

static void gen_qname(struct soap *ctx, ULONG64 n)
{
  if (verb)
    printf("%%[[QNAME]]%%");
  gen_ncname(ctx, n);
  gen_send(ctx, ":");
  gen_ncname(ctx, n);
}

static void gen_ncname(struct soap *ctx, ULONG64 n)
{
  for (ULONG64 i = 0; i < n; ++i)
  {
    char c = (n ^ i) % 78 + 45;
    if (!isalnum(c) && c != '.' && c != '-' && c != '_')
      c = 'a' + c % 26;
    if (soap_send_raw(ctx, &c, 1))
    {
      fprintf(stderr, "testmsgr: failed to send to %s\n", URL);
      soap_print_fault(ctx, stderr);
      exit(EXIT_FAILURE);
    }
  }
}

static void gen_lang(struct soap *ctx, ULONG64 x, ULONG64 n)
{
  for (ULONG64 i = 0; i < n; ++i)
  {
    char c = (x ^ i) % 26 + 'a';
    if (soap_send_raw(ctx, &c, 1))
    {
      fprintf(stderr, "testmsgr: failed to send to %s\n", URL);
      soap_print_fault(ctx, stderr);
      exit(EXIT_FAILURE);
    }
  }
}

static void gen_text(struct soap *ctx, ULONG64 n)
{
  if (verb)
    printf("%%[[TEXT]]%% " SOAP_ULONG_FORMAT " bytes\n", n);
  for (ULONG64 i = 0; i < n; ++i)
  {
    char c = (n ^ i) % 95 + 32;
    switch (c)
    {
      case '\'':
        gen_send(ctx, "&apos;");
        break;
      case '"':
        gen_send(ctx, "&quot;");
        break;
      case '&':
        gen_send(ctx, "&amp;");
        break;
      case '<':
        gen_send(ctx, "&lt;");
        break;
      case '>':
        gen_send(ctx, "&gt;");
        break;
      default:
        if (soap_send_raw(ctx, &c, 1))
        {
          fprintf(stderr, "testmsgr: failed to send to %s\n", URL);
          soap_print_fault(ctx, stderr);
          exit(EXIT_FAILURE);
        }
    }
  }
}

static void gen_hex(struct soap *ctx, ULONG64 n)
{
  if (verb)
    printf("%%[[HEX]]%% " SOAP_ULONG_FORMAT " bytes\n", n);
  n /= 2;
  for (ULONG64 i = 0; i < n; ++i)
  {
    int c = (n ^ i) % 256;
    char h[2];
    h[0] = ((c >> 4) & 0xF) + '0';
    if (c > 0x9F)
      h[0] += 7;
    h[1] = (c & 0xF) + '0';
    if ((c & 0xF) > 9)
      h[1] += 7;
    if (soap_send_raw(ctx, h, 2))
    {
      fprintf(stderr, "testmsgr: failed to send to %s\n", URL);
      soap_print_fault(ctx, stderr);
      exit(EXIT_FAILURE);
    }
  }
}

static void gen_base64(struct soap *ctx, ULONG64 n)
{
  if (verb)
    printf("%%[[BASE64]]%% " SOAP_ULONG_FORMAT " bytes\n", n);
  unsigned long m;
  char b[4];
  for (ULONG64 i = 0; i < n; i += 3)
  {
    int c = (n ^ i) & 0xFFFFFF;
    m = (c >> 16) & 0xFF;
    m = (m << 8) | ((c >> 8) & 0xFF);
    m = (m << 8) | (c & 0xFF);
    for (int j = 4; j > 0; m >>= 6)
      b[--j] = soap_base64o[m & 0x3F];
    if (soap_send_raw(ctx, b, 4))
    {
      fprintf(stderr, "testmsgr: failed to send to %s\n", URL);
      soap_print_fault(ctx, stderr);
      exit(EXIT_FAILURE);
    }
  }
  n %= 3;
  if (n > 0)
  { int c = n & 0xFFFFFF;
    m = c & 0xFF;
    m <<= 8;
    if (n > 1)
      m |= (c >> 8) & 0xFF;
    m <<= 8;
    for (int j = (int)n; j > 0; m >>= 6)
      b[--j] = soap_base64o[m & 0x3F];
    for (int j = 3; j > (int)n; j--)
      b[j] = '=';
    if (soap_send_raw(ctx, b, 4))
    {
      fprintf(stderr, "testmsgr: failed to send to %s\n", URL);
      soap_print_fault(ctx, stderr);
      exit(EXIT_FAILURE);
    }
  }
}

static int receive_request(struct soap *ctx)
{
  ULONG64 n = 0;
  if (soap_begin_recv(ctx))
  {
    fprintf(stderr, "testmsgr: failed to receive on port %lu\n", port);
    soap_print_fault(ctx, stderr);
    return ctx->error;
  }
  if (!(ctx->mode & SOAP_ENC_ZLIB) && (ctx->mode & SOAP_IO) != SOAP_IO_CHUNK && ctx->length == 0)
  {
    fprintf(stderr, "testmsgr: no HTTP content-length header in request message\n");
  }
  else if ((ctx->mode & SOAP_ENC_MIME))
  {
    // MTOM/MIME attachments: just execute soap_end_recv() to get them all
  }
  else
  {
    for (;;)
    {
      n++;
      if (ctx->length > 0 && n > ctx->length)
        break;
      if ((int)soap_getchar(ctx) == EOF)
        break;
    }
  }
  if (soap_end_recv(ctx))
  {
    fprintf(stderr, "testmsgr: failed to receive on port %lu\n", port);
    soap_print_fault(ctx, stderr);
    return ctx->error;
  }
  return SOAP_OK;
}

static void receive_response(struct soap *ctx)
{
  if (soap_begin_recv(ctx))
  {
    fprintf(stderr, "testmsgr: failed to receive from %s\n", URL);
    soap_print_fault(ctx, stderr);
  }
  else if (!(ctx->mode & SOAP_ENC_ZLIB) && (ctx->mode & SOAP_IO) != SOAP_IO_CHUNK && ctx->length == 0)
  {
    printf("Server responded with status %d and an empty body\n\n", ctx->status);
  }
  else if ((ctx->mode & SOAP_ENC_MIME))
  {
    printf("Server responded with status %d and MIME body:\n\n", ctx->status);
    if (soap_end_recv(ctx))
    {
      fprintf(stderr, "testmsgr: failed to receive from %s\n", URL);
      soap_print_fault(ctx, stderr);
    }
    soap_closesock(ctx);
    for (soap_multipart::iterator i = ctx->mime.begin(); i != ctx->mime.end(); ++i)
    {
      printf("MIME id=\"%s\" type=\"%s\"\n---- begin ----\n", i->id ? i->id : "", i->type ? i->type : "");
      if (i->ptr)
      {
        size_t k = i->size;
        if (nlen > 0 && nlen < (LONG64)k)
          k = (size_t)nlen;
        fwrite(i->ptr, 1, k, stdout);
        if (k < i->size)
          printf("\n---- cut ----\n...");
      }
      printf("\n---- end ----\n\n");
    }
    return;
  }
  else
  {
    printf("Server responded with status %d and a body:\n\n---- begin ----\n", ctx->status);
    ULONG64 n = 0;
    for (;;)
    {
      soap_wchar c;
      n++;
      if (ctx->length > 0 && n > ctx->length)
        break;
      c = soap_getchar(ctx);
      if ((int)c == EOF)
        break;
      if (nlen < 0 || n <= (ULONG64)nlen)
        putchar(c);
    }
    if (nlen > 0 && n > (ULONG64)nlen)
      printf("\n---- cut ----\n...");
    printf("\n---- end ----\n\n");
  }
  if (soap_end_recv(ctx))
  {
    fprintf(stderr, "testmsgr: failed to receive from %s\n", URL);
    soap_print_fault(ctx, stderr);
  }
  soap_closesock(ctx);
}
