/* 
 * Copyright (C) 2004, 2005 Jean-Yves Lefort <jylefort@brutele.be>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "config.h"
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <gnome.h>
#include <libgnomevfs/gnome-vfs.h>
#include <eel/eel-alert-dialog.h>
#include <translate.h>
#include "egg-editable-toolbar.h"
#include "egg-toolbar-editor.h"
#include "gt-app.h"
#include "gt-conf.h"
#include "gt-util.h"
#ifdef WITH_LANGUAGE_DETECTION
#include "gt-language-detection.h"
#endif
#include "gt-preferences.h"
#include "gt-language-view.h"
#include "gt-twp.h"
#include "gt-shell.h"
#include "gt-stock.h"

#define STATUS_PUSH(cid, str) \
  gtk_statusbar_push(GTK_STATUSBAR(app.statusbar), (cid), (str))
#define STATUS_POP(cid) \
  gtk_statusbar_pop(GTK_STATUSBAR(app.statusbar), (cid))
#define GET_ACTION(name) \
  gtk_action_group_get_action(action_group, (name))
#define FULLSCREEN \
  (app.leave_fullscreen_popup != NULL)

typedef struct
{
  GtkWidget	*menubar;
  GtkWidget	*toolbar;
  GtkWidget	*statusbar;
  GtkWidget	*progressbar;
  GtkWidget	*paned;

  GtkWidget	*source_language_view;
  GtkWidget	*source_text_view;

  GtkWidget	*dest_language_view;
  GtkWidget	*dest_text_view;

  GtkWidget	*save_as_dialog;
  GtkWidget	*insert_from_file_dialog;
  GtkWidget	*edit_toolbars_dialog;
  GtkWidget	*edit_toolbars_editor;
  GtkWidget	*about_dialog;
  GtkWidget	*leave_fullscreen_popup;
} App;

typedef struct
{
  TranslateSession		*session;
  char				*text;
  char				*from;
  char				*to;
  double			start_time;
  gboolean			cancelled;
} TranslateInfo;

static GtkUIManager *ui_manager = NULL;
static GtkActionGroup *action_group = NULL;

static char *toolbars_file = NULL;
static EggToolbarsModel *toolbars_model = NULL;
static unsigned int toolbars_model_save_idle_id = 0;

static App app = { NULL };

static unsigned int help_cid = 0;
static unsigned int task_cid = 0;

static TranslateInfo *current_translation = NULL;
static unsigned int stalled_timeout_id = 0;

static gboolean leave_fullscreen_popup_visible = FALSE;
unsigned int leave_fullscreen_timeout_id = 0;

GtkWindow *gt_app_window = NULL;

/* action callbacks */
static void gt_app_file_save_translated_text_as_activate_h (void);
static void gt_app_file_translate_activate_h (void);
static void gt_app_file_stop_activate_h (void);
static void gt_app_file_translate_web_page_activate_h (void);
static void gt_app_file_quit_activate_h (void);
static void gt_app_edit_clear_activate_h (void);
static void gt_app_edit_swap_panes_activate_h (void);
static void gt_app_edit_insert_from_file_activate_h (void);
static void gt_app_edit_toolbars_activate_h (void);
static void gt_app_edit_preferences_activate_h (void);
static void gt_app_help_contents_activate_h (void);
static void gt_app_help_about_activate_h (void);

static GtkActionEntry menu_entries[] = {
  /* Toplevel */
  { "File", NULL, N_("_File") },
  { "Edit", NULL, N_("_Edit") },
  { "View", NULL, N_("_View") },
  { "Help", NULL, N_("_Help") },
  
  /* File menu */
  {
    "FileSaveTranslatedTextAs",
    GTK_STOCK_SAVE_AS,
    N_("_Save Translated Text As..."),
    "<shift><control>S",
    N_("Save the translated text to a file"),
    gt_app_file_save_translated_text_as_activate_h
  },
  {
    "FileTranslate",
    GT_STOCK_TRANSLATE,
    N_("_Translate"),
    "<control>T",
    N_("Translate the source text"),
    gt_app_file_translate_activate_h
  },
  {
    "FileStop",
    GTK_STOCK_STOP,
    N_("_Stop"),
    "Escape",
    N_("Stop the current translation"),
    gt_app_file_stop_activate_h
  },
  {
    "FileTranslateWebPage",
    GT_STOCK_TRANSLATE_WEB_PAGE,
    N_("Translate _Web Page..."),
    NULL,
    N_("Translate a web page"),
    gt_app_file_translate_web_page_activate_h
  },
  {
    "FileQuit",
    GTK_STOCK_QUIT,
    N_("_Quit"),
    NULL,
    N_("Quit the application"),
    gt_app_file_quit_activate_h
  },

  /* Edit menu */
  {
    "EditClear",
    GTK_STOCK_CLEAR,
    N_("_Clear"),
    NULL,
    N_("Clear the source text"),
    gt_app_edit_clear_activate_h
  },
  {
    "EditSwapPanes",
    GT_STOCK_SWAP_PANES,
    N_("_Swap Panes"),
    NULL,
    N_("Swap the contents of the source and destination panes"),
    gt_app_edit_swap_panes_activate_h
  },
  {
    "EditInsertFromFile",
    GT_STOCK_INSERT_FROM_FILE,
    N_("Insert _From File..."),
    NULL,
    N_("Insert a file into the source text"),
    gt_app_edit_insert_from_file_activate_h
  },
  {
    "EditToolbars",
    NULL,
    N_("_Toolbars"),
    NULL,
    N_("Customize the toolbars"),
    gt_app_edit_toolbars_activate_h
  },
  {
    "EditPreferences",
    GTK_STOCK_PREFERENCES,
    N_("_Preferences"),
    NULL,
    N_("Configure the application"),
    gt_app_edit_preferences_activate_h
  },

  /* View menu */
  { "ViewToolbarsStyle", NULL, N_("Toolbars Styl_e") },

  /* Help menu */
  {
    "HelpContents",
    GTK_STOCK_HELP,
    N_("_Contents"),
    "F1",
    N_("Display help"),
    gt_app_help_contents_activate_h,
  },
  {
    "HelpAbout",
    GNOME_STOCK_ABOUT,
    N_("_About"),
    NULL,
    N_("Display credits"),
    gt_app_help_about_activate_h
  }
};

static GtkToggleActionEntry toggle_entries[] = {
  /* View menu */
  {
    "ViewToolbars",
    NULL,
    N_("_Toolbars"),
    NULL,
    N_("Show or hide the toolbars"),
    NULL,
    FALSE
  },
  {
    "ViewStatusbar",
    NULL,
    N_("_Statusbar"),
    NULL,
    N_("Show or hide the statusbar"),
    NULL,
    FALSE
  }
};

static GtkRadioActionEntry toolbars_style_entries[] = {
  /* View menu */
  {
    "ViewToolbarsStyleDesktopDefault",
    NULL,
    N_("_Desktop Default"),
    NULL,
    N_("Set the toolbars style to the desktop default setting"),
    GT_TOOLBAR_STYLE_DESKTOP_DEFAULT
  },
  {
    "ViewToolbarsStyleIconsOnly",
    NULL,
    N_("I_cons Only"),
    NULL,
    N_("Only display the toolbars icons"),
    GT_TOOLBAR_STYLE_ICONS
  },
  {
    "ViewToolbarsStyleTextOnly",
    NULL,
    N_("_Text Only"),
    NULL,
    N_("Only display the toolbars text"),
    GT_TOOLBAR_STYLE_TEXT
  },
  {
    "ViewToolbarsStyleTextBelowIcons",
    NULL,
    N_("Text Belo_w Icons"),
    NULL,
    N_("Display the toolbars text below the icons"),
    GT_TOOLBAR_STYLE_BOTH
  },
  {
    "ViewToolbarsStyleTextBesideIcons",
    NULL,
    N_("Text Be_side Icons"),
    NULL,
    N_("Display the toolbars text beside the icons"),
    GT_TOOLBAR_STYLE_BOTH_HORIZ
  }
};
    
static void gt_app_create (void);
static void gt_app_create_toolbars_model (gboolean use_dot_file);

static void gt_app_ui_manager_connect_proxy_h (GtkUIManager *manager,
					       GtkAction *action,
					       GtkWidget *proxy,
					       gpointer user_data);
static void gt_app_ui_manager_disconnect_proxy_h (GtkUIManager *manager,
						  GtkAction *action,
						  GtkWidget *proxy,
						  gpointer user_data);
static void gt_app_menu_item_select_h (GtkMenuItem *item, gpointer user_data);
static void gt_app_menu_item_deselect_h (GtkMenuItem *item, gpointer user_data);

static void gt_app_toolbars_model_save_changes (void);
static gboolean gt_app_toolbars_model_save_changes_cb (gpointer data);

static void gt_app_toolbars_model_update_flags_and_save_changes (void);

static void gt_app_link_view_item (const char *name, GCallback cb);

static void gt_app_update_toolbars_visibility (void);
static void gt_app_update_statusbar_visibility (void);

static void gt_app_update_toolbars_style (void);

#ifdef WITH_LANGUAGE_DETECTION
static void gt_app_source_text_changed_h (GtkTextBuffer *textbuffer,
					  gpointer user_data);
static void gt_app_source_text_detect_result_cb (const char *tag,
						 gpointer user_data);
#endif /* WITH_LANGUAGE_DETECTION */

static void gt_app_insert_from_file (const char *uri);

static void gt_app_update_sensitivity (void);

static gpointer gt_app_translate_thread (gpointer data);
static gboolean gt_app_translate_progress_cb (double progress, gpointer user_data);
static gboolean gt_app_translate_stalled_cb (gpointer data);
static void gt_app_stop_ui (void);

static void gt_app_edit_toolbars_create_editor (void);
static void gt_app_edit_toolbars_weak_notify_cb (gpointer data,
						 GObject *former_object);

static void gt_app_insert_from_file_response_h (GtkDialog *dialog,
						int response,
						gpointer user_data);
static void gt_app_save_as_response_h (GtkDialog *dialog,
				       int response,
				       gpointer user_data);

static void gt_app_fullscreen (void);
static void gt_app_unfullscreen (void);
static gboolean gt_app_leave_fullscreen_timeout_cb (gpointer data);
static gboolean gt_app_leave_fullscreen_enter_notify_event_h (GtkWidget *widget,
							      GdkEventCrossing *event,
							      gpointer user_data);
static gboolean gt_app_leave_fullscreen_leave_notify_event_h (GtkWidget *widget,
							      GdkEventCrossing *event,
							      gpointer user_data);
static void gt_app_update_leave_fullscreen_popup_position (void);
static gboolean gt_app_window_fullscreen_key_press_event_h (GtkWidget *widget,
							    GdkEventKey *event,
							    gpointer user_data);
static void gt_app_window_fullscreen_menubar_deactivate_h (GtkMenuShell *menushell,
							   gpointer user_data);

void
gt_app_main (const char *uri)
{
  if (uri && (g_str_has_prefix(uri, "http://")
	      || g_str_has_prefix(uri, "https://")))
    gt_twp_display(uri, TRUE);
  else
    {
      gt_app_create();
      if (uri)
	gt_app_insert_from_file(uri);
    }

  gtk_main();
}

static void
gt_app_create (void)
{
  GError *err = NULL;
  GtkAccelGroup *accel_group;
  GtkRequisition requisition;
  
  ui_manager = gtk_ui_manager_new();

  g_object_connect(ui_manager,
		   "signal::connect-proxy", gt_app_ui_manager_connect_proxy_h, NULL,
		   "signal::disconnect-proxy", gt_app_ui_manager_disconnect_proxy_h, NULL,
		   NULL);

  action_group = gtk_action_group_new("WindowActions");
  gtk_action_group_set_translation_domain(action_group, NULL);

  gtk_action_group_add_actions(action_group,
			       menu_entries,
			       G_N_ELEMENTS(menu_entries),
			       NULL);
  gtk_action_group_add_toggle_actions(action_group,
				      toggle_entries,
				      G_N_ELEMENTS(toggle_entries),
				      NULL);
  gtk_action_group_add_radio_actions(action_group,
				     toolbars_style_entries,
				     G_N_ELEMENTS(toolbars_style_entries),
				     -1,
				     G_CALLBACK(gt_app_update_toolbars_style),
				     NULL);
  gtk_ui_manager_insert_action_group(ui_manager, action_group, 0);

  g_object_set(G_OBJECT(GET_ACTION("FileSaveTranslatedTextAs")),
	       "short-label", _("Save Translated Text As"),
	       NULL);
  g_object_set(G_OBJECT(GET_ACTION("FileTranslate")),
	       "is-important", TRUE,
	       NULL);
  g_object_set(G_OBJECT(GET_ACTION("FileStop")),
	       "is-important", TRUE,
	       NULL);
  g_object_set(G_OBJECT(GET_ACTION("FileTranslateWebPage")),
	       "is-important", TRUE,
	       "short-label", _("Translate Web Page"),
	       NULL);
  g_object_set(G_OBJECT(GET_ACTION("EditInsertFromFile")),
	       "short-label", _("Insert From File"),
	       NULL);
  
  if (! gtk_ui_manager_add_ui_from_file(ui_manager, GT_UI_FILE("menus.xml"), &err))
    {
      g_critical(_("unable to load menus.xml: %s"), err->message);
      g_error_free(err);
    }
  
  if (! g_file_test(gt_dot_dir(), G_FILE_TEST_IS_DIR))
    {
      if (mkdir(gt_dot_dir(), 0755) < 0)
	gt_error_dialog(gt_app_window,
			_("A directory creation error has occurred"),
			_("Unable to create directory \"%s\": %s."),
			gt_dot_dir(),
			g_strerror(errno));
    }

  toolbars_file = g_build_filename(gt_dot_dir(), "toolbars.xml", NULL);
  gt_app_create_toolbars_model(TRUE);

#define W(name) #name, &app.name
  gt_create_interface("app",
		      "app", &gt_app_window,
		      W(menubar),
		      W(toolbar),
		      W(statusbar),
		      W(paned),
		      W(source_language_view),
		      W(source_text_view),
		      W(dest_language_view),
		      W(dest_text_view),
		      NULL);
#undef W

  g_object_weak_ref(G_OBJECT(gt_app_window), (GWeakNotify) gtk_main_quit, NULL);

  accel_group = gtk_ui_manager_get_accel_group(ui_manager);
  gtk_window_add_accel_group(gt_app_window, accel_group);

  /*
   * ~/.gnome2/accels/gnome-translate is automatically loaded by the
   * libgnomeui module, we just need to save them ourselves.
   */
  g_signal_connect(accel_group, "accel-changed", G_CALLBACK(gnome_accelerators_sync), NULL);

  app.progressbar = gtk_progress_bar_new();
  /*
   * Set the height of the progressbar to the statusbar height, to
   * avoid resizing-flicker when hiding/showing the progressbar.
   */
  gtk_widget_size_request(app.statusbar, &requisition);
  gtk_widget_set_size_request(app.progressbar, -1, requisition.height);
  
  help_cid = gtk_statusbar_get_context_id(GTK_STATUSBAR(app.statusbar), _("Help messages"));
  task_cid = gtk_statusbar_get_context_id(GTK_STATUSBAR(app.statusbar), _("Task messages"));

  gtk_box_pack_end(GTK_BOX(app.statusbar), app.progressbar, FALSE, FALSE, 0);

  gt_language_view_setup_pair(GTK_TREE_VIEW(app.source_language_view),
			      GT_CONF_SOURCE_LANGUAGE,
			      GTK_TREE_VIEW(app.dest_language_view),
			      GT_CONF_DEST_LANGUAGE,
			      TRANSLATE_PAIR_TEXT,
			      gt_app_update_sensitivity);

  gt_conf_link(gt_app_window, GT_CONF_WINDOW,
	       GET_ACTION("ViewToolbars"), GT_CONF_VIEW_TOOLBARS,
	       GET_ACTION("ViewStatusbar"), GT_CONF_VIEW_STATUSBAR,
	       GET_ACTION("ViewToolbarsStyleDesktopDefault"), GT_CONF_TOOLBARS_STYLE, GT_TYPE_TOOLBAR_STYLE,
	       app.paned, GT_CONF_PANED_POSITION, "position",
	       NULL);

  gt_app_link_view_item("ViewToolbars", gt_app_update_toolbars_visibility);
  gt_app_link_view_item("ViewStatusbar", gt_app_update_statusbar_visibility);

  gtk_widget_grab_focus(app.source_text_view);

#ifdef WITH_LANGUAGE_DETECTION
  gt_language_detection_init(gt_app_source_text_detect_result_cb, NULL);
  g_signal_connect(GT_TEXT_BUFFER(app.source_text_view),
		   "changed",
		   G_CALLBACK(gt_app_source_text_changed_h),
		   NULL);
#endif /* WITH_LANGUAGE_DETECTION */

  gt_app_update_sensitivity();
  gtk_widget_show(GTK_WIDGET(gt_app_window));
}

static void
gt_app_create_toolbars_model (gboolean use_dot_file)
{
  if (toolbars_model)
    g_object_unref(toolbars_model);

  toolbars_model = egg_toolbars_model_new();

  g_object_connect(toolbars_model,
		   "signal-after::item-added", gt_app_toolbars_model_save_changes, NULL,
		   "signal-after::item-removed", gt_app_toolbars_model_save_changes, NULL,
		   "signal-after::toolbar-added", gt_app_toolbars_model_update_flags_and_save_changes, NULL,
		   "signal-after::toolbar-removed", gt_app_toolbars_model_update_flags_and_save_changes, NULL,
		   NULL);

  if (! use_dot_file || ! egg_toolbars_model_load(toolbars_model, toolbars_file))
    {
      if (! egg_toolbars_model_load(toolbars_model, GT_UI_FILE("toolbars.xml")))
	g_critical(_("unable to load toolbars.xml"));
    }
  
  if (egg_toolbars_model_n_toolbars(toolbars_model) < 1)
    egg_toolbars_model_add_toolbar(toolbars_model, 0, "DefaultToolbar");

  if (app.toolbar)
    egg_editable_toolbar_set_model(EGG_EDITABLE_TOOLBAR(app.toolbar), toolbars_model);
}

static void
gt_app_ui_manager_connect_proxy_h (GtkUIManager *manager,
				   GtkAction *action,
				   GtkWidget *proxy,
				   gpointer user_data)
{
  if (GTK_IS_MENU_ITEM(proxy))
    g_object_connect(proxy,
		     "signal::select", gt_app_menu_item_select_h, NULL,
		     "signal::deselect", gt_app_menu_item_deselect_h, NULL,
		     NULL);
}

static void
gt_app_ui_manager_disconnect_proxy_h (GtkUIManager *manager,
				      GtkAction *action,
				      GtkWidget *proxy,
				      gpointer user_data)
{
  if (GTK_IS_MENU_ITEM(proxy))
    g_object_disconnect(proxy,
			"any-signal", gt_app_menu_item_select_h, NULL,
			"any-signal", gt_app_menu_item_deselect_h, NULL,
			NULL);
}

static void
gt_app_menu_item_select_h (GtkMenuItem *item, gpointer user_data)
{
  GtkAction *action;
  char *tooltip = NULL;

  action = g_object_get_data(G_OBJECT(item), "gtk-action");
  g_return_if_fail(action != NULL);
  
  g_object_get(G_OBJECT(action), "tooltip", &tooltip, NULL);
  if (tooltip)
    {
      STATUS_PUSH(help_cid, tooltip);
      g_free(tooltip);
    }
}

static void
gt_app_menu_item_deselect_h (GtkMenuItem *item, gpointer user_data)
{
  STATUS_POP(help_cid);
}

static void
gt_app_toolbars_model_save_changes (void)
{
  if (! toolbars_model_save_idle_id)
    toolbars_model_save_idle_id = g_idle_add(gt_app_toolbars_model_save_changes_cb, NULL);
}

static gboolean
gt_app_toolbars_model_save_changes_cb (gpointer data)
{
  GDK_THREADS_ENTER();
  egg_toolbars_model_save(toolbars_model, toolbars_file, "1.0");
  toolbars_model_save_idle_id = 0;
  GDK_THREADS_LEAVE();

  return FALSE;			/* remove */
}

static void
gt_app_toolbars_model_update_flags_and_save_changes (void)
{
  int i;
  int n_toolbars;
  EggTbModelFlags flag = EGG_TB_MODEL_ACCEPT_ITEMS_ONLY;

  n_toolbars = egg_toolbars_model_n_toolbars(toolbars_model);
  
  if (n_toolbars <= 1)
    flag |= EGG_TB_MODEL_NOT_REMOVABLE;

  for (i = 0; i < n_toolbars; i++)
    {
      EggTbModelFlags flags;

      flags = egg_toolbars_model_get_flags(toolbars_model, i);
      flags &= ~EGG_TB_MODEL_NOT_REMOVABLE;
      flags |= flag;
      egg_toolbars_model_set_flags(toolbars_model, i, flags);
    }

  gt_app_toolbars_model_save_changes();
}

static void
gt_app_link_view_item (const char *name, GCallback cb)
{
  GtkAction *action;

  g_return_if_fail(name != NULL);
  g_return_if_fail(cb != NULL);

  action = GET_ACTION(name);
  g_return_if_fail(GTK_IS_TOGGLE_ACTION(action));

  cb();
  g_signal_connect(action, "toggled", cb, NULL);
}

static void
gt_app_update_toolbars_visibility (void)
{
  g_object_set(G_OBJECT(app.toolbar), "visible", gtk_toggle_action_get_active(GTK_TOGGLE_ACTION(GET_ACTION("ViewToolbars"))), NULL);
}

static void
gt_app_update_statusbar_visibility (void)
{
  g_object_set(G_OBJECT(app.statusbar), "visible", ! FULLSCREEN && gtk_toggle_action_get_active(GTK_TOGGLE_ACTION(GET_ACTION("ViewStatusbar"))), NULL);
}

static void
gt_app_update_toolbars_style (void)
{
  GTToolbarStyle style;
  EggTbModelFlags flag;
  int n_toolbars;
  int i;

  style = gtk_radio_action_get_current_value(GTK_RADIO_ACTION(GET_ACTION("ViewToolbarsStyleDesktopDefault")));
  if (style == GT_TOOLBAR_STYLE_DESKTOP_DEFAULT)
    {
      GtkToolbarStyle gstyle;

      g_object_get(gtk_widget_get_settings(app.toolbar), "gtk-toolbar-style", &gstyle, NULL);
      switch (gstyle)
	{
	case GTK_TOOLBAR_ICONS:		style = GT_TOOLBAR_STYLE_ICONS; break;
	case GTK_TOOLBAR_TEXT:		style = GT_TOOLBAR_STYLE_TEXT; break;
	case GTK_TOOLBAR_BOTH:		style = GT_TOOLBAR_STYLE_BOTH; break;
	case GTK_TOOLBAR_BOTH_HORIZ:	style = GT_TOOLBAR_STYLE_BOTH_HORIZ; break;
	}
    }

  if (FULLSCREEN && style == GT_TOOLBAR_STYLE_BOTH)
    style = GT_TOOLBAR_STYLE_ICONS;

  flag = 1 << style;
  n_toolbars = egg_toolbars_model_n_toolbars(toolbars_model);

  for (i = 0; i < n_toolbars; i++)
    {
      EggTbModelFlags flags;

      flags = egg_toolbars_model_get_flags(toolbars_model, i);
      flags &= ~ (EGG_TB_MODEL_ICONS_ONLY
		  | EGG_TB_MODEL_TEXT_ONLY
		  | EGG_TB_MODEL_ICONS_TEXT
		  | EGG_TB_MODEL_ICONS_TEXT_HORIZ);
      flags |= flag;
      egg_toolbars_model_set_flags(toolbars_model, i, flags);
    }
}

#ifdef WITH_LANGUAGE_DETECTION
static void
gt_app_source_text_changed_h (GtkTextBuffer *textbuffer, gpointer user_data)
{
  char *text;

  text = gt_text_buffer_get_all_text(GT_TEXT_BUFFER(app.source_text_view));
  gt_language_detection_detect_language(text);
  g_free(text);
}

static void
gt_app_source_text_detect_result_cb (const char *tag, gpointer user_data)
{
  GDK_THREADS_ENTER();
  gt_language_view_select(GTK_TREE_VIEW(app.source_language_view), tag);
  GDK_THREADS_LEAVE();
}
#endif /* WITH_LANGUAGE_DETECTION */

static void
gt_app_insert_from_file (const char *uri)
{
  GError *err = NULL;
  char *contents;
  
  g_return_if_fail(uri != NULL);

  contents = gt_vfs_read_entire_file(uri, &err);
  if (contents)
    {
      gtk_text_buffer_insert_at_cursor(GT_TEXT_BUFFER(app.source_text_view), contents, -1);
      g_free(contents);
    }
  else
    {
      char *primary;

      primary = g_strdup_printf(_("Unable to open \"%s\""), uri);
      gt_error_dialog(gt_app_window, primary, "%s", err->message);
      g_free(primary);
      g_error_free(err);
    }
}

static void
gt_app_update_sensitivity (void)
{
  gboolean translate_sensitive = ! current_translation
    && gt_language_view_has_selected(GTK_TREE_VIEW(app.source_language_view))
    && gt_language_view_has_selected(GTK_TREE_VIEW(app.dest_language_view));
  gboolean stop_sensitive = current_translation != NULL;

  g_object_set(GET_ACTION("FileTranslate"),
	       "sensitive", translate_sensitive,
	       NULL);
  g_object_set(GET_ACTION("FileStop"),
	       "sensitive", stop_sensitive,
	       NULL);
}

static void
gt_app_file_save_translated_text_as_activate_h (void)
{
  if (app.save_as_dialog)
    {
      gtk_window_present(GTK_WINDOW(app.save_as_dialog));
      return;
    }

  app.save_as_dialog = gtk_file_chooser_dialog_new_with_backend(_("Save Translated Text As"),
								gt_app_window,
								GTK_FILE_CHOOSER_ACTION_SAVE,
								"gnome-vfs",
								GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
								GTK_STOCK_SAVE, GTK_RESPONSE_OK,
								NULL);
  eel_add_weak_pointer(&app.save_as_dialog);

  gtk_file_chooser_set_local_only(GTK_FILE_CHOOSER(app.save_as_dialog), FALSE);
  gt_conf_link(app.save_as_dialog, GT_CONF_SAVE_TRANSLATED_TEXT_AS_FOLDER,
	       NULL);

  g_signal_connect(app.save_as_dialog, "response", G_CALLBACK(gt_app_save_as_response_h), NULL);
  gtk_widget_show(app.save_as_dialog);
}

static void
gt_app_file_translate_activate_h (void)
{
  TranslateInfo *info;

  STATUS_PUSH(task_cid, _("Translating..."));
  gtk_widget_show(app.progressbar);
  
  gt_text_buffer_clear(GT_TEXT_BUFFER(app.dest_text_view));

  info = g_new(TranslateInfo, 1);
  info->session = g_object_ref(gt_shell_get_translate_session(gt_shell));
  info->text = gt_text_buffer_get_all_text(GT_TEXT_BUFFER(app.source_text_view));
  info->from = gt_language_view_get_selected(GTK_TREE_VIEW(app.source_language_view));
  info->to = gt_language_view_get_selected(GTK_TREE_VIEW(app.dest_language_view));
  info->cancelled = FALSE;

  g_return_if_fail(info->from != NULL);
  g_return_if_fail(info->to != NULL);

  current_translation = info;

  gt_app_update_sensitivity();
  gt_thread_create(gt_app_window, gt_app_translate_thread, info);
}

static gpointer
gt_app_translate_thread (gpointer data)
{
  TranslateInfo *info = data;
  char *translated;
  GError *err = NULL;

  info->start_time = gt_get_current_time();

  translated = translate_session_translate_text(info->session,
						info->text,
						info->from,
						info->to,
						gt_app_translate_progress_cb,
						info,
						&err);

  GDK_THREADS_ENTER();

  if (! info->cancelled)
    {
      if (translated)
	gtk_text_buffer_set_text(GT_TEXT_BUFFER(app.dest_text_view), translated, -1);
      else
	gt_error_dialog(gt_app_window, _("Unable to translate"), "%s", err->message);

      gt_app_stop_ui();
    }

  g_free(translated);
  if (err)
    g_error_free(err);

  gdk_flush();
  GDK_THREADS_LEAVE();

  g_object_unref(info->session);
  g_free(info->text);
  g_free(info->from);
  g_free(info->to);
  g_free(info);

  return NULL;
}

static gboolean
gt_app_translate_progress_cb (double progress, gpointer user_data)
{
  TranslateInfo *info = user_data;
  gboolean ret;

  GDK_THREADS_ENTER();

  if (info->cancelled)
    ret = FALSE;
  else
    {
      gt_source_remove(&stalled_timeout_id);

      if (progress < 0)
	gtk_progress_bar_pulse(GTK_PROGRESS_BAR(app.progressbar));
      else
	{
	  gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR(app.progressbar), progress);

	  if (progress > 0)
	    {
	      double elapsed;
	      int eta;
	      char *status;

	      elapsed = gt_get_current_time() - info->start_time;
	      eta = elapsed / progress - elapsed;
	      
	      status = g_strdup_printf(ngettext("Translating (about %i second left)...",
						"Translating (about %i seconds left)...",
						eta),
				       eta);
	      STATUS_POP(task_cid);
	      STATUS_PUSH(task_cid, status);
	      g_free(status);
	    }
	}
      
      stalled_timeout_id = g_timeout_add(3000, gt_app_translate_stalled_cb, NULL);

      ret = TRUE;
    }

  gdk_flush();
  GDK_THREADS_LEAVE();

  return ret;
}

static gboolean
gt_app_translate_stalled_cb (gpointer data)
{
  GDK_THREADS_ENTER();

  STATUS_POP(task_cid);
  STATUS_PUSH(task_cid, _("Translating (stalled)..."));

  stalled_timeout_id = 0;

  GDK_THREADS_LEAVE();

  return FALSE;			/* remove timeout */
}

static void
gt_app_stop_ui (void)
{
  gt_source_remove(&stalled_timeout_id);

  current_translation = NULL;
  gt_app_update_sensitivity();

  STATUS_POP(task_cid);
  gtk_widget_hide(app.progressbar);
  gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR(app.progressbar), 0);
}

static void
gt_app_file_stop_activate_h (void)
{
  g_return_if_fail(current_translation != NULL);

  current_translation->cancelled = TRUE;
  gt_app_stop_ui();
}

static void
gt_app_file_translate_web_page_activate_h (void)
{
  gt_twp_display(NULL, FALSE);
}

static void
gt_app_file_quit_activate_h (void)
{
  gtk_widget_destroy(GTK_WIDGET(gt_app_window));
}

static void
gt_app_edit_clear_activate_h (void)
{
  gt_text_buffer_clear(GT_TEXT_BUFFER(app.source_text_view));
}

static void
gt_app_edit_swap_panes_activate_h (void)
{
  char *from;
  char *to;
  char *source_text;
  char *dest_text;

  from = gt_language_view_get_selected(GTK_TREE_VIEW(app.source_language_view));
  to = gt_language_view_get_selected(GTK_TREE_VIEW(app.dest_language_view));
  
  gt_language_view_select(GTK_TREE_VIEW(app.source_language_view), to);
  gt_language_view_select(GTK_TREE_VIEW(app.dest_language_view), from);

  g_free(from);
  g_free(to);

  source_text = gt_text_buffer_get_all_text(GT_TEXT_BUFFER(app.source_text_view));
  dest_text = gt_text_buffer_get_all_text(GT_TEXT_BUFFER(app.dest_text_view));

  gtk_text_buffer_set_text(GT_TEXT_BUFFER(app.source_text_view), dest_text, -1);
  gtk_text_buffer_set_text(GT_TEXT_BUFFER(app.dest_text_view), source_text, -1);

  g_free(source_text);
  g_free(dest_text);
}

static void
gt_app_edit_insert_from_file_activate_h (void)
{
  if (app.insert_from_file_dialog)
    {
      gtk_window_present(GTK_WINDOW(app.insert_from_file_dialog));
      return;
    }

  app.insert_from_file_dialog
    = gtk_file_chooser_dialog_new_with_backend(_("Insert From File"),
					       gt_app_window,
					       GTK_FILE_CHOOSER_ACTION_OPEN,
					       "gnome-vfs",
					       GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
					       GTK_STOCK_OPEN, GTK_RESPONSE_OK,
					       NULL);
  eel_add_weak_pointer(&app.insert_from_file_dialog);

  gtk_file_chooser_set_local_only(GTK_FILE_CHOOSER(app.insert_from_file_dialog), FALSE);
  gt_conf_link(app.insert_from_file_dialog, GT_CONF_INSERT_FROM_FILE_FOLDER,
	       NULL);
						  
  g_signal_connect(app.insert_from_file_dialog, "response", G_CALLBACK(gt_app_insert_from_file_response_h), NULL);
  gtk_widget_show(app.insert_from_file_dialog);
}

static void
gt_app_edit_toolbars_activate_h (void)
{
  if (app.edit_toolbars_dialog)
    {
      gtk_window_present(GTK_WINDOW(app.edit_toolbars_dialog));
      return;
    }
  
  gt_create_interface("edit-toolbars",
		      "dialog", &app.edit_toolbars_dialog,
		      NULL);

  eel_add_weak_pointer(&app.edit_toolbars_dialog);
  g_object_weak_ref(G_OBJECT(app.edit_toolbars_dialog), gt_app_edit_toolbars_weak_notify_cb, NULL);

  gt_app_edit_toolbars_create_editor();
  egg_editable_toolbar_set_edit_mode(EGG_EDITABLE_TOOLBAR(app.toolbar), TRUE);

  gt_conf_link(app.edit_toolbars_dialog, GT_CONF_EDIT_TOOLBARS, NULL);
  gtk_widget_show(app.edit_toolbars_dialog);
}

static void
gt_app_edit_toolbars_create_editor (void)
{
  g_return_if_fail(app.edit_toolbars_dialog != NULL);

  if (app.edit_toolbars_editor)
    gtk_widget_destroy(app.edit_toolbars_editor);

  app.edit_toolbars_editor = egg_toolbar_editor_new(ui_manager, toolbars_model);
  eel_add_weak_pointer(&app.edit_toolbars_editor);

  gtk_container_set_border_width(GTK_CONTAINER(app.edit_toolbars_editor), 5);
  gtk_box_set_spacing(GTK_BOX(app.edit_toolbars_editor), 6);

  egg_toolbar_editor_load_actions(EGG_TOOLBAR_EDITOR(app.edit_toolbars_editor), GT_UI_FILE("toolbars.xml"));

  gtk_container_add(GTK_CONTAINER(GTK_DIALOG(app.edit_toolbars_dialog)->vbox), app.edit_toolbars_editor);
  gtk_widget_show(app.edit_toolbars_editor);
}

static void
gt_app_edit_toolbars_weak_notify_cb (gpointer data, GObject *former_object)
{
  egg_editable_toolbar_set_edit_mode(EGG_EDITABLE_TOOLBAR(app.toolbar), FALSE);
}

static void
gt_app_edit_preferences_activate_h (void)
{
  gt_preferences_display(gt_app_window);
}

static void
gt_app_help_contents_activate_h (void)
{
  gt_display_help(gt_app_window, NULL);
}

static void
gt_app_help_about_activate_h (void)
{
  const char *authors[] = { "Jean-Yves Lefort <jylefort@brutele.be>", NULL };
  const char *documenters[] = { "Jean-Yves Lefort <jylefort@brutele.be>", NULL };
  GdkPixbuf *logo;
  GError *err = NULL;

  if (app.about_dialog)
    {
      gtk_window_present(GTK_WINDOW(app.about_dialog));
      return;
    }
  
  logo = gdk_pixbuf_new_from_file(UIDIR G_DIR_SEPARATOR_S "logo.png", &err);
  if (! logo)
    {
      g_warning(_("unable to load logo: %s"), err->message);
      g_error_free(err);
    }

  app.about_dialog = gnome_about_new(_("GNOME Translate"),
				     VERSION,
				     "Copyright \302\251 2004, 2005 Jean-Yves Lefort",
				     _("A natural language translator"),
				     authors,
				     documenters,
				     /* translator: replace with your name and email */
				     _("Jean-Yves Lefort <jylefort@brutele.be>"),
				     logo);

  if (logo)
    g_object_unref(logo);

  eel_add_weak_pointer(&app.about_dialog);
  
  gtk_window_set_transient_for(GTK_WINDOW(app.about_dialog), gt_app_window);
  gtk_widget_show(app.about_dialog);
}

static void
gt_app_insert_from_file_response_h (GtkDialog *dialog,
				    int response,
				    gpointer user_data)
{
  if (response == GTK_RESPONSE_OK)
    {
      char *uri;
	
      uri = gtk_file_chooser_get_uri(GTK_FILE_CHOOSER(dialog));
      gt_app_insert_from_file(uri);
      g_free(uri);
    }

  gtk_widget_destroy(GTK_WIDGET(dialog));
}

static void
gt_app_save_as_response_h (GtkDialog *dialog,
			   int response,
			   gpointer user_data)
{
  if (response == GTK_RESPONSE_OK)
    {
      char *uri;
      char *text;
      GnomeVFSResult result;
      gboolean save = TRUE;
      
      uri = gtk_file_chooser_get_uri(GTK_FILE_CHOOSER(dialog));
      if (gt_vfs_uri_exists(uri))
	{
	  GtkWidget *confirmation_dialog;
	  char *displayable_uri;
	  char *secondary;

	  displayable_uri = gnome_vfs_format_uri_for_display(uri);
	  secondary = g_strdup_printf(_("The file \"%s\" already exists. If you choose yes, its contents will be lost."), displayable_uri);
	  g_free(displayable_uri);
	  
	  confirmation_dialog = eel_alert_dialog_new(GTK_WINDOW(dialog),
						     GTK_DIALOG_DESTROY_WITH_PARENT,
						     GTK_MESSAGE_WARNING,
						     GTK_BUTTONS_YES_NO,
						     _("Overwrite file?"),
						     secondary,
						     NULL);
	  g_free(secondary);

	  gtk_dialog_set_default_response(GTK_DIALOG(confirmation_dialog), GTK_RESPONSE_NO); /* safe default */
	  if (gtk_dialog_run(GTK_DIALOG(confirmation_dialog)) != GTK_RESPONSE_YES)
	    save = FALSE;
	  
	  gtk_widget_destroy(confirmation_dialog);
	}
      
      if (save)
	{
	  text = gt_text_buffer_get_all_text(GT_TEXT_BUFFER(app.dest_text_view));
	  result = gt_vfs_write_file(uri, text);
	  g_free(text);
	  
	  if (result != GNOME_VFS_OK)
	    {
	      char *primary;

	      primary = g_strdup_printf(_("Unable to save \"%s\""), uri);
	      gt_error_dialog(gt_app_window, primary, "%s", gnome_vfs_result_to_string(result));
	      g_free(primary);
	    }
	}
      
      g_free(uri);
    }

  gtk_widget_destroy(GTK_WIDGET(dialog));
}

static void
gt_app_fullscreen (void)
{
  GtkWidget *button;

  app.leave_fullscreen_popup = gtk_window_new(GTK_WINDOW_POPUP);
  eel_add_weak_pointer(&app.leave_fullscreen_popup);

  button = gtk_button_new_from_stock(GT_STOCK_LEAVE_FULLSCREEN);
  gtk_container_add(GTK_CONTAINER(app.leave_fullscreen_popup), button);
  gtk_widget_show(button);

  g_signal_connect_swapped(button, "clicked", G_CALLBACK(gtk_window_unfullscreen), gt_app_window);

  gtk_widget_hide(app.menubar);
  gt_app_update_toolbars_style();
  gt_app_update_statusbar_visibility();

  leave_fullscreen_popup_visible = TRUE;
  gt_app_update_leave_fullscreen_popup_position();
  gtk_widget_show(app.leave_fullscreen_popup);

  g_signal_connect(gdk_screen_get_default(),
		   "size-changed",
		   G_CALLBACK(gt_app_update_leave_fullscreen_popup_position),
		   NULL);
  g_signal_connect(gt_app_window,
		   "key-press-event",
		   G_CALLBACK(gt_app_window_fullscreen_key_press_event_h),
		   NULL);

  g_object_connect(app.leave_fullscreen_popup,
		   "signal::enter-notify-event", gt_app_leave_fullscreen_enter_notify_event_h, NULL,
		   "signal::leave-notify-event", gt_app_leave_fullscreen_leave_notify_event_h, NULL,
		   NULL);

  leave_fullscreen_timeout_id = g_timeout_add(5000, gt_app_leave_fullscreen_timeout_cb, NULL);
}

static void
gt_app_unfullscreen (void)
{
  g_signal_handlers_disconnect_by_func(gdk_screen_get_default(), gt_app_update_leave_fullscreen_popup_position, NULL);
  g_signal_handlers_disconnect_by_func(gt_app_window, gt_app_window_fullscreen_key_press_event_h, NULL);

  gt_source_remove(&leave_fullscreen_timeout_id);

  gtk_widget_destroy(app.leave_fullscreen_popup);

  gtk_widget_show(app.menubar);
  gt_app_update_toolbars_style();
  gt_app_update_statusbar_visibility();
}

static gboolean
gt_app_leave_fullscreen_timeout_cb (gpointer data)
{
  GDK_THREADS_ENTER();

  leave_fullscreen_popup_visible = FALSE;
  gt_app_update_leave_fullscreen_popup_position();
  leave_fullscreen_timeout_id = 0;

  GDK_THREADS_LEAVE();
  
  return FALSE;			/* remove timeout */
}

static gboolean
gt_app_leave_fullscreen_enter_notify_event_h (GtkWidget *widget,
					      GdkEventCrossing *event,
					      gpointer user_data)
{
  gt_source_remove(&leave_fullscreen_timeout_id);
  leave_fullscreen_popup_visible = TRUE;
  gt_app_update_leave_fullscreen_popup_position();

  return FALSE;			/* propagate event */
}

static gboolean
gt_app_leave_fullscreen_leave_notify_event_h (GtkWidget *widget,
					      GdkEventCrossing *event,
					      gpointer user_data)
{
  gt_source_remove(&leave_fullscreen_timeout_id);
  leave_fullscreen_timeout_id = g_timeout_add(5000, gt_app_leave_fullscreen_timeout_cb, NULL);

  return FALSE;			/* propagate event */
}

static void
gt_app_update_leave_fullscreen_popup_position (void)
{
  GdkScreen *screen;
  int monitor_num;
  GdkRectangle screen_rect;
  int popup_width;
  int popup_height;
  int y;

  g_return_if_fail(app.leave_fullscreen_popup != NULL);

  screen = gdk_screen_get_default();
  monitor_num = gdk_screen_get_monitor_at_window(screen, GTK_WIDGET(gt_app_window)->window);
  gdk_screen_get_monitor_geometry(screen, monitor_num, &screen_rect);
  gtk_window_get_size(GTK_WINDOW(app.leave_fullscreen_popup), &popup_width, &popup_height);

  y = screen_rect.y;
  if (! leave_fullscreen_popup_visible)
    y -= popup_height - 2;

  gtk_window_move(GTK_WINDOW(app.leave_fullscreen_popup), screen_rect.width - popup_width, y);
}

static gboolean
gt_app_window_fullscreen_key_press_event_h (GtkWidget *widget,
					    GdkEventKey *event,
					    gpointer user_data)
{
  unsigned int modifiers = gtk_accelerator_get_default_mod_mask();

  if (event->keyval == GDK_Escape && (event->state & modifiers) == 0)
    {
      gtk_window_unfullscreen(gt_app_window);
      return TRUE;	/* Escape is the shortcut for "Stop", block it */
    }
  else if (event->keyval == GDK_F10 && (event->state & modifiers) == 0)
    {
      g_signal_connect(app.menubar, "deactivate", G_CALLBACK(gt_app_window_fullscreen_menubar_deactivate_h), NULL);

      gtk_widget_show(app.menubar);
      gtk_menu_shell_select_first(GTK_MENU_SHELL(app.menubar), FALSE);

      return TRUE;	/* do not propagate event */
    }
  else
    return FALSE;	/* propagate event */
}

static void
gt_app_window_fullscreen_menubar_deactivate_h (GtkMenuShell *menushell,
					       gpointer user_data)
{
  g_signal_handlers_disconnect_by_func(app.menubar, G_CALLBACK(gt_app_window_fullscreen_menubar_deactivate_h), NULL);
  gtk_menu_shell_deselect(menushell);
  gtk_widget_hide(app.menubar);
}

/* libglade callbacks */

GtkWidget *
gt_app_get_widget_cb (const char *widget_name,
		      const char *string1,
		      const char *string2,
		      int int1,
		      int int2)
{
  GtkWidget *widget;

  widget = gtk_ui_manager_get_widget(ui_manager, string1);
  g_return_val_if_fail(widget != NULL, NULL);

  return widget;
}

GtkWidget *
gt_app_toolbar_new_cb (void)
{
  GtkWidget *toolbar;

  toolbar = egg_editable_toolbar_new_with_model(ui_manager, toolbars_model);
  gtk_widget_show(toolbar);

  return toolbar;
}

void
gt_app_edit_toolbars_response_h (GtkDialog *dialog,
				 int response,
				 gpointer user_data)
{
  switch (response)
    {
    case GTK_RESPONSE_HELP:
      gt_display_help(GTK_WINDOW(app.edit_toolbars_dialog), "edit-toolbars");
      break;

    case 1:			/* use default */
      /* recreate the model, discarding the dot file */
      gt_app_create_toolbars_model(FALSE);
      /* and recreate the editor, since the model has changed */
      gt_app_edit_toolbars_create_editor();
      break;

    case 2:			/* add toolbar */
      egg_toolbars_model_add_toolbar(toolbars_model, -1, "UserCreated");
      break;

    case GTK_RESPONSE_CLOSE:
      gtk_widget_destroy(app.edit_toolbars_dialog);
      break;
    }
}

gboolean
gt_app_window_state_event_h (GtkWidget *widget,
			     GdkEventWindowState *event,
			     gpointer user_data)
{
  if (event->changed_mask & GDK_WINDOW_STATE_FULLSCREEN)
    {
      if (event->new_window_state & GDK_WINDOW_STATE_FULLSCREEN)
	gt_app_fullscreen();
      else
	gt_app_unfullscreen();
    }

  return FALSE;			/* propagate event */
}
