# Copyright (C) 2011-2012 - Curtis Hovey <sinzui.is at verizon.net>
# This software is licensed under the MIT license (see the file COPYING).
"""
Test Gedit fakes.

This set of tests provide general testing of the fake gedit objects
used in testing. The fake object are not intended to be robust. They
exist to mimic basic gedit object behaviours and collaborations.
Dummy value may, and are, used inplace of the basic functions when
needed.
"""

import unittest

from gi.repository import (
    Gio,
    Gtk,
    )

from testing import Gedit


class DocumentTestCase(unittest.TestCase):

    def test_init(self):
        document = Gedit.Document()
        self.assertIs(None, document.uri)
        self.assertIs(None, document.encoding)

    def test_load(self):
        document = Gedit.Document()
        location = Gio.file_new_for_path('plugins/gdp/data/snark.py')
        document.load(location, 'utf-8', 0, False)
        self.assertIn('plugins/gdp/data/snark.py', document.uri)
        self.assertEqual('utf-8', document.encoding)
        self.assertEqual('python', document.get_language().props.id)

    def test_get_mime_type(self):
        location = Gio.file_new_for_path('plugins/gdp/data/snark.py')
        document = Gedit.Document()
        document.load(location, None, 0, False)
        self.assertEqual('text/x-python', document.get_mime_type())


class ViewTestCase(unittest.TestCase):

    def test_init(self):
        document = Gedit.Document()
        view = Gedit.View(document)
        self.assertIs(document, view.get_buffer())


class TabTestCase(unittest.TestCase):

    def test_init(self):
        document = Gedit.Document()
        view = Gedit.View(document)
        tab = Gedit.Tab()
        scrolled_win = Gtk.ScrolledWindow()
        scrolled_win.set_policy(
            Gtk.PolicyType.AUTOMATIC, Gtk.PolicyType.AUTOMATIC)
        scrolled_win.add_with_viewport(view)
        tab.pack_start(scrolled_win, True, True, 0)
        self.assertIs(view, tab.get_view())
        self.assertIs(document, tab.get_document())


class WindowTestCase(unittest.TestCase):

    def test_init(self):
        window = Gedit.Window()
        self.assertIsInstance(window.action_group, Gtk.ActionGroup)

    def test_create_tab(self):
        window = Gedit.Window()
        self.assertEqual(0, len(window.tabs_))
        window.create_tab(False)
        self.assertEqual(1, len(window.tabs_))
        self.assertIsInstance(window.tabs_[0], Gedit.Tab)

    def test_create_tab_from_location(self):
        window = Gedit.Window()
        self.assertEqual(0, len(window.tabs_))
        location = Gio.file_new_for_path('plugins/gdp/data/snark.py')
        window.create_tab_from_location(location, None, 0, 0, False, True)
        self.assertEqual(1, len(window.tabs_))
        self.assertIn(
            'plugins/gdp/data/snark.py',
            window.get_active_tab().get_document().uri)

    def test_get_set_active_tab(self):
        window = Gedit.Window()
        self.assertEqual(0, len(window.tabs_))
        window.create_tab(False)
        window.create_tab(False)
        self.assertEqual(2, len(window.tabs_))
        self.assertIs(window.tabs_[0], window.get_active_tab())
        window.set_active_tab(window.tabs_[1])
        self.assertIs(window.tabs_[1], window.get_active_tab())

    def test_get_active_view(self):
        window = Gedit.Window()
        self.assertEqual(0, len(window.tabs_))
        window.create_tab(False)
        window.create_tab(False)
        window.set_active_tab(window.tabs_[1])
        self.assertIs(window.tabs_[1].get_view(), window.get_active_view())
        self.assertEqual(2, len(window.get_views()))

    def test_get_active_document(self):
        window = Gedit.Window()
        self.assertEqual(0, len(window.tabs_))
        window.create_tab(False)
        window.create_tab(False)
        window.set_active_tab(window.tabs_[1])
        self.assertIs(
            window.tabs_[1].get_document(), window.get_active_document())
        self.assertEqual(2, len(window.get_documents()))


class AppTestCase(unittest.TestCase):

    def test_singleton(self):
        app = Gedit.app_get_default()
        other_app = Gedit.app_get_default()
        self.assertIs(app, other_app)

    def test_create_window(self):
        app = Gedit.app_get_default()
        window = app.create_window()
        self.assertIsInstance(window, Gedit.Window)
