/*
 *
 *  Embedded Linux library
 *
 *  Copyright (C) 2018  Intel Corporation. All rights reserved.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <sys/socket.h>
#include <linux/if_arp.h>
#include <unistd.h>
#include <errno.h>
#include <sys/ioctl.h>

#include "net.h"
#include "private.h"

/**
 * SECTION:net
 * @short_description: Network Interface related utilities
 *
 * Network Interface utilities
 */

/**
 * l_net_get_mac_address:
 * @ifindex: Interface index to query
 * @out_addr: Buffer to copy the mac address to.  Must be able to hold 6 bytes
 *
 * Obtains the mac address of the network interface given by @ifindex
 *
 * Returns: #true on success and #false on failure
 **/
LIB_EXPORT bool l_net_get_mac_address(uint32_t ifindex, uint8_t *out_addr)
{
	struct ifreq ifr;
	int sk, err;

	sk = socket(PF_INET, SOCK_DGRAM | SOCK_CLOEXEC, 0);
	if (sk < 0)
		return false;

	memset(&ifr, 0, sizeof(ifr));
	ifr.ifr_ifindex = ifindex;

	err = ioctl(sk, SIOCGIFNAME, &ifr);
	if (err < 0)
		goto error;

	err = ioctl(sk, SIOCGIFHWADDR, &ifr);
	if (err < 0)
		goto error;

	close(sk);

	if (ifr.ifr_hwaddr.sa_family != ARPHRD_ETHER)
		return false;

	memcpy(out_addr, ifr.ifr_hwaddr.sa_data, 6);
	return true;

error:
	close(sk);
	return false;
}

/**
 * l_net_get_name:
 * @ifindex: Interface index to query
 *
 * Obtains the name of the network inderface given by @ifindex
 *
 * Returns: A newly allocated string with the name or NULL on failure
 **/
LIB_EXPORT char *l_net_get_name(uint32_t ifindex)
{
	struct ifreq ifr;
	int sk, err;

	sk = socket(PF_INET, SOCK_DGRAM | SOCK_CLOEXEC, 0);
	if (sk < 0)
		return NULL;

	memset(&ifr, 0, sizeof(ifr));
	ifr.ifr_ifindex = ifindex;

	err = ioctl(sk, SIOCGIFNAME, &ifr);
	close(sk);

	if (err < 0)
		return NULL;

	return l_strdup(ifr.ifr_name);
}

/**
 * l_net_hostname_is_root:
 * @hostname: Hostname to validate
 *
 * Identifies if the hostname given by @hostname is root domain name or
 * not.
 *
 * Returns: #true if the given hostname is root and #false otherwise.
 **/
LIB_EXPORT bool l_net_hostname_is_root(const char *hostname)
{
	if (unlikely(!hostname))
		return false;

	if (!strcmp(hostname, ""))
		return true;

	if (!strcmp(hostname, "."))
		return true;

	return false;
}

static bool str_has_suffix(const char *str, const char *suffix)
{
	size_t str_len;
	size_t suffix_len;
	size_t len_diff;

	str_len = strlen(str);
	suffix_len = strlen(suffix);

	if (str_len < suffix_len)
		return false;

	len_diff = str_len - suffix_len;

	return !strcasecmp(&str[len_diff], suffix);
}

/**
 * l_net_hostname_is_localhost:
 * @hostname: Hostname to validate
 *
 * Identifies if the hostname given by @hostname is localhost or not.
 *
 * Returns: #true if the given hostname is localhost and #false otherwise.
 **/
LIB_EXPORT bool l_net_hostname_is_localhost(const char *hostname)
{
	if (unlikely(!hostname))
		return false;

	if (!strcasecmp(hostname, "localhost") ||
			!strcasecmp(hostname, "localhost.") ||
			!strcasecmp(hostname, "localhost.localdomain") ||
			!strcasecmp(hostname, "localhost.localdomain."))
		return true;

	if (str_has_suffix(hostname, ".localhost") ||
			str_has_suffix(hostname, ".localhost.") ||
			str_has_suffix(hostname, ".localhost.localdomain") ||
			str_has_suffix(hostname, ".localhost.localdomain."))
		return true;

	return false;
}
