/*
* modulator.h
* DIN Is Noise is copyright (c) 2006-2021 Jagannathan Sampath
* DIN Is Noise is released under GNU Public License 2.0
* For more information, please visit https://dinisnoise.org/
*/
#ifndef __modulator__
#define __modulator__

#include "beat2value.h"
#include "angle.h"
#include "alarm.h"

#include <fstream>
#include <string>

struct multi_curve;
struct drone;

struct autoflipt {
  int yes;
  anglet angle;
  float total;
  autoflipt ();
  void calc (float theta, int& dir);
  void set (float deg);
};

std::istream& operator>> (std::istream& file, autoflipt& af);
std::ostream& operator<< (std::ostream& file, autoflipt& af);

struct autorotator {

  int yes;

  int dir; // +1 = anti-clockwise , -1 = clockwise
  autoflipt autoflip;

  enum {TICK, SMOOTH};
  int movement;

  alarm_t tik;
  float deg;
  float tps;

  float rpm;

  struct anglett {
    float persec;
    float theta;
    anglett () {
      persec = 0.0f;
      theta = 0.0f;
    }
  } angle;

  autorotator () {
    yes = 0;
    dir = -1;
    movement = SMOOTH;
    set_rpm (0.0f);
    deg = 6.0f;
    tps = 1.0f;
  }

  void set_rpm (float r);

  void calc (float dt);

  inline void setdeg (float d) {
    deg = d;
    if (deg < 0.0f) deg = 0.0f;
    extern const float PI_BY_180;
    angle.persec = deg * PI_BY_180;
  }

  inline void settps (float t) {
    tps = t;
    if (tps < 1.0) tps = 1.0;
    tik.triggert = 1.0 / tps;
  }

  void chgdeg (float dd);
  void chgtps (float t);

}; 

std::ostream& operator<< (std::ostream&, autorotator& ar);
std::istream& operator>> (std::istream&, autorotator& ar);

struct mod_params { 

  double t;

  float depth;
	beat2value bv; // rate

	float result;

  // drone modulation direction
  //
  enum {VERTICAL, HORIZONTAL, VELOCITY, ACCELERATION}; // possible modulation directions
  int id; 
  static point<float> vertical, horizontal; // constant directions
  float *dirx, *diry; // current modulation direction
  autorotator autorot; // for autorotating direction

  void setdir (int i, float* x, float* y) {
    id = i;
    dirx = x;
    diry = y;
  }

  void calcdir (drone& d);

  float initial; // for microtonal range modulation

	mod_params (multi_curve* c, int d, point<float>& p, const std::string& n = "noname");
	void calc ();
	void clear ();

};

struct modulator {

  enum {AM, FM};
  mod_params am, fm;
  mod_params* arr[2];
  
	enum {FROZEN=-1, NOT_ACTIVE, ACTIVE};
  int active;

	int scrubbed; // used by point_modulator

  modulator (multi_curve* fmc, multi_curve* amc) : 
    am (amc, mod_params::VERTICAL, mod_params::vertical), 
    fm (fmc, mod_params::HORIZONTAL, mod_params::horizontal) 
  {
    active = scrubbed = 0;
    arr[0] = &am;
    arr[1] = &fm;
  }

  void clear () {
		am.clear ();
    fm.clear ();
    active = 0;
  }

  void calc () {
		am.calc ();
    fm.calc ();
  }

};

#endif
