# -*- coding: utf-8 -*-
"""The SleuthKit (TSK) partition file system implementation."""

import pytsk3

# This is necessary to prevent a circular import.
import dfvfs.vfs.tsk_partition_file_entry

from dfvfs.lib import definitions
from dfvfs.lib import errors
from dfvfs.lib import tsk_image
from dfvfs.lib import tsk_partition
from dfvfs.path import tsk_partition_path_spec
from dfvfs.resolver import resolver
from dfvfs.vfs import file_system


class TSKPartitionFileSystem(file_system.FileSystem):
  """Class that implements a file system object using pytsk3."""

  LOCATION_ROOT = u'/'
  TYPE_INDICATOR = definitions.TYPE_INDICATOR_TSK_PARTITION

  def __init__(self, resolver_context):
    """Initializes a file system object.

    Args:
      resolver_context: the resolver context (instance of resolver.Context).
    """
    super(TSKPartitionFileSystem, self).__init__(resolver_context)
    self._file_object = None
    self._tsk_volume = None

  def _Close(self):
    """Closes the file system object.

    Raises:
      IOError: if the close failed.
    """
    self._tsk_volume = None

    self._file_object.close()
    self._file_object = None

  def _Open(self, path_spec, mode='rb'):
    """Opens the file system object defined by path specification.

    Args:
      path_spec: a path specification (instance of path.PathSpec).
      mode: optional file access mode. The default is 'rb' read-only binary.

    Raises:
      AccessError: if the access to open the file was denied.
      IOError: if the file system object could not be opened.
      PathSpecError: if the path specification is incorrect.
      ValueError: if the path specification is invalid.
    """
    if not path_spec.HasParent():
      raise errors.PathSpecError(
          u'Unsupported path specification without parent.')

    file_object = resolver.Resolver.OpenFileObject(
        path_spec.parent, resolver_context=self._resolver_context)

    try:
      tsk_image_object = tsk_image.TSKFileSystemImage(file_object)
      tsk_volume = pytsk3.Volume_Info(tsk_image_object)
    except:
      file_object.close()
      raise

    self._file_object = file_object
    self._tsk_volume = tsk_volume

  def FileEntryExistsByPathSpec(self, path_spec):
    """Determines if a file entry for a path specification exists.

    Args:
      path_spec: a path specification (instance of path.PathSpec).

    Returns:
      Boolean indicating if the file entry exists.
    """
    tsk_vs_part, _ = tsk_partition.GetTSKVsPartByPathSpec(
        self._tsk_volume, path_spec)

    # The virtual root file has not corresponding TSK volume system part object
    # but should have a location.
    if tsk_vs_part is None:
      location = getattr(path_spec, u'location', None)
      return location is not None and location == self.LOCATION_ROOT

    return True

  def GetFileEntryByPathSpec(self, path_spec):
    """Retrieves a file entry for a path specification.

    Args:
      path_spec: a path specification (instance of path.PathSpec).

    Returns:
      A file entry (instance of vfs.TSKPartitionFileEntry) or None.
    """
    tsk_vs_part, partition_index = tsk_partition.GetTSKVsPartByPathSpec(
        self._tsk_volume, path_spec)

    location = getattr(path_spec, u'location', None)

    # The virtual root file has not corresponding TSK volume system part object
    # but should have a location.
    if tsk_vs_part is None:
      if location is None or location != self.LOCATION_ROOT:
        return
      return dfvfs.vfs.tsk_partition_file_entry.TSKPartitionFileEntry(
          self._resolver_context, self, path_spec, is_root=True,
          is_virtual=True)

    if location is None and partition_index is not None:
      path_spec.location = u'/p{0:d}'.format(partition_index)

    return dfvfs.vfs.tsk_partition_file_entry.TSKPartitionFileEntry(
        self._resolver_context, self, path_spec)

  def GetRootFileEntry(self):
    """Retrieves the root file entry.

    Returns:
      A file entry (instance of vfs.FileEntry).
    """
    path_spec = tsk_partition_path_spec.TSKPartitionPathSpec(
        location=self.LOCATION_ROOT, parent=self._path_spec.parent)
    return self.GetFileEntryByPathSpec(path_spec)

  def GetTSKVolume(self):
    """Retrieves the TSK volume object.

    Returns:
      The TSK volume object (instance of pytsk3.Volume_Info).
    """
    return self._tsk_volume
