#include <arpa/inet.h>
#include <netinet/in.h>
#include <stdint.h>
#include <sys/socket.h>

#include "client.h"
#include "log.h"
#include "markpool.h"
#include "state.pb.h"
#include "stream.h"

#ifndef SO_MAX_PACING_RATE
#define SO_MAX_PACING_RATE 47
#endif

using namespace std;

Client::Client(int sock)
	: sock(sock),
	  fwmark(0),
	  connect_time(time(NULL)),
	  state(Client::READING_REQUEST),
	  stream(NULL),
	  header_or_error_bytes_sent(0),
	  stream_pos(0),
	  bytes_sent(0),
	  bytes_lost(0),
	  num_loss_events(0)
{
	request.reserve(1024);

	// Find the remote address, and convert it to ASCII.
	sockaddr_in6 addr;
	socklen_t addr_len = sizeof(addr);

	if (getpeername(sock, reinterpret_cast<sockaddr *>(&addr), &addr_len) == -1) {
		log_perror("getpeername");
		remote_addr = "";
		return;
	}

	char buf[INET6_ADDRSTRLEN];
	if (IN6_IS_ADDR_V4MAPPED(&addr.sin6_addr)) {
		// IPv4 address, really.
		if (inet_ntop(AF_INET, &addr.sin6_addr.s6_addr32[3], buf, sizeof(buf)) == NULL) {
			log_perror("inet_ntop");
			remote_addr = "";
		} else {
			remote_addr = buf;
		}
	} else {
		if (inet_ntop(addr.sin6_family, &addr.sin6_addr, buf, sizeof(buf)) == NULL) {
			log_perror("inet_ntop");
			remote_addr = "";
		} else {
			remote_addr = buf;
		}
	}
}
	
Client::Client(const ClientProto &serialized, Stream *stream)
	: sock(serialized.sock()),
	  remote_addr(serialized.remote_addr()),
	  connect_time(serialized.connect_time()),
	  state(State(serialized.state())),
	  request(serialized.request()),
	  url(serialized.url()),
	  stream(stream),
	  header_or_error(serialized.header_or_error()),
	  header_or_error_bytes_sent(serialized.header_or_error_bytes_sent()),
	  stream_pos(serialized.stream_pos()),
	  bytes_sent(serialized.bytes_sent()),
	  bytes_lost(serialized.bytes_lost()),
	  num_loss_events(serialized.num_loss_events())
{
	if (stream != NULL && stream->mark_pool != NULL) {
		fwmark = stream->mark_pool->get_mark();
	} else {
		fwmark = 0;  // No mark.
	}
	if (setsockopt(sock, SOL_SOCKET, SO_MARK, &fwmark, sizeof(fwmark)) == -1) {
		if (fwmark != 0) {
			log_perror("setsockopt(SO_MARK)");
		}
		fwmark = 0;
	}
	if (stream != NULL) {
		if (setsockopt(sock, SOL_SOCKET, SO_MAX_PACING_RATE, &stream->pacing_rate, sizeof(stream->pacing_rate)) == -1) {
			if (stream->pacing_rate != ~0U) {
				log_perror("setsockopt(SO_MAX_PACING_RATE)");
			}
		}
	}
}

ClientProto Client::serialize() const
{
	ClientProto serialized;
	serialized.set_sock(sock);
	serialized.set_remote_addr(remote_addr);
	serialized.set_connect_time(connect_time);
	serialized.set_state(state);
	serialized.set_request(request);
	serialized.set_url(url);
	serialized.set_header_or_error(header_or_error);
	serialized.set_header_or_error_bytes_sent(serialized.header_or_error_bytes_sent());
	serialized.set_stream_pos(stream_pos);
	serialized.set_bytes_sent(bytes_sent);
	serialized.set_bytes_lost(bytes_lost);
	serialized.set_num_loss_events(num_loss_events);
	return serialized;
}
	
ClientStats Client::get_stats() const
{
	ClientStats stats;
	if (url.empty()) {
		stats.url = "-";
	} else {
		stats.url = url;
	}
	stats.sock = sock;
	stats.fwmark = fwmark;
	stats.remote_addr = remote_addr;
	stats.connect_time = connect_time;
	stats.bytes_sent = bytes_sent;
	stats.bytes_lost = bytes_lost;
	stats.num_loss_events = num_loss_events;
	return stats;
}
