/**
 * Copyright (c) 2013, 2017 ControlsFX
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *     * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 *     * Neither the name of ControlsFX, any associated website, nor the
 * names of its contributors may be used to endorse or promote products
 * derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL CONTROLSFX BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package org.controlsfx.glyphfont;

import java.io.InputStream;
import java.util.Arrays;


/**
 * Defines a {@link GlyphFont} for the FontAwesome font set (see
 * <a href="http://fortawesome.github.io/Font-Awesome/">the FontAwesome website</a>
 * for more details). Note that at present the FontAwesome font is not distributed
 * with ControlsFX, and is, by default, instead loaded from a CDN at runtime.
 *
 * <p>To use FontAwesome (or indeed any glyph font) in your JavaFX application,
 * you firstly have to get access to the FontAwesome glyph font. You do this by
 * doing the following:
 *
 * <pre>GlyphFont fontAwesome = GlyphFontRegistry.font("FontAwesome");</pre>
 *
 * <p>This code works because all glyph fonts are found dynamically at runtime
 * by the {@link GlyphFontRegistry} class, so you can simply request the font
 * set you want from there.
 *
 * <p>Once the font set has been loaded, you can simply start creating
 * {@link Glyph} nodes and place them in your user interface. For example:
 *
 * <pre>new Button("", fontAwesome.create(&#92;uf013).fontColor(Color.RED));</pre>
 *
 * <p>Of course, this requires you to know that <code>&#92;uf013</code> maps to
 * a 'gear' icon, which is not always intuitive (especially when you re-read the
 * code in the future). A simpler approach is to do the following:
 *
 * <pre>new Button("", fontAwesome.create(FontAwesome.Glyph.GEAR));</pre>
 * or
 * <pre>new Button("", fontAwesome.create("GEAR"));</pre>
 *
 * It is possible to achieve the same result without creating a reference to icon font by simply using
 * {@link org.controlsfx.glyphfont.Glyph} constructor
 *
 * <pre>new Button("", new Glyph("FontAwesome","GEAR");</pre>
 *
 * You can use the above Glyph class also in FXML and set the
 * fontFamily and icon property there.
 *
 * @see GlyphFont
 * @see GlyphFontRegistry
 * @see Glyph
 */
public class FontAwesome extends GlyphFont {

    private static String fontName = "FontAwesome"; //$NON-NLS-1$


    /**
     * The individual glyphs offered by the FontAwesome font.
     */
    public static enum Glyph implements INamedCharacter {

        ADJUST('\uf042'),
        ADN('\uf170'),
        ALIGN_CENTER('\uf037'),
        ALIGN_JUSTIFY('\uf039'),
        ALIGN_LEFT('\uf036'),
        ALIGN_RIGHT('\uf038'),
        AMBULANCE('\uf0F9'),
        ANCHOR('\uf13D'),
        ANDROID('\uf17B'),
        ANGELLIST('\uf209'),
        ANGLE_DOUBLE_DOWN('\uf103'),
        ANGLE_DOUBLE_LEFT('\uf100'),
        ANGLE_DOUBLE_RIGHT('\uf101'),
        ANGLE_DOUBLE_UP('\uf102'),
        ANGLE_DOWN('\uf107'),
        ANGLE_LEFT('\uf104'),
        ANGLE_RIGHT('\uf105'),
        ANGLE_UP('\uf106'),
        APPLE('\uf179'),
        ARCHIVE('\uf187'),
        AREA_CHART('\uf1FE'),
        ARROW_CIRCLE_DOWN('\uf0AB'),
        ARROW_CIRCLE_LEFT('\uf0A8'),
        ARROW_CIRCLE_O_DOWN('\uf01A'),
        ARROW_CIRCLE_O_LEFT('\uf190'),
        ARROW_CIRCLE_O_RIGHT('\uf18E'),
        ARROW_CIRCLE_O_UP('\uf01B'),
        ARROW_CIRCLE_RIGHT('\uf0A9'),
        ARROW_CIRCLE_UP('\uf0AA'),
        ARROW_DOWN('\uf063'),
        ARROW_LEFT('\uf060'),
        ARROW_RIGHT('\uf061'),
        ARROW_UP('\uf062'),
        ARROWS('\uf047'),
        ARROWS_ALT('\uf0B2'),
        ARROWS_H('\uf07E'),
        ARROWS_V('\uf07D'),
        ASTERISK('\uf069'),
        AT('\uf1FA'),
        AUTOMOBILE('\uf1B9'),
        BACKWARD('\uf04A'),
        BAN('\uf05E'),
        BANK('\uf19C'),
        BAR_CHART('\uf080'),
        BAR_CHART_O('\uf080'),
        BARCODE('\uf02A'),
        BARS('\uf0C9'),
        BED('\uf236'),
        BEER('\uf0FC'),
        BEHANCE('\uf1B4'),
        BEHANCE_SQUARE('\uf1B5'),
        BELL('\uf0F3'),
        BELL_O('\uf0A2'),
        BELL_SLASH('\uf1F6'),
        BELL_SLASH_O('\uf1F7'),
        BICYCLE('\uf206'),
        BINOCULARS('\uf1E5'),
        BIRTHDAY_CAKE('\uf1FD'),
        BITBUCKET('\uf171'),
        BITBUCKET_SQUARE('\uf172'),
        BITCOIN('\uf15A'),
        BOLD('\uf032'),
        BOLT('\uf0E7'),
        BOMB('\uf1E2'),
        BOOK('\uf02D'),
        BOOKMARK('\uf02E'),
        BOOKMARK_ALT('\uf097'),
        BRIEFCASE('\uf0B1'),
        BTC('\uf15A'),
        BUG('\uf188'),
        BUILDING('\uf1AD'),
        BUILDING_ALT('\uf0F7'),
        BULLHORN('\uf0A1'),
        BULLSEYE('\uf140'),
        BUS('\uf207'),
        BUYSELLADS('\uf20D'),
        CAB('\uf1BA'),
        CALCULATOR('\uf1EC'),
        CALENDAR('\uf073'),
        CALENDAR_ALT('\uf133'),
        CAMERA('\uf030'),
        CAMERA_RETRO('\uf083'),
        CAR('\uf1B9'),
        CARET_DOWN('\uf0D7'),
        CARET_LEFT('\uf0D9'),
        CARET_RIGHT('\uf0DA'),
        CARET_SQUARE_ALT_DOWN('\uf150'),
        CARET_SQUARE_ALT_LEFT('\uf191'),
        CARET_SQUARE_ALT_RIGHT('\uf152'),
        CARET_SQUARE_ALT_UP('\uf151'),
        CARET_UP('\uf0D8'),
        CART_ARROW_DOWN('\uf218'),
        CART_PLUS('\uf217'),
        CC('\uf20A'),
        CC_AMEX('\uf1F3'),
        CC_DISCOVER('\uf1F2'),
        CC_MASTERCARD('\uf1F1'),
        CC_PAYPAL('\uf1F4'),
        CC_STRIPE('\uf1F5'),
        CC_VISA('\uf1F0'),
        CERTIFICATE('\uf0A3'),
        CHAIN('\uf0C1'),
        CHAIN_BROKEN('\uf127'),
        CHECK('\uf00C'),
        CHECK_CIRCLE('\uf058'),
        CHECK_CIRCLE_ALT('\uf05D'),
        CHECK_SQUARE('\uf14A'),
        CHECK_SQUARE_ALT('\uf046'),
        CHEVRON_CIRCLE_DOWN('\uf13A'),
        CHEVRON_CIRCLE_LEFT('\uf137'),
        CHEVRON_CIRCLE_RIGHT('\uf138'),
        CHEVRON_CIRCLE_UP('\uf139'),
        CHEVRON_DOWN('\uf078'),
        CHEVRON_LEFT('\uf053'),
        CHEVRON_RIGHT('\uf054'),
        CHEVRON_UP('\uf077'),
        CHILD('\uf1AE'),
        CIRCLE('\uf111'),
        CIRCLE_ALT('\uf10C'),
        CIRCLE_ALT_NOTCH('\uf1CE'),
        CIRCLE_THIN('\uf1DB'),
        CLIPBOARD('\uf0EA'),
        CLOCK_ALT('\uf017'),
        CLOSE('\uf00D'),
        CLOUD('\uf0C2'),
        CLOUD_DOWNLOAD('\uf0ED'),
        CLOUD_UPLOAD('\uf0EE'),
        CNY('\uf157'),
        CODE('\uf121'),
        CODE_FORK('\uf126'),
        CODEPEN('\uf1CB'),
        COFFEE('\uf0F4'),
        COG('\uf013'),
        COGS('\uf085'),
        COLUMNS('\uf0DB'),
        COMMENT('\uf075'),
        COMMENT_ALT('\uf0E5'),
        COMMENTS('\uf086'),
        COMMENTS_ALT('\uf0E6'),
        COMPASS('\uf14E'),
        COMPRESS('\uf066'),
        CONNECTDEVELOP('\uf20E'),
        COPY('\uf0C5'),
        COPYRIGHT('\uf1F9'),
        CREDIT_CARD('\uf09D'),
        CROP('\uf125'),
        CROSSHAIRS('\uf05B'),
        CSS3('\uf13C'),
        CUBE('\uf1B2'),
        CUBES('\uf1B3'),
        CUT('\uf0C4'),
        CUTLERY('\uf0F5'),
        DASHBOARD('\uf0E4'),
        DASHCUBE('\uf210'),
        DATABASE('\uf1C0'),
        DEDENT('\uf03B'),
        DELICIOUS('\uf1A5'),
        DESKTOP('\uf108'),
        DEVIANTART('\uf1BD'),
        DIAMOND('\uf219'),
        DIGG('\uf1A6'),
        DOLLAR('\uf155'),
        DOT_CIRCLE_ALT('\uf192'),
        DOWNLOAD('\uf019'),
        DRIBBBLE('\uf17D'),
        DROPBOX('\uf16B'),
        DRUPAL('\uf1A9'),
        EDIT('\uf044'),
        EJECT('\uf052'),
        ELLIPSIS_H('\uf141'),
        ELLIPSIS_V('\uf142'),
        EMPIRE('\uf1D1'),
        ENVELOPE('\uf0E0'),
        ENVELOPE_ALT('\uf003'),
        ENVELOPE_SQUARE('\uf199'),
        ERASER('\uf12D'),
        EUR('\uf153'),
        EURO('\uf153'),
        EXCHANGE('\uf0EC'),
        EXCLAMATION('\uf12A'),
        EXCLAMATION_CIRCLE('\uf06A'),
        EXCLAMATION_TRIANGLE('\uf071'),
        EXPAND('\uf065'),
        EXTERNAL_LINK('\uf08E'),
        EXTERNAL_LINK_SQUARE('\uf14C'),
        EYE('\uf06E'),
        EYE_SLASH('\uf070'),
        EYEDROPPER('\uf1FB'),
        FACEBOOK('\uf09A'),
        FACEBOOK_F('\uf09A'),
        FACEBOOK_ALTFFICIAL('\uf230'),
        FACEBOOK_SQUARE('\uf082'),
        FAST_BACKWARD('\uf049'),
        FAST_FORWARD('\uf050'),
        FAX('\uf1AC'),
        FEMALE('\uf182'),
        FIGHTER_JET('\uf0FB'),
        FILE('\uf15B'),
        FILE_ARCHIVE_ALT('\uf1C6'),
        FILE_AUDIO_ALT('\uf1C7'),
        FILE_CODE_ALT('\uf1C9'),
        FILE_EXCEL_ALT('\uf1C3'),
        FILE_IMAGE_ALT('\uf1C5'),
        FILE_MOVIE_ALT('\uf1C8'),
        FILE_ALT('\uf016'),
        FILE_PDF_ALT('\uf1C1'),
        FILE_PHOTO_ALT('\uf1C5'),
        FILE_PICTURE_ALT('\uf1C5'),
        FILE_POWERPOINT_ALT('\uf1C4'),
        FILE_SOUND_ALT('\uf1C7'),
        FILE_TEXT('\uf15C'),
        FILE_TEXT_ALT('\uf0F6'),
        FILE_VIDEO_ALT('\uf1C8'),
        FILE_WORD_ALT('\uf1C2'),
        FILE_ZIP_ALT('\uf1C6'),
        FILES_ALT('\uf0C5'),
        FILM('\uf008'),
        FILTER('\uf0B0'),
        FIRE('\uf06D'),
        FIRE_EXTINGUISHER('\uf134'),
        FLAG('\uf024'),
        FLAG_CHECKERED('\uf11E'),
        FLAG_ALT('\uf11D'),
        FLASH('\uf0E7'),
        FLASK('\uf0C3'),
        FLICKR('\uf16E'),
        FLOPPY_ALT('\uf0C7'),
        FOLDER('\uf07B'),
        FOLDER_ALT('\uf114'),
        FOLDER_OPEN('\uf07C'),
        FOLDER_OPEN_ALT('\uf115'),
        FONT('\uf031'),
        FORUMBEE('\uf211'),
        FORWARD('\uf04E'),
        FOURSQUARE('\uf180'),
        FROWN_ALT('\uf119'),
        FUTBOL_ALT('\uf1E3'),
        GAMEPAD('\uf11B'),
        GAVEL('\uf0E3'),
        GBP('\uf154'),
        GE('\uf1D1'),
        GEAR('\uf013'),
        GEARS('\uf085'),
        GENDERLESS('\uf1DB'),
        GIFT('\uf06B'),
        GIT('\uf1D3'),
        GIT_SQUARE('\uf1D2'),
        GITHUB('\uf09B'),
        GITHUB_ALT('\uf113'),
        GITHUB_SQUARE('\uf092'),
        GITTIP('\uf184'),
        GLASS('\uf000'),
        GLOBE('\uf0AC'),
        GOOGLE('\uf1A0'),
        GOOGLE_PLUS('\uf0D5'),
        GOOGLE_PLUS_SQUARE('\uf0D4'),
        GOOGLE_WALLET('\uf1EE'),
        GRADUATION_CAP('\uf19D'),
        GRATIPAY('\uf184'),
        GROUP('\uf0C0'),
        H_SQUARE('\uf0FD'),
        HACKER_NEWS('\uf1D4'),
        HAND_ALT_DOWN('\uf0A7'),
        HAND_ALT_LEFT('\uf0A5'),
        HAND_ALT_RIGHT('\uf0A4'),
        HAND_ALT_UP('\uf0A6'),
        HDD_ALT('\uf0A0'),
        HEADER('\uf1DC'),
        HEADPHONES('\uf025'),
        HEART('\uf004'),
        HEART_ALT('\uf08A'),
        HEARTBEAT('\uf21E'),
        HISTORY('\uf1DA'),
        HOME('\uf015'),
        HOSPITAL_ALT('\uf0F8'),
        HOTEL('\uf236'),
        HTML5('\uf13B'),
        ILS('\uf20B'),
        IMAGE('\uf03E'),
        INBOX('\uf01C'),
        INDENT('\uf03C'),
        INFO('\uf129'),
        INFO_CIRCLE('\uf05A'),
        INR('\uf156'),
        INSTAGRAM('\uf16D'),
        INSTITUTION('\uf19C'),
        IOXHOST('\uf208'),
        ITALIC('\uf033'),
        JOOMLA('\uf1AA'),
        JPY('\uf157'),
        JSFIDDLE('\uf1CC'),
        KEY('\uf084'),
        KEYBOARD_ALT('\uf11C'),
        KRW('\uf159'),
        LANGUAGE('\uf1AB'),
        LAPTOP('\uf109'),
        LASTFM('\uf202'),
        LASTFM_SQUARE('\uf203'),
        LEAF('\uf06C'),
        LEANPUB('\uf212'),
        LEGAL('\uf0E3'),
        LEMON_ALT('\uf094'),
        LEVEL_DOWN('\uf149'),
        LEVEL_UP('\uf148'),
        LIFE_BOUY('\uf1CD'),
        LIFE_BUOY('\uf1CD'),
        LIFE_RING('\uf1CD'),
        LIFE_SAVER('\uf1CD'),
        LIGHTBULB_ALT('\uf0EB'),
        LINE_CHART('\uf201'),
        LINK('\uf0C1'),
        LINKEDIN('\uf0E1'),
        LINKEDIN_SQUARE('\uf08C'),
        LINUX('\uf17C'),
        LIST('\uf03A'),
        LIST_ALT('\uf022'),
        LIST_OL('\uf0CB'),
        LIST_UL('\uf0CA'),
        LOCATION_ARROW('\uf124'),
        LOCK('\uf023'),
        LONG_ARROW_DOWN('\uf175'),
        LONG_ARROW_LEFT('\uf177'),
        LONG_ARROW_RIGHT('\uf178'),
        LONG_ARROW_UP('\uf176'),
        MAGIC('\uf0D0'),
        MAGNET('\uf076'),
        MAIL_FORWARD('\uf064'),
        MAIL_REPLY('\uf112'),
        MAIL_REPLY_ALL('\uf122'),
        MALE('\uf183'),
        MAP_MARKER('\uf041'),
        MARS('\uf222'),
        MARS_DOUBLE('\uf227'),
        MARS_STROKE('\uf229'),
        MARS_STROKE_H('\uf22B'),
        MARS_STROKE_V('\uf22A'),
        MAXCDN('\uf136'),
        MEANPATH('\uf20C'),
        MEDIUM('\uf23A'),
        MEDKIT('\uf0FA'),
        MEH_ALT('\uf11A'),
        MERCURY('\uf223'),
        MICROPHONE('\uf130'),
        MICROPHONE_SLASH('\uf131'),
        MINUS('\uf068'),
        MINUS_CIRCLE('\uf056'),
        MINUS_SQUARE('\uf146'),
        MINUS_SQUARE_ALT('\uf147'),
        MOBILE('\uf10B'),
        MOBILE_PHONE('\uf10B'),
        MONEY('\uf0D6'),
        MOON_ALT('\uf186'),
        MORTAR_BOARD('\uf19D'),
        MOTORCYCLE('\uf21C'),
        MUSIC('\uf001'),
        NAVICON('\uf0C9'),
        NEUTER('\uf22C'),
        NEWSPAPER_ALT('\uf1EA'),
        OPENID('\uf19B'),
        OUTDENT('\uf03B'),
        PAGELINES('\uf18C'),
        PAINT_BRUSH('\uf1FC'),
        PAPER_PLANE('\uf1D8'),
        PAPER_PLANE_ALT('\uf1D9'),
        PAPERCLIP('\uf0C6'),
        PARAGRAPH('\uf1DD'),
        PASTE('\uf0EA'),
        PAUSE('\uf04C'),
        PAW('\uf1B0'),
        PAYPAL('\uf1ED'),
        PENCIL('\uf040'),
        PENCIL_SQUARE('\uf14B'),
        PENCIL_SQUARE_ALT('\uf044'),
        PHONE('\uf095'),
        PHONE_SQUARE('\uf098'),
        PHOTO('\uf03E'),
        PICTURE_ALT('\uf03E'),
        PIE_CHART('\uf200'),
        PIED_PIPER('\uf1A7'),
        PIED_PIPER_ALT('\uf1A8'),
        PINTEREST('\uf0D2'),
        PINTEREST_P('\uf231'),
        PINTEREST_SQUARE('\uf0D3'),
        PLANE('\uf072'),
        PLAY('\uf04B'),
        PLAY_CIRCLE('\uf144'),
        PLAY_CIRCLE_ALT('\uf01D'),
        PLUG('\uf1E6'),
        PLUS('\uf067'),
        PLUS_CIRCLE('\uf055'),
        PLUS_SQUARE('\uf0FE'),
        PLUS_SQUARE_ALT('\uf196'),
        POWER_OFF('\uf011'),
        PRINT('\uf02F'),
        PUZZLE_PIECE('\uf12E'),
        QQ('\uf1D6'),
        QRCODE('\uf029'),
        QUESTION('\uf128'),
        QUESTION_CIRCLE('\uf059'),
        QUOTE_LEFT('\uf10D'),
        QUOTE_RIGHT('\uf10E'),
        RA('\uf1D0'),
        RANDOM('\uf074'),
        REBEL('\uf1D0'),
        RECYCLE('\uf1B8'),
        REDDIT('\uf1A1'),
        REDDIT_SQUARE('\uf1A2'),
        REFRESH('\uf021'),
        REMOVE('\uf00D'),
        RENREN('\uf18B'),
        REORDER('\uf0C9'),
        REPEAT('\uf01E'),
        REPLY('\uf112'),
        REPLY_ALL('\uf122'),
        RETWEET('\uf079'),
        RMB('\uf157'),
        ROAD('\uf018'),
        ROCKET('\uf135'),
        ROTATE_LEFT('\uf0E2'),
        ROTATE_RIGHT('\uf01E'),
        ROUBLE('\uf158'),
        RSS('\uf09E'),
        RSS_SQUARE('\uf143'),
        RUB('\uf158'),
        RUBLE('\uf158'),
        RUPEE('\uf156'),
        SAVE('\uf0C7'),
        SCISSORS('\uf0C4'),
        SEARCH('\uf002'),
        SEARCH_MINUS('\uf010'),
        SEARCH_PLUS('\uf00E'),
        SELLSY('\uf213'),
        SEND('\uf1D8'),
        SEND_ALT('\uf1D9'),
        SERVER('\uf233'),
        SHARE('\uf064'),
        SHARE_ALT('\uf1E0'),
        SHARE_ALT_SQUARE('\uf1E1'),
        SHARE_SQUARE('\uf14D'),
        SHARE_SQUARE_ALT('\uf045'),
        SHEKEL('\uf20B'),
        SHEQEL('\uf20B'),
        SHIELD('\uf132'),
        SHIP('\uf21A'),
        SHIRTSINBULK('\uf214'),
        SHOPPING_CART('\uf07A'),
        SIGN_IN('\uf090'),
        SIGN_OUT('\uf08B'),
        SIGNAL('\uf012'),
        SIMPLYBUILT('\uf215'),
        SITEMAP('\uf0E8'),
        SKYATLAS('\uf216'),
        SKYPE('\uf17E'),
        SLACK('\uf198'),
        SLIDERS('\uf1DE'),
        SLIDESHARE('\uf1E7'),
        SMILE_ALT('\uf118'),
        SOCCER_BALL_ALT('\uf1E3'),
        SORT('\uf0DC'),
        SORT_ALPHA_ASC('\uf15D'),
        SORT_ALPHA_DESC('\uf15E'),
        SORT_AMOUNT_ASC('\uf160'),
        SORT_AMOUNT_DESC('\uf161'),
        SORT_ASC('\uf0DE'),
        SORT_DESC('\uf0DD'),
        SORT_DOWN('\uf0DD'),
        SORT_NUMERIC_ASC('\uf162'),
        SORT_NUMERIC_DESC('\uf163'),
        SORT_UP('\uf0DE'),
        SOUNDCLOUD('\uf1BE'),
        SPACE_SHUTTLE('\uf197'),
        SPINNER('\uf110'),
        SPOON('\uf1B1'),
        SPOTIFY('\uf1BC'),
        SQUARE('\uf0C8'),
        SQUARE_ALT('\uf096'),
        STACK_EXCHANGE('\uf18D'),
        STACK_OVERFLOW('\uf16C'),
        STAR('\uf005'),
        STAR_HALF('\uf089'),
        STAR_HALF_EMPTY('\uf123'),
        STAR_HALF_FULL('\uf123'),
        STAR_HALF_ALT('\uf123'),
        STAR_ALT('\uf006'),
        STEAM('\uf1B6'),
        STEAM_SQUARE('\uf1B7'),
        STEP_BACKWARD('\uf048'),
        STEP_FORWARD('\uf051'),
        STETHOSCOPE('\uf0F1'),
        STOP('\uf04D'),
        STREET_VIEW('\uf21D'),
        STRIKETHROUGH('\uf0CC'),
        STUMBLEUPON('\uf1A4'),
        STUMBLEUPON_CIRCLE('\uf1A3'),
        SUBSCRIPT('\uf12C'),
        SUBWAY('\uf239'),
        SUITCASE('\uf0F2'),
        SUN_ALT('\uf185'),
        SUPERSCRIPT('\uf12B'),
        SUPPORT('\uf1CD'),
        TABLE('\uf0CE'),
        TABLET('\uf10A'),
        TACHOMETER('\uf0E4'),
        TAG('\uf02B'),
        TAGS('\uf02C'),
        TASKS('\uf0AE'),
        TAXI('\uf1BA'),
        TENCENT_WEIBO('\uf1D5'),
        TERMINAL('\uf120'),
        TEXT_HEIGHT('\uf034'),
        TEXT_WIDTH('\uf035'),
        TH('\uf00A'),
        TH_LARGE('\uf009'),
        TH_LIST('\uf00B'),
        THUMB_TACK('\uf08D'),
        THUMBS_DOWN('\uf165'),
        THUMBS_ALT_DOWN('\uf088'),
        THUMBS_ALT_UP('\uf087'),
        THUMBS_UP('\uf164'),
        TICKET('\uf145'),
        TIMES('\uf00D'),
        TIMES_CIRCLE('\uf057'),
        TIMES_CIRCLE_ALT('\uf05C'),
        TINT('\uf043'),
        TOGGLE_DOWN('\uf150'),
        TOGGLE_LEFT('\uf191'),
        TOGGLE_OFF('\uf204'),
        TOGGLE_ON('\uf205'),
        TOGGLE_RIGHT('\uf152'),
        TOGGLE_UP('\uf151'),
        TRAIN('\uf238'),
        TRANSGENDER('\uf224'),
        TRANSGENDER_ALT('\uf225'),
        TRASH('\uf1F8'),
        TRASH_ALT('\uf014'),
        TREE('\uf1BB'),
        TRELLO('\uf181'),
        TROPHY('\uf091'),
        TRUCK('\uf0D1'),
        TRY('\uf195'),
        TTY('\uf1E4'),
        TUMBLR('\uf173'),
        TUMBLR_SQUARE('\uf174'),
        TURKISH_LIRA('\uf195'),
        TWITCH('\uf1E8'),
        TWITTER('\uf099'),
        TWITTER_SQUARE('\uf081'),
        UMBRELLA('\uf0E9'),
        UNDERLINE('\uf0CD'),
        UNDO('\uf0E2'),
        UNIVERSITY('\uf19C'),
        UNLINK('\uf127'),
        UNLOCK('\uf09C'),
        UNLOCK_ALT('\uf13E'),
        UNSORTED('\uf0DC'),
        UPLOAD('\uf093'),
        USD('\uf155'),
        USER('\uf007'),
        USER_MD('\uf0F0'),
        USER_PLUS('\uf234'),
        USER_SECRET('\uf21B'),
        USER_TIMES('\uf235'),
        USERS('\uf0C0'),
        VENUS('\uf221'),
        VENUS_DOUBLE('\uf226'),
        VENUS_MARS('\uf228'),
        VIACOIN('\uf237'),
        VIDEO_CAMERA('\uf03D'),
        VIMEO_SQUARE('\uf194'),
        VINE('\uf1CA'),
        VK('\uf189'),
        VOLUME_DOWN('\uf027'),
        VOLUME_OFF('\uf026'),
        VOLUME_UP('\uf028'),
        WARNING('\uf071'),
        WECHAT('\uf1D7'),
        WEIBO('\uf18A'),
        WEIXIN('\uf1D7'),
        WHATSAPP('\uf232'),
        WHEELCHAIR('\uf193'),
        WIFI('\uf1EB'),
        WINDOWS('\uf17A'),
        WON('\uf159'),
        WORDPRESS('\uf19A'),
        WRENCH('\uf0AD'),
        XING('\uf168'),
        XING_SQUARE('\uf169'),
        YAHOO('\uf19E'),
        YELP('\uf1E9'),
        YEN('\uf157'),
        YOUTUBE('\uf167'),
        YOUTUBE_PLAY('\uf16A'),
        YOUTUBE_SQUARE('\uf166');

        private final char ch;

        /**
         * Creates a named Glyph mapped to the given character
         * @param ch
         */
        Glyph( char ch ) {
            this.ch = ch;
        }

        @Override
        public char getChar() {
            return ch;
        }
    };

    /**
     * Do not call this constructor directly - instead access the
     * {@link FontAwesome.Glyph} public static enumeration method to create the glyph nodes), or
     * use the {@link GlyphFontRegistry} class to get access.
     *
     * Note: Do not remove this public constructor since it is used by the service loader!
     */
    public FontAwesome() {
        this(FontAwesome.class.getResourceAsStream("fontawesome-webfont.ttf")); //$NON-NLS-1$
    }

    /**
     * Creates a new FontAwesome instance which uses the provided font source.
     * @param url
     */
    public FontAwesome(String url){
        super(fontName, 14, url, true);
        registerAll(Arrays.asList(Glyph.values()));
    }

    /**
     * Creates a new FontAwesome instance which uses the provided font source.
     * @param is
     */
    public FontAwesome(InputStream is){
        super(fontName, 14, is, true);
        registerAll(Arrays.asList(Glyph.values()));
    }

}
