//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      Sample/Particle/Mesocrystal.cpp
//! @brief     Implements class Mesocrystal.
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "Sample/Particle/Mesocrystal.h"
#include "Base/Types/Span.h"
#include "Base/Util/Assert.h"
#include "Sample/Particle/Crystal.h"
#include "Sample/Particle/IFormFactor.h"
#include "Sample/Scattering/Rotations.h"

Mesocrystal::Mesocrystal(Crystal* crystal, IFormFactor* formfactor)
    : m_crystal(crystal)
    , m_meso_formfactor(formfactor)
{
}

Mesocrystal::Mesocrystal(const Crystal& crystal, const IFormFactor& formfactor)
    : Mesocrystal(crystal.clone(), formfactor.clone())
{
}

Mesocrystal::~Mesocrystal() = default;

Mesocrystal* Mesocrystal::clone() const
{
    auto* result = new Mesocrystal(m_crystal->clone(), m_meso_formfactor->clone());
    result->setAbundance(m_abundance);
    if (rotation())
        result->rotate(*rotation());
    result->translate(particlePosition());
    return result;
}

std::vector<const INode*> Mesocrystal::nodeChildren() const
{
    return std::vector<const INode*>()
           << IParticle::nodeChildren() << m_crystal << m_meso_formfactor;
}

const Crystal& Mesocrystal::particleStructure() const
{
    ASSERT(m_crystal);
    return *m_crystal;
}

Span Mesocrystal::zSpan() const
{
    return m_meso_formfactor->spanZ(rotation()) + particlePosition().z();
}
