//
// C++ Interface: trackmodel
//
// Description:
//
//
// Author: Thibaut GRIDEL <tgridel@free.fr>
//
// Copyright (c) 2008-2019 Thibaut GRIDEL
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
//
#ifndef TRACKMODEL_H
#define TRACKMODEL_H

#include <QtGui>
#ifdef QML
#include <QQmlListProperty>
#endif
#include "boats.h"

class SituationModel;
class BoatModel;

/**
    \class TrackModel

    \brief The Model for the Track of a Boat

    The class represents the Model for a Track, according to an
    Observer Pattern.

    TrackModel contains data which describe one boat, like the series of
    the boat, the color it is drawn and the List of Positions where the boat
    navigates.

    It shall not be mistaken with BoatModel, which holds one position at
    a given time. More exactly, a TrackModel will hold a List of BoatModels.

    \sa SituationModel, BoatModel

*/

class TrackModel : public QObject {
        Q_OBJECT
    public:

        Q_PROPERTY(int order READ order WRITE setOrder NOTIFY orderChanged)
        Q_PROPERTY(QColor trackColor READ color WRITE setColor NOTIFY colorChanged)
        Q_PROPERTY(int series READ series WRITE setSeries NOTIFY seriesChanged)
        Q_PROPERTY(bool showPath READ showPath WRITE setShowPath NOTIFY showPathChanged)
        Q_PROPERTY(bool followTrack READ followTrack WRITE setFollowTrack NOTIFY followTrackChanged)
#ifdef QML
        Q_PROPERTY(QQmlListProperty<BoatModel> boatList READ boatList NOTIFY boatsChanged)
#endif
        Q_PROPERTY(int size READ size NOTIFY boatsChanged)

        TrackModel(SituationModel* situation = 0, QObject *parent = 0);
        ~TrackModel();

        BoatModel * addBoat(BoatModel *boat, int order = -1);
        int deleteBoat(BoatModel *boat);

        void displayBoats();
        void hideBoats();

        // Setters and Getters for Model Data
        int order() const { return m_order; }
        void setOrder(const int theValue);

        QColor color() const { return m_color;}
        void setColor(const QColor& theValue);

        int series() const { return m_series;}
        void setSeries(const int theValue);

        bool showPath() const { return m_showPath;}
        void setShowPath(const bool theValue);

        bool followTrack() const { return m_followTrack; }
        void setFollowTrack(bool theValue);

        int size() const { return m_boats.size();}
        const QList<BoatModel*> boats() const { return m_boats; }
#ifdef QML
        QQmlListProperty<BoatModel> boatList();
#endif
        const QPainterPath path() const { return m_path; }

        // Setters and Getters for Non model Data
        SituationModel* situation() const { return m_situation; }

        int length() const { return m_length; }
        bool hasSpin() const {return m_hasSpin; }
        qreal maxNormalSailAngle() const {return m_maxNormalSailAngle; }
        qreal maxNormalJibAngle() const {return m_maxNormalJibAngle; }
        qreal maxWithSpinSailAngle() const {return m_maxWithSpinSailAngle; }
        qreal maxWithSpinJibAngle() const {return m_maxWithSpinJibAngle; }

        QStringList discardedXml() const { return m_discardedXml; }
        void appendDiscardedXml(const QString& theValue);

        Q_INVOKABLE qreal headingForNext(int index, QPointF point);

        void changingTrack(TrackModel *track);

        Q_INVOKABLE void setSelected(bool selected);

    signals:
        // Signals for TrackModel parameters
        void orderChanged(int order);
        void colorChanged(QColor color);
        void seriesChanged(int series);
        void showPathChanged(bool showPath);
        void followTrackChanged(bool followTrack);
        void trackChanged(TrackModel *track);
        void boatsChanged();
        void trackSelected(bool selected);

    private:
        // Model Data
        /// \a m_order holds the stacking order of the Track
        int m_order;

        /// \a m_color holds the color of the Track
        QColor m_color;

        /// \a m_series holds the series of the Track
        int m_series;

        /// \a m_boats holds the List of Boat Positions of the Track
        QList<BoatModel*> m_boats;

        /// \a m_showPath holds whether the track path will be displayed
        bool m_showPath;

        /// \a m_followTrack holds whether this track will be followed
        /// during the animation
        bool m_followTrack;

        // Non model Data
        /// \a m_situation keeps a pointer to the SituationModel to which
        /// it belongs
        SituationModel *m_situation;

        /// \a m_length holds the size of the Boat in World Coordinates
        int m_length;

        /// \a m_hasSpin holds whether this type of boat has a spinnaker (or gennaker)
        bool m_hasSpin;

        /// \a m_maxNormalSailAngle holds the max sail angle to use when sailing without a spinnaker
        qreal m_maxNormalSailAngle;

        /// \a m_maxNormalJibAngle holds the max jib angle to use when sailing without a spinnaker
        qreal m_maxNormalJibAngle;

        /// \a m_maxWithSpinSailAngle holds the max sail angle to use when sailing with a spinnaker
        qreal m_maxWithSpinSailAngle;

        /// \a m_maxWithSpinJibAngle holds the max jib angle to use when sailing with a spinnaker
        qreal m_maxWithSpinJibAngle;

        /// \a m_path holds the QPainterPath of the Track
        QPainterPath m_path;

        /// \a m_discardedXml keeps all unparsed xml tags
        QStringList m_discardedXml;
};

#endif
