/*###############################################################################
# Linux Management Providers (LMP), CPU provider package
# Copyright (C) 2007 Frederic Desmons, ETRI <desmons@etri.re.kr ,desmons_frederic@yahoo.fr>
# 
# This program is being developed under the "OpenDRIM" project.
# The "OpenDRIM" project web page: http://opendrim.sourceforge.net
# The "OpenDRIM" project mailing list: opendrim@googlegroups.com
# 
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; version 2
# of the License.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
#################################################################################

#################################################################################
# To contributors, please leave your contact information in this section
# AND comment your changes in the source code.
# 
# Modified by 2009 Rakhimov Rustam, TUIT <rusyasoft@gmail.com>
# Modified by 2009 Guillaume BOTTEX, ETRI <guillaumebottex@etri.re.kr>
###############################################################################*/

#include "OpenDRIM_CacheMemoryAccess.h"

const string systemCreationClassName="OpenDRIM_ComputerSystem";
const string creationClassName="OpenDRIM_CacheMemory";
string systemName;
vector<_processor_information> SMBIOS_processors_information;
vector<_processor_topology> cpu_info_processors_information;
vector<string> DeviceIDs;

int CPU_OpenDRIM_CacheMemory_load(const CMPIBroker* broker, string& errorMessage) {
	_E_;
	CF_assert(CF_getSystemName(systemName, errorMessage));
	
	// get info from SMBIOS
	CF_assert(SMBIOS_getProcessorsInformation(SMBIOS_processors_information, errorMessage));
	
	// get info from cpuinfo
	CF_assert(CPU_getProcessors("/proc/cpuinfo", cpu_info_processors_information, errorMessage));
	
	// different processors count...
	if (cpu_info_processors_information.size() != SMBIOS_processors_information.size())
	{
		errorMessage = "Different processor count";
		return ERROR;
	}
	
	// We create l1 and l2 cache instances
	for (unsigned int i = 0; i < cpu_info_processors_information.size(); i++)
	{
		cout << "Associated processor memory: " << cpu_info_processors_information[i].associated_cache_memory << endl;
		
		for (unsigned int j = 0; j < cpu_info_processors_information[i].processor_cores.size(); j++)
		{
			cout << "Associated core memory: " << cpu_info_processors_information[i].processor_cores[j].associated_cache_memory << endl;
			
			// L1 cache
			if (SMBIOS_processors_information[i].has_l1_cache)
				DeviceIDs.push_back(CF_intToStr(i) + ":" + CF_intToStr(j) +":L1");
			
			// l2 cache
			if (SMBIOS_processors_information[i].has_l2_cache && cpu_info_processors_information[i].processor_cores[j].associated_cache_memory)
				DeviceIDs.push_back(CF_intToStr(i) + ":" + CF_intToStr(j) + ":L2");
		}
		
		// Cache associated with processors
		// l2 cache
		if (SMBIOS_processors_information[i].has_l2_cache && cpu_info_processors_information[i].associated_cache_memory)
			DeviceIDs.push_back(CF_intToStr(i) + ":L2");
		
		// l3 cache
		if (SMBIOS_processors_information[i].has_l3_cache)
			DeviceIDs.push_back(CF_intToStr(i) + ":L3");
	}
	_L_;
	return OK;
}

int CPU_OpenDRIM_CacheMemory_unload(string& errorMessage) {
	_E_;
	// TODO
	_L_;
	return OK;
}

int CPU_OpenDRIM_CacheMemory_retrieve(const CMPIBroker* broker, const CMPIContext* ctx, vector<OpenDRIM_CacheMemory>& result, const char** properties, string& errorMessage, const string& discriminant) {
	_E_;
	for(unsigned int i=0;i<DeviceIDs.size();i++)
	{
		OpenDRIM_CacheMemory instance;
		
		instance.setSystemCreationClassName(systemCreationClassName);
		instance.setCreationClassName(creationClassName);
		instance.setSystemName(systemName);
		instance.setDeviceID(DeviceIDs[i]);
		
		if(discriminant=="ei")
		{
			int processor_num, level;
			
			processor_num = atoi(DeviceIDs[i].substr(0,DeviceIDs[i].find_first_of(":")).c_str());
			level = atoi(DeviceIDs[i].substr(DeviceIDs[i].find_last_of(":")+2).c_str());
			
			CF_assert(CPU_OpenDRIM_CacheMemory_populate(instance, cpu_info_processors_information[processor_num], SMBIOS_processors_information[processor_num], level, errorMessage));
		}
		
		result.push_back(instance);
	}
	_L_;
	return OK;
}

int CPU_OpenDRIM_CacheMemory_getInstance(const CMPIBroker* broker, const CMPIContext* ctx, OpenDRIM_CacheMemory& instance, const char** properties, string& errorMessage) {
	_E_;
	vector<string>::size_type index;
	
	if (instance.SystemCreationClassName != systemCreationClassName ||
			instance.CreationClassName != creationClassName ||
			instance.SystemName != systemName ||
			!CF_foundInList(instance.DeviceID, DeviceIDs, index))
	{
		errorMessage = "No instance";
		return NOT_FOUND;
	}
	
	int processor_num, level;
	
	processor_num = atoi(instance.DeviceID.substr(0,instance.DeviceID.find_first_of(":")).c_str());
	level = atoi(instance.DeviceID.substr(instance.DeviceID.find_last_of(":")+2).c_str());
	
	CF_assert(CPU_OpenDRIM_CacheMemory_populate(instance, cpu_info_processors_information[processor_num], SMBIOS_processors_information[processor_num], level, errorMessage));
	_L_;
	return OK;
}

int CPU_OpenDRIM_CacheMemory_setInstance(const CMPIBroker* broker, const CMPIContext* ctx, const OpenDRIM_CacheMemory& newInstance, const OpenDRIM_CacheMemory& oldInstance, const char** properties, string& errorMessage) {
	_E_;
	// TODO
	_L_;
	return NOT_SUPPORTED;
}

int CPU_OpenDRIM_CacheMemory_createInstance(const CMPIBroker* broker, const CMPIContext* ctx, const OpenDRIM_CacheMemory& instance, string& errorMessage) {
	_E_;
	// TODO
	_L_;
	return NOT_SUPPORTED;
}

int CPU_OpenDRIM_CacheMemory_deleteInstance(const CMPIBroker* broker, const CMPIContext* ctx, const OpenDRIM_CacheMemory& instance, string& errorMessage) {
	_E_;
	// TODO
	_L_;
	return NOT_SUPPORTED;
}

int CPU_OpenDRIM_CacheMemory_RequestStateChange(const CMPIBroker* broker, const CMPIContext* ctx, const OpenDRIM_CacheMemory& instance, unsigned int& returnValue, const OpenDRIM_CacheMemory_RequestStateChange_In& in, OpenDRIM_CacheMemory_RequestStateChange_Out& out, string& errorMessage) {
	_E_;
	// TODO
	_L_;
	return NOT_SUPPORTED;
}

int CPU_OpenDRIM_CacheMemory_SetPowerState(const CMPIBroker* broker, const CMPIContext* ctx, const OpenDRIM_CacheMemory& instance, unsigned int& returnValue, const OpenDRIM_CacheMemory_SetPowerState_In& in, string& errorMessage) {
	_E_;
	// TODO
	_L_;
	return NOT_SUPPORTED;
}

int CPU_OpenDRIM_CacheMemory_Reset(const CMPIBroker* broker, const CMPIContext* ctx, const OpenDRIM_CacheMemory& instance, unsigned int& returnValue, string& errorMessage) {
	_E_;
	// TODO
	_L_;
	return NOT_SUPPORTED;
}

int CPU_OpenDRIM_CacheMemory_EnableDevice(const CMPIBroker* broker, const CMPIContext* ctx, const OpenDRIM_CacheMemory& instance, unsigned int& returnValue, const OpenDRIM_CacheMemory_EnableDevice_In& in, string& errorMessage) {
	_E_;
	// TODO
	_L_;
	return NOT_SUPPORTED;
}

int CPU_OpenDRIM_CacheMemory_OnlineDevice(const CMPIBroker* broker, const CMPIContext* ctx, const OpenDRIM_CacheMemory& instance, unsigned int& returnValue, const OpenDRIM_CacheMemory_OnlineDevice_In& in, string& errorMessage) {
	_E_;
	// TODO
	_L_;
	return NOT_SUPPORTED;
}

int CPU_OpenDRIM_CacheMemory_QuiesceDevice(const CMPIBroker* broker, const CMPIContext* ctx, const OpenDRIM_CacheMemory& instance, unsigned int& returnValue, const OpenDRIM_CacheMemory_QuiesceDevice_In& in, string& errorMessage) {
	_E_;
	// TODO
	_L_;
	return NOT_SUPPORTED;
}

int CPU_OpenDRIM_CacheMemory_SaveProperties(const CMPIBroker* broker, const CMPIContext* ctx, const OpenDRIM_CacheMemory& instance, unsigned int& returnValue, string& errorMessage) {
	_E_;
	// TODO
	_L_;
	return NOT_SUPPORTED;
}

int CPU_OpenDRIM_CacheMemory_RestoreProperties(const CMPIBroker* broker, const CMPIContext* ctx, const OpenDRIM_CacheMemory& instance, unsigned int& returnValue, string& errorMessage) {
	_E_;
	// TODO
	_L_;
	return NOT_SUPPORTED;
}

int CPU_OpenDRIM_CacheMemory_populate(OpenDRIM_CacheMemory& instance, const _processor_topology& cpu_info_processor_information, const _processor_information& SMBIOS_processor_information, int level, string& errorMessage) {
	_E_;
	
	/*
	 * Properties to fill from profile
	 * + Mandatory:
	 * [X] SystemCreationClassName
	 * [X] CreationClassName
	 * [X] SystemName
	 * [X] DeviceID
	 * [X] BlockSize
	 * [X] NumberOfBlocks
	 * [X] EnabledState
	 * [X] RequestedState
	 * [X] HealthState
	 * [X] OperationalStatus
	 * [X] ElementName
	 */
	
	vector<unsigned short> OperationalStatus;
	
	// default
	// we don't support RequestedStateChange()
	instance.setRequestedState(12); // Not Applicable
	instance.setEnabledState(5);    // Not Applicable
	OperationalStatus.push_back(0); // unknown
	instance.setOperationalStatus(OperationalStatus);
	instance.setHealthState(0);     // unknown
	
	if (level == 1)
	{
		instance.setBlockSize(SMBIOS_processor_information.l1_cache.granularity);
		instance.setNumberOfBlocks((SMBIOS_processor_information.l1_cache.installed_size)/(cpu_info_processor_information.processor_cores.size()));
	}
	else if (level == 2)
	{
		if(!cpu_info_processor_information.associated_cache_memory)
		{
			instance.setBlockSize(SMBIOS_processor_information.l2_cache.granularity);
			instance.setNumberOfBlocks((SMBIOS_processor_information.l2_cache.installed_size)/(cpu_info_processor_information.processor_cores.size()));
		}
		else
		{
			instance.setBlockSize(SMBIOS_processor_information.l2_cache.granularity);
			instance.setNumberOfBlocks(SMBIOS_processor_information.l2_cache.installed_size);
		}
	}
	else if (level == 3)
	{
		instance.setBlockSize(SMBIOS_processor_information.l3_cache.granularity);
		instance.setNumberOfBlocks(SMBIOS_processor_information.l3_cache.installed_size);
	}
	
	// set info from /proc/cpuinfo
	if(level == 1 || (level ==2 && !cpu_info_processor_information.associated_cache_memory))
		instance.setElementName(cpu_info_processor_information.name + " core level " + CF_intToStr(level) + " cache memory");
	else
		instance.setElementName(cpu_info_processor_information.name + " level " + CF_intToStr(level) + " cache memory");
	_L_;
	return OK;
}

