/* whoopsie-preferences
 * 
 * Copyright © 2011-2013 Canonical Ltd.
 * Author: Evan Dandrea <evan.dandrea@canonical.com>
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#define _XOPEN_SOURCE 500

#include <errno.h>
#include <gio/gio.h>
#include <glib/gstdio.h>
#include <polkit/polkit.h>
#include <stdio.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>

#include <libwhoopsie/identifier.h>

#include "libwhoopsie-preferences.h"

#define CONFIG_PATH "/etc/whoopsie"

static GMainLoop* loop = NULL;
static guint loop_shutdown = 0;
static GKeyFile* key_file = NULL;
static PolkitAuthority* authority = NULL;

/* Hack for Ubuntu phone: check if path is an existing symlink to
 * /etc/writable; if it is, update that instead */
static const char* writable_filename (const char *config) {
        const char *result = config;
        struct stat sb;
        char *linkname = NULL;
        char *dirname = NULL;
        char *resolved_path = NULL;
        ssize_t r;
        static char realfile_buf[PATH_MAX];

        g_assert (config);
        g_assert (g_path_is_absolute (config));

        if (lstat (config, &sb) == -1) {
            goto out;
        }

        linkname = g_malloc (sb.st_size + 1);
        if (linkname == NULL) {
            goto out;
        }

        r = readlink (config, linkname, sb.st_size + 1);

        if (r < 0 || r > sb.st_size) {
            goto out;
        }

        linkname[sb.st_size] = '\0';

        /* Make absolute if relative */
        if (!g_path_is_absolute (linkname)) {
            dirname = g_path_get_dirname (config);
            resolved_path = g_build_filename (
                    dirname,
                    linkname,
                    NULL);
            g_free (linkname);
            linkname = resolved_path;
        }

        if (g_str_has_prefix (linkname, "/etc/writable")) {
            g_snprintf (realfile_buf, sizeof (realfile_buf), "%s", linkname);
            result = realfile_buf;
        }

out:
        g_debug ("Writable path for %s is %s", config, result);

        g_free (linkname);
        g_free (dirname);
        return result;
}

gboolean
whoopsie_preferences_load_configuration (void)
{
    char* data;
    gsize data_size;
    gboolean ret = TRUE;
    GError* error = NULL;

    if (g_key_file_load_from_file (key_file,
                                   writable_filename(CONFIG_PATH),
                                   G_KEY_FILE_KEEP_COMMENTS, NULL)) {
        return TRUE;
    }

    g_key_file_set_boolean (key_file, "General", "report_metrics", TRUE);
    data = g_key_file_to_data (key_file, &data_size, &error);
    if (error) {
        g_print ("Could not process configuration: %s\n", error->message);
        ret = FALSE;
        goto out;
    }
    if (!g_file_set_contents (writable_filename(CONFIG_PATH), data, data_size, &error)) {
        g_print ("Could not write configuration: %s\n", error->message);
        ret = FALSE;
        goto out;
    }

out:
    if (data)
        g_free (data);
    if (error)
        g_error_free (error);
    return ret;
}

static gboolean
is_systemd() {
    return g_file_test ("/run/systemd/system", G_FILE_TEST_IS_DIR);
}

static gboolean
whoopsie_preferences_whoopsie_enabled (GError **error) {
    char* systemd_check_command[] = { "/bin/systemctl", "-q", "is-enabled", "whoopsie.service", NULL };
    gchar* upstart_override_content = NULL;
    gboolean report_crashes = FALSE;
    int exit_code = 0;

    /* systemd running */
    if (is_systemd()) {
        if (g_spawn_sync (NULL, systemd_check_command, NULL, 0, NULL, NULL, NULL, NULL, &exit_code, error))
            report_crashes = (exit_code == 0);
    }
    /* upstart running */
    else {
        report_crashes = TRUE;
        if (!g_file_get_contents("/etc/init/whoopsie.override", &upstart_override_content, NULL, error)) {
            /* Only warn if the file exist but we couldn't access */
            if (g_error_matches (*error, G_FILE_ERROR, G_FILE_ERROR_NOENT))
                g_clear_error (error);
        }
        else {
           if (g_strrstr(upstart_override_content, "manual") != NULL)
               report_crashes = FALSE;
        }
        g_free(upstart_override_content);
    }

    return report_crashes;
}

static void
whoopsie_preferences_dbus_notify (WhoopsiePreferences* interface)
{
    GError* error = NULL;
    gboolean report_crashes = FALSE;
    gboolean report_metrics = FALSE;
    gboolean automatically_report_crashes = FALSE;

    report_crashes = whoopsie_preferences_whoopsie_enabled (&error);
    if (error) {
        g_warning ("Could not get if whoopsie is enabled: %s", error->message);
        g_error_free (error);
        error = NULL;
    }

    report_metrics = g_key_file_get_boolean (key_file, "General",
                                             "report_metrics", &error);
    if (error) {
        g_warning ("Could not get metrics: %s", error->message);
        g_error_free (error);
    }
    automatically_report_crashes = g_file_test ("/var/lib/apport/autoreport",
                                                G_FILE_TEST_EXISTS);
    whoopsie_preferences_set_report_crashes (interface, report_crashes);
    whoopsie_preferences_set_automatically_report_crashes (
            interface,
            automatically_report_crashes);
    whoopsie_preferences_set_report_metrics (interface, report_metrics);
}

static void
whoopsie_preferences_file_changed (GFileMonitor *monitor, GFile *file,
                                   GFile *other_file,
                                   GFileMonitorEvent event_type,
                                   gpointer user_data)
{
    if (event_type == G_FILE_MONITOR_EVENT_CHANGED) {
        whoopsie_preferences_load_configuration ();
        whoopsie_preferences_dbus_notify (user_data);
    }
}

gboolean
autoreport_file_changed (GFileMonitor* monitor, GFile* file, GFile* other_file, GFileMonitorEvent event_type, gpointer user_data)
{
    WhoopsiePreferences* interface = user_data;
    if (g_file_test ("/var/lib/apport/autoreport", G_FILE_TEST_EXISTS)) {
        whoopsie_preferences_set_automatically_report_crashes (interface, TRUE);
    } else {
        whoopsie_preferences_set_automatically_report_crashes (interface, FALSE);
    }
    return TRUE;
}

void
monitor_autoreport_file (WhoopsiePreferences* iface)
{
    GFile* path = NULL;
    GError* err = NULL;
    GFileMonitor* monitor = NULL;

    path = g_file_new_for_path ("/var/lib/apport/autoreport");
    monitor = g_file_monitor_file (path, G_FILE_MONITOR_NONE, NULL, &err);
    g_object_unref (path);
    if (err) {
        g_print ("Unable to monitor /var/lib/apport/autoreport: %s", err->message);
        g_error_free (err);
    } else {
        g_signal_connect (monitor, "changed", G_CALLBACK (autoreport_file_changed), iface);
    }

}
void
whoopsie_preferences_load (WhoopsiePreferences* interface)
{
    GError* error = NULL;
    GFile* fp = NULL;
    GFileMonitor* file_monitor = NULL;

    fp = g_file_new_for_path (CONFIG_PATH);
    file_monitor = g_file_monitor_file (fp, G_FILE_MONITOR_NONE,
                                        NULL, &error);
    g_signal_connect (file_monitor, "changed",
                      G_CALLBACK (whoopsie_preferences_file_changed),
                      interface);
    if (error) {
        g_print ("Could not set up file monitor: %s\n", error->message);
        g_error_free (error);
    }
    g_object_unref (fp);
    
    key_file = g_key_file_new ();
    whoopsie_preferences_load_configuration ();
}

void
whoopsie_preferences_notify_init (gboolean run_whoopsie)
{
    GError* error = NULL;
    char* command[] = { "/usr/sbin/invoke-rc.d", "whoopsie", run_whoopsie ? "start" : "stop", NULL };
    if (!g_spawn_sync (NULL, command, NULL, 0, NULL, NULL, NULL, NULL, NULL, &error))
        g_print ("Could not run invoke-rc.d: %s\n", error->message);
}

void
increase_shutdown_timeout (void)
{
    if (loop_shutdown)
        g_source_remove (loop_shutdown);
    loop_shutdown = g_timeout_add_seconds (60, (GSourceFunc) g_main_loop_quit, loop);
}

gboolean
whoopsie_automatic_reporting_changed (WhoopsiePreferences* object,
                                      GParamSpec* pspec, gpointer user_data)
{
    WhoopsiePreferencesIface* iface = WHOOPSIE_PREFERENCES_GET_IFACE (object);

    increase_shutdown_timeout ();

    /* I'm aware this is usually a race, but it's really just to guard against
     * writing the file when it already exists. */
    if (iface->get_automatically_report_crashes (object)) {
        if ((g_mkdir_with_parents("/var/lib/apport/", 0755) == 0) &&
                !g_file_test ("/var/lib/apport/autoreport",
                              G_FILE_TEST_EXISTS)) {
            fclose (g_fopen ("/var/lib/apport/autoreport", "w"));
        }
    } else {
        if (g_file_test ("/var/lib/apport/autoreport", G_FILE_TEST_EXISTS)) {
            g_unlink ("/var/lib/apport/autoreport");
        }
    }

    return TRUE;
}

gboolean
whoopsie_preferences_changed (WhoopsiePreferences* object, GParamSpec* pspec,
                              gpointer user_data)
{
    WhoopsiePreferencesIface* iface;
    gboolean saved_value, new_value;
    GError* error = NULL;
    char* data;
    char* errstr;
    gsize data_size;
    gboolean crashes = !g_strcmp0(user_data, "report_crashes");
    char* crash_enable_command[] = { "/usr/sbin/update-rc.d", "whoopsie", NULL, NULL };
    char* initctl_reload_command[] = { "/sbin/initctl", "reload-configuration", NULL };
    gboolean spawn_ok;
    gint exit_status;
    FILE* f = NULL;

    increase_shutdown_timeout ();

    iface = WHOOPSIE_PREFERENCES_GET_IFACE (object);
    if (crashes) {
        saved_value = whoopsie_preferences_whoopsie_enabled (&error);
        new_value = iface->get_report_crashes (object);
        crash_enable_command[2] = new_value ? "enable" : "disable";
    } else {
        new_value = iface->get_report_metrics (object);
        saved_value = g_key_file_get_boolean (key_file, "General", user_data,
                                              &error);
    }
    if (error) {
        g_print ("Could not process configuration: %s\n", error->message);
    }

    if (error || (saved_value != new_value)) {
        if (error)
            g_error_free (error);
        error = NULL;

        if (crashes) {
            spawn_ok = g_spawn_sync (NULL, crash_enable_command, NULL, 0, NULL,
                            NULL, NULL, NULL, &exit_status, &error);
            if (!spawn_ok || exit_status != 0) {
                if (error) {
                    errstr = g_strdup (error->message);
                    g_error_free (error);
                } else
                    errstr = g_strdup_printf ("exited with code %d", exit_status);
                g_print ("Could not run update-rc.d: %s\n", errstr);
                g_free (errstr);

                /* It's a real error for systemd */
                if (is_systemd())
                    return FALSE;

                /* Works for enabling (truncating) on upstart only atm */
                if (new_value) {
                    g_debug ("Truncating the override file manually");
                    f = fopen ("/etc/init/whoopsie.override", "w");
                    if (f) {
                        fclose (f);
			if (!g_spawn_sync (NULL, initctl_reload_command, NULL, 0, NULL,
					NULL, NULL, NULL, &exit_status, &error))
				g_print ("Could not run initctl reload-configuration: %s\n", errstr);
		    } else {
                        g_warning ("Couldn't open override file: %s", strerror(errno));
                        return FALSE;
                    }
                }
            }
        } else {
            g_key_file_set_boolean (key_file, "General", user_data, new_value);
            data = g_key_file_to_data (key_file, &data_size, &error);
            if (error) {
                g_print ("Could not process configuration: %s\n", error->message);
                return FALSE;
            }
            if (!g_file_set_contents (writable_filename(CONFIG_PATH), data, data_size, &error)) {
                g_print ("Could not write configuration: %s\n", error->message);
                return FALSE;
            }
        }
    }
    if (crashes)
        whoopsie_preferences_notify_init (new_value);

    return TRUE;
}

static gboolean
whoopsie_preferences_on_set_report_crashes (WhoopsiePreferences* object,
                                            GDBusMethodInvocation* invocation,
                                            gboolean report,
                                            gpointer user_data)
{
    whoopsie_preferences_set_report_crashes (object, report);
    whoopsie_preferences_complete_set_report_crashes (object, invocation);
    return TRUE;
}

static gboolean
whoopsie_preferences_on_set_report_metrics (WhoopsiePreferences* object,
                                            GDBusMethodInvocation* invocation,
                                            gboolean report,
                                            gpointer user_data)
{
    whoopsie_preferences_set_report_metrics (object, report);
    whoopsie_preferences_complete_set_report_metrics (object, invocation);
    return TRUE;
}

static gboolean
whoopsie_preferences_on_set_automatically_report_crashes (
                                            WhoopsiePreferences* object,
                                            GDBusMethodInvocation* invocation,
                                            gboolean report,
                                            gpointer user_data)
{
    whoopsie_preferences_set_automatically_report_crashes (object, report);
    whoopsie_preferences_complete_set_automatically_report_crashes (object, invocation);
    return TRUE;
}

static gboolean
whoopsie_preferences_on_get_identifier (WhoopsiePreferences* object,
                                        GDBusMethodInvocation* invocation)
{
    char* identifier = NULL;
    GError* error = NULL;

    whoopsie_identifier_generate (&identifier, &error);
    if (error) {
        g_printerr ("Error getting identifier: %s\n", error->message);
        g_error_free (error);
    }
    whoopsie_preferences_complete_get_identifier (object, invocation, identifier);
    return TRUE;
}

static gboolean
whoopsie_preferences_authorize_method (GDBusInterfaceSkeleton* interface,
                                       GDBusMethodInvocation* invocation,
                                       gpointer user_data)
{
    PolkitSubject* subject = NULL;
    PolkitAuthorizationResult* result = NULL;
    GError* error = NULL;
    const char* sender = NULL;
    gboolean ret = FALSE;

    sender = g_dbus_method_invocation_get_sender (invocation);
    subject = polkit_system_bus_name_new (sender);
    result = polkit_authority_check_authorization_sync (authority, subject,
                                    "com.ubuntu.whoopsiepreferences.change",
                                    NULL,
                                    POLKIT_CHECK_AUTHORIZATION_FLAGS_ALLOW_USER_INTERACTION,
                                    NULL, &error);
    if (result == NULL) {
        g_dbus_method_invocation_return_error (invocation, G_DBUS_ERROR,
                           G_DBUS_ERROR_AUTH_FAILED, "Could not authorize: %s",
                           error->message);
        g_error_free (error);
        goto out;
    }
    if (!polkit_authorization_result_get_is_authorized (result)) {
        g_dbus_method_invocation_return_error (invocation, G_DBUS_ERROR,
                           G_DBUS_ERROR_AUTH_FAILED, "Not authorized.");
        goto out;
    }
    ret = TRUE;
    out:
        if (result != NULL)
            g_object_unref (result);
        g_object_unref (subject);
        return ret;
}

static void
on_bus_acquired (GDBusConnection* connection, const gchar* name,
                 gpointer user_data)
{
    WhoopsiePreferences* interface;
    GError* error = NULL;

    interface = whoopsie_preferences_skeleton_new ();
    if (!g_dbus_interface_skeleton_export (
                G_DBUS_INTERFACE_SKELETON (interface),
                connection,
                "/com/ubuntu/WhoopsiePreferences", &error)) {

        g_print ("Could not export path: %s\n", error->message);
        g_error_free (error);
        g_main_loop_quit (loop);
        return;
    }

    authority = polkit_authority_get_sync (NULL, &error);
    if (authority == NULL) {
        g_print ("Could not get authority: %s\n", error->message);
        g_error_free (error);
        g_main_loop_quit (loop);
        return;
    }
    loop_shutdown = g_timeout_add_seconds (60, (GSourceFunc) g_main_loop_quit,
                                           loop);

    g_signal_connect (interface, "notify::report-crashes",
                      G_CALLBACK (whoopsie_preferences_changed),
                      "report_crashes");
    g_signal_connect (interface, "notify::report-metrics",
                      G_CALLBACK (whoopsie_preferences_changed),
                      "report_metrics");
    g_signal_connect (interface, "notify::automatically-report-crashes",
                      G_CALLBACK (whoopsie_automatic_reporting_changed), NULL);
    g_signal_connect (interface, "handle-set-report-crashes",
                      G_CALLBACK (whoopsie_preferences_on_set_report_crashes),
                      NULL);
    g_signal_connect (interface, "handle-set-report-metrics",
                      G_CALLBACK (whoopsie_preferences_on_set_report_metrics),
                      NULL);
    g_signal_connect (interface, "handle-set-automatically-report-crashes",
                      G_CALLBACK (whoopsie_preferences_on_set_automatically_report_crashes),
                      NULL);
    g_signal_connect (interface, "handle-get-identifier",
                      G_CALLBACK (whoopsie_preferences_on_get_identifier),
                      NULL);
    g_signal_connect (interface, "g-authorize-method", G_CALLBACK
                      (whoopsie_preferences_authorize_method), authority);

    monitor_autoreport_file (interface);
    whoopsie_preferences_load (interface);
    whoopsie_preferences_dbus_notify (interface);
}

static void
on_name_acquired (GDBusConnection* connection, const gchar* name,
                  gpointer user_data)
{
    g_print ("Acquired the name: %s\n", name);
}

static void
on_name_lost (GDBusConnection* connection, const gchar* name,
              gpointer user_data)
{
    g_print ("Lost the name: %s\n", name);
}

int
main (int argc, char** argv)
{
    guint owner_id;

    if (getuid () != 0) {
        g_print ("This program must be run as root.\n");
        return 1;
    }

    g_type_init();

    owner_id = g_bus_own_name (G_BUS_TYPE_SYSTEM,
                               "com.ubuntu.WhoopsiePreferences",
                               G_BUS_NAME_OWNER_FLAGS_NONE,
                               on_bus_acquired,
                               on_name_acquired,
                               on_name_lost,
                               NULL, NULL);

    loop = g_main_loop_new (NULL, FALSE);
    g_main_loop_run (loop);
    g_bus_unown_name (owner_id);
    g_main_loop_unref (loop);
    if (key_file)
        g_key_file_free (key_file);
    return 0;
}
