/*
 * Port of im-client.c using libindicate-qt
 *
 * Copyright 2009 Canonical Ltd.
 *
 * Authors:
 * - Aurélien Gâteau <aurelien.gateau@canonical.com>
 *
 * License: LGPL v2.1 or LGPL v3
 */
#include <QDebug>
#include <QDateTime>
#include <QImage>
#include <QTimer>

#include <qindicateserver.h>
#include <qindicateindicator.h>

#include "qim-client.h"

void Controller::slotServerDisplay()
{
    qDebug() << "A listener wants us to display the server";
    if (mAttentionIndicator) {
        delete mAttentionIndicator;
        mAttentionIndicator = 0;
        QTimer::singleShot(2000, this, SLOT(createAttentionIndicator()));
    }
    if (mCountIndicator) {
        delete mCountIndicator;
        mCountIndicator = 0;
        QTimer::singleShot(2000, this, SLOT(createCountIndicator()));
    }
}

void Controller::slotInterestAdded(QIndicate::Interest interest)
{
    qDebug() << __FUNCTION__ << interest;
}

void Controller::slotInterestRemoved(QIndicate::Interest interest)
{
    qDebug() << __FUNCTION__ << interest;
}

void Controller::createAttentionIndicator()
{
    qDebug() << "Creating an attention indicator";
    mAttentionIndicator = new QIndicate::Indicator(this);

    mPos = (mPos + 1) % mUserList.count();
    mAttentionIndicator->setNameProperty(mUserList[mPos]);

    mAttentionIndicator->setTimeProperty(QDateTime::currentDateTime());

    QImage image(SOURCE_DIR "/test.png");
    mAttentionIndicator->setIconProperty(image);

    QObject::connect(mAttentionIndicator, SIGNAL(display(QIndicate::Indicator*)),
                     SLOT(slotIndicatorDisplay(QIndicate::Indicator*)));

    mAttentionIndicator->show();

    QTimer::singleShot(1000, this, SLOT(modifyAttentionIndicator()));
}

void Controller::createCountIndicator()
{
    qDebug() << "Creating a count indicator";
    mCountIndicator = new QIndicate::Indicator(this);

    mCountIndicator->setNameProperty("Inbox");
    mCountIndicator->setCountProperty(5);

    QObject::connect(mCountIndicator, SIGNAL(display(QIndicate::Indicator*)),
                     SLOT(slotIndicatorDisplay(QIndicate::Indicator*)));

    mCountIndicator->show();
}

void Controller::modifyAttentionIndicator()
{
    if (!mAttentionIndicator) {
        return;
    }

    qDebug() << __FUNCTION__;
    bool attention = !mAttentionIndicator->drawAttentionProperty();
    QString message = attention
                      ? " wants to talk to you now!"
                      : " does not want to talk anymore";
    mAttentionIndicator->setNameProperty(mUserList[mPos] + message);
    mAttentionIndicator->setDrawAttentionProperty(attention);

    QTimer::singleShot(2000, this, SLOT(modifyAttentionIndicator()));
}

void Controller::slotIndicatorDisplay(QIndicate::Indicator* indicator)
{
    qDebug() << "A listener wants us to display an indicator";
    Q_ASSERT(indicator);
    if (indicator == mCountIndicator) {
        delete mCountIndicator;
        mCountIndicator = 0;
        QTimer::singleShot(2000, this, SLOT(createCountIndicator()));
    } else {
        delete mAttentionIndicator;
        mAttentionIndicator = 0;
        QTimer::singleShot(2000, this, SLOT(createAttentionIndicator()));
    }
}

int main (int argc, char ** argv)
{
    QApplication app(argc, argv);

    if (argc != 3) {
        qCritical("Syntax: %s <desktop/file> <serverType>", argv[0]);
        return 1;
    }

    QString desktopFile = QString::fromLocal8Bit(argv[1]);
    QString serverType = QString::fromLocal8Bit(argv[2]);

    QIndicate::Server* server = QIndicate::Server::defaultInstance();

    Controller controller;

    server->setType(serverType);
    server->setDesktopFile(desktopFile);
    server->show();
    QObject::connect(server, SIGNAL(serverDisplay()), &controller, SLOT(slotServerDisplay()));
    QObject::connect(server, SIGNAL(interestAdded(QIndicate::Interest)), &controller, SLOT(slotInterestAdded(QIndicate::Interest)));
    QObject::connect(server, SIGNAL(interestRemoved(QIndicate::Interest)), &controller, SLOT(slotInterestRemoved(QIndicate::Interest)));

    controller.createAttentionIndicator();
    controller.createCountIndicator();
    return app.exec();
}

#include "qim-client.moc"
